package uicc.hci.framework;

/**
 * This interface is the base interface for any HCI service.
 * <P> 
 * An HCI service instance shall be a permanent JCRE Entry Point Object
 * <P>
 * It provides the means for an Applet instance to get access to the resources
 * of a specific HCI service. An Applet instance can retrieve an implementation of
 * a specific service via the <code>HCIDevice</code>.
 * <P>
 * For a service object that is implementing one of the subinterfaces defined in this API
 * the Applet can only register the corresponding Listener defined in the same package
 * (e.g. a <code>CardEmulationService</code> shall only accept the regsitration of a
 * <code>CardEmulationListener</code>.
 * <P>
 * An Applet instance shall only be able to register one type of listener per service
 * at any time. Listener registration is persistent.
 * <P>
 * To receive events via the registered Listeners an Applet instance shall activate the events.
 * All events shall be deactivated by default. Event activation is persistent.
 **/
public interface HCIService {

    /**
     * This method registers a Listener object to the Service instance
     *
     * @param listener listener to be registered with the given service
     * @exception HCIException with reason code
     *            <ul>
     *            <li><code>HCIException.HCI_WRONG_LISTENER_TYPE</code> in case the listener Object registerd to the service
     *            is not implementing the corresponding interface of the HCIService instance.
     * <ul>
     * <li>a <code>ReaderListener</code> can only be registered to a <code>ReaderService</code>,
     * <li>a <code>CardEmulationListener</code> can only be registered to a <code>CardEmulationService</code>,
     * <li>a <code>ConnectivityListener</code> can only be registered to a <code>ConnectivityService</code>
     * </ul>
     *           <li><code>HCIException.HCI_LISTENER_ALREADY_REGISTERED</code> in case a listener Object of the same type has already been registered.
     * </ul>
     */
    public void register(HCIListener listener) throws HCIException;

    /** This method unregisters the given listener from the service.
     * <P>
     *  Calling <code>deregister</code> with an argument that does not identify a
     *  currently registered <code>HCIListener</code> has no effect.
     * 
     * @param listener listener to be unregistered from the service
     */
    public void deregister(HCIListener listener);

    /** 
     * This method requests the HCI framework to call the listener method 
     * <code>onCallBackTriggering(short event, HCIMessage message)<code>
     * to provide access to the HCIMessage object of the addressed service.
     * <P>
     * Calling <code>requestCallbackNotification</code> while no corresponding Listener
     * to this Service is registered has no effect.
     * 
     * <P>The activation of this event shall only be valid one time and transient.
     * <P>The notification will go to the corresponding Listener interface of this Service instance
     * <P>Any incoming HCI event may trigger the HCIListener before this request is
     * forwarded to the <code>HCIListener</code>.
     * 
     * <P>The value of the parameter shall not collide with any event constant
     * which is defined in the <code>HCIListener</code> subinterfaces.
     * 
     * @param event event value to be passed to the HCIListener. The value of the
     * 			parameter shall not collide with any event constant which is defined
     * 			in this API
     * @throws HCIEception with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE</code> in case
     *                    an allready in this specification defined event value was used
     * /editor add some blurbs abourt deregistraton of not registered Listeners/
     * 
     */
    public void requestCallbackNotification(byte event) throws HCIException;

    /**
     * This method is used to activate an event.
     *
     * <ul>
     * <li>Possible values for the parameter event for the different Service objects shall be:
     * <ul>
     * For CardEmulationService:
     * <li>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>CardEmulationListener.EVENT_SEND_DATA,
     * <li>CardEmulationListener.EVENT_FIELD_OFF.
     * </ul>
     * <ul>
     * For ReaderService:
     * <li>ReaderListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>ReaderListener.EVENT_TARGET_DISCOVERED.
     * </ul>
     * <ul>
     * For ConnectivityService:
     * <li>ConnectivityListener.EVENT_STANDBY.
     * </ul>
     * </ul>
     * @param event event to be activated
     * @throws HCIEception with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE<code> in case a wrong event was activated for this Service instance
     */
    public void activateEvent(byte event) throws HCIException;

    /**
     * This method is used to deactivate an event from the list of activated events.
     * 
     * <ul>
     * <li>Possible values for the parameter event for the different Service objects shall be:
     * <ul>
     * For CardEmulationService:
     * <li>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>CardEmulationListener.EVENT_SEND_DATA
     * <li>CardEmulationListener.EVENT_FIELD_OFF
     * </ul>
     * <ul>
     * For ReaderService:
     * <li>ReaderListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>ReaderListener.EVENT_TARGET_DISCOVERED.
     * </ul>
     * <ul>
     * For ConnectivityService:
     * <li>ConnectivityListener.EVENT_STANDBY.
     * </ul>
     * </ul>
     * @param event event to be activated
      * @throws HCIEception with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE<code> in case a wrong event was deactivated for this Service instance	 or in case the event wasn't activated before.
     */
    public void deactivateEvent(byte event) throws HCIException;

    /**
     * return the activation state of an event.
     * 
     * <ul>
     * <li>Possible values for the parameter event for the different Service objects shall be:
     * <ul>
     * For CardEmulationService:
     * <li>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>CardEmulationListener.EVENT_SEND_DATA
     * <li>CardEmulationListener.EVENT_FIELD_OFF
     * </ul>
     * <ul>
     * For ReaderService:
     * <li>ReaderListener.EVENT_GET_PARAMETER_RESPONSE
     * <li>ReaderListener.EVENT_TARGET_DISCOVERED.
     * </ul>
     * <ul>
     * For ConnectivityService:
     * <li>ConnectivityListener.EVENT_STANDBY.
     * </ul>
     * </ul>
     *
     * @param event event to request the activation state
     * @return true if the event is activated for this listener, else false
      * @throws HCIEception with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE<code>  in case the event was not one of the possible values.
     */
    public boolean getEventNotificationStatus(byte event) throws HCIException;
}

