/******************************************************************************
 *
 ******************************************************************************/
package uicc.hci.framework;

/**
 * An instance of HCIMessage is used to get access to the content of a received 
 * HCI message.
 *
 * Every HCI message that is not longer then 270 bytes has to be presented to the
 * Applet as a complete message. In this case the contactless framework is responsible
 * to manage the defragmentation.
 *
 * <P>Sub interfaces will typically have additional methods for sending HCI service
 * specific HCI messages.
 * <P>Instances of HCIMessage hide the fragmentation of messages on the HCI layer
 * (see TS 102 622). However, as receive and send buffers with limited capacity will
 * be used it provides methods to check if
 * a message is complete (see <code>isHeading()</code> and <code>isComplete()</code>) in order to support
 * streaming of long messages. 
 *
 * <P>Instances of this object shall be temporary JCRE entry point objects.
 * 
 * @see uicc.hci.framework.HCIListener
 **/
public interface HCIMessage {

    // ---------  Message types -------------
    /** Indicates the HCI message type "command" */
    public static final byte TYPE_COMMAND = (byte) 0x00;
    
    /** Indicates the HCI message type "event" */
    public static final byte TYPE_EVENT = (byte) 0x40;
    
    /** Indicates the HCI message type "response" */
    public static final byte TYPE_RESPONSE = (byte) 0x80;

    // --------- All Gate Message response -------------
    /** command completed successfully (with optional parameters)*/
    public static final byte RESP_ANY_OK = (byte) 0x00;
    
    /** the destination host is not connected*/
    public static final byte RESP_ANY_E_NOT_CONNECTED = (byte) 0x01;
    
    /** the format of the command parameters is wrong*/
    public static final byte RESP_ANY_E_CMD_PAR_UNKNOWN = (byte) 0x02;
    
    /** command was rejected and/or not completed */
    public static final byte RESP_ANY_E_NOK = (byte) 0x03;

    /** the registry parameter identifier is either unknown to the registry or an optional registry parameter is not implemented*/
    public static final byte RESP_ANY_E_REG_PAR_UNKNOWN = (byte) 0x05;
     
    /** the command is not supported by the gate*/
    public static final byte RESP_ANY_E_CMD_NOT_SUPPORTED = (byte) 0x07;
    
    /** command is inhibited due to failure of lower layer identity check*/
    public static final byte RESP_ANY_E_INHIBITED = (byte) 0x08;
    
    /** permission denied to write/read a value to/from a registry */
    public static final byte RESP_ANY_E_REG_ACCESS_DENIED = (byte) 0x0A;
    

    /** 
     * Checks if this is the heading part of an HCI message
     * 
     * The heading information indicates the first part of an HCI message with 
     * <code>isComplete()</code> it is possible to check weather the complete message
     * has already been received.
     *
     * <P>The preparing of an outgoing message (command response or event)
     * will reset the current HCI message states according the outgoing message.
     *
     * @return true if the current message is the heading part of the message
     * e.g the first part of an incoming message.
     **/
    public boolean isHeading();

    /** 
     * Cheks for the completnes of the HCI message.
     *
     * <P>The preparing of an outgoing message (command response or event)
     * will reset the current HCI message states according the outgoing message.
     *
     * @return true if the message is complete
     **/
    public boolean isComplete();

    /**
     * Returns the type of the incomming HCI message
     * 
     * <P>
     * In the case of a fragmented incoming message this method shall return
     * the HCI message type coded in the first part of the HCI message.
     * 
     * @return return the HCI message type, must be one of constant values
     *         <code>TYPE_*</code> defined in this interface
     **/
    public byte getType();

    /** 
     * Returns the instruction part of the HCP message header of the current
     * incomming HCP message, see TS 102 622 for further information.
     * 
     * <P>
     * In the case of a fragmented incoming message this method shall return
     * the HCI message instruction coded in the first part of the HCI message.
     *
     * @return return the message instruction, b8,b7 of the return vlue are set to zero.
     **/
    public byte getInstruction();

    /** 
     * Return a refernce to the underlying HCI message receive buffer. The length of this
     * buffer is system dependend. 
     *
     * <P>When the listener is triggered with an incoming message the usable data
     * shall be found in the buffer returned by this method, starting at the offset given by
     * the method <code>getReceiveOffset()</code>. The length of the received message or message fragment
     * can be retrieved from <code>getReceiveLength()</code>.
     * 
     * <ul>
     * <li>The framework shall copy data into the receive buffer up to the end of the buffer.
     * When the HCIMessage is longer then the available buffer length the HCIMessage shall be set
     * as not complete.
     * 
     * <li> The application may use the whole receive buffer for
     * its internal purposes. If the buffer is used for manipulation of sensitive
     * data it shall be cleared by the Applet before returning to the contactless framework.
     *
     * <li>The content of the array starts at the location indicated by getReceiveOffset() and has the length indicated by getReceiveLength()
     * 
     * <li>The content of the buffer outside of the area which is used for the HCI message part is undefined.
     * </ul>
     *
     * <p>
     * Note:
     * <ul>
     * <li><em>References to the receive buffer byte array
     * may be stored in local variables or method parameters.</em>
     * <li><em>References to the receive buffer byte array
     * cannot be stored in class variables or instance variables or array components.
     * </ul>
     * 
     * @return the buffer holding the current HCI message.
     **/
    public byte[] getReceiveBuffer();

    /** 
     * This method returns the offset of the received HCI message data in the receive buffer.
     * 
     * @return the offset into the buffer, retrieved via <code>getReceiveBuffer()</code>.
     **/
    public short getReceiveOffset();

    /**
     * This method returns the length of the received HCI message data. If the message is
     * not complete then the returned value is the actual HCI message fragment length.
     * 
     * @return the size of the message data available in the receive buffer.
     **/
    public short getReceiveLength();
}
