/******************************************************************************
 *
 ******************************************************************************/
package uicc.hci.framework;

/**
 * This interface is the base interface for any HCI service.
 * <P> A service is a JCRE Entry Point Object which allows to 
 * register and to unregister listeners in order to get access
 * to their services.
 * 
 **/
public interface HCIService {

    /** Registers the Listener Object to the service.
     * <P>  
     * <ul>
     * <li>The listener must correspond to the instance of the HCIService:
     * A ReaderListener can only be registered with a ReaderService,
     * a CardEmulationListener can only be registered with a CardEmulationService
     * and a ConnectivityListener can only be registered with a ConnectivityService,
     * else it will be refused.
     * 
     * <li>When a listener is registered all specific events of that listener
     * are by default in the state unregistered. The Applet must
     * activate any event in order to be notified on the event condition.
     * 
     * <li>
     * An Applet shall only be able to register one listener per service
     * at any time. 
     * </ul>
     * 
     * @param listener listener to be registered with the given service
     * (editor's note: add exceptions)
     */
    public void register(HCIListener listener) throws HCIException;

    /** This method unregisters the given listener from the service.
     * 
     * </ul>
     * @param listener listener to be unregistered from the given service
     */
    public void unregister(HCIListener listener);

    /** 
     * This method requests the HCI framework to call the listener method 
     * onCallBackTriggering(short event, HCIMessage message) with the event parameter
     * to provide access to the HCIMessage object of the addressed service.
     * 
     * <li>This request is volatile. In case of tear down the
     * Applet request is lost.
     * 
     * <li>Due to the event driven nature of the HCI framework any incoming
     * HCI event may trigger the HCIListener before this request is
     * forwarded to the HCIListener.
     * 
     * <li>The HCI framework shall buffer exactly one event per Service. 
     * If this entry point is called several times before the event can be 
     * forwarded to the listener only the last event will be delivered. 
     * 
     * The value of the parameter must not collide with any event constant 
     * which is defined in the HCILister interface(s). Therefore
     * the range of short values < 0 shall be reserved for the use with this method.
     * 
     * </ul>
     * @param event event value to be passed to the HCIListener. The value of the 
     * 			parameter must not collide with any event constant which is defined 
     * 			in the HCILister interface(s). Therefore the range of short values 
     * 			< 0 shall be reserved for the use with this method.  
     * @see uicc.hci.framework.HCIListener
     * (editor's note: add exception)
     **/
    public void requestCallBackNotification(short event) throws HCIException;

    /**
     * This method is used to configure which events will trigger this service.
     * 
     * <ul>
     * <li>Possible values for the parameter event are...
     * <li>For CardEmulationService:
     * CardEmulationListener.EVT_SEND_DATA, 
     * CardEmulationListener.EVT_FIELD_ON, 
     * CardEmulationListener.EVT_CARD_DEACTIVATED,
     * CardEmulationListener.EVT_CARD_ACTIVATED,
     * CardEmulationListener.EVT_FIELD_OFF.
     * 
     * <li>For ReaderService:
     * ReaderListener.EVT_TARGET_DISCOVERED.
     * 
     * <li>For ConnectivityService:
     * ConnectivityListener.EVT_STANDBY.
     * 
     * </ul>
     * @param event event to be activated
     * (editor's note: add exceptions)
     */
    public void activateEvent(byte event) throws HCIException;

    /**
     * This method is used remove an event from the list of events which 
     * cause notification of this service.
     * 
     * <ul>
     * <li>Possible values for the parameter event are...
     * <li>For CardEmulationService:
     * CardEmulationListener.EVT_SEND_DATA, 
     * CardEmulationListener.EVT_FIELD_ON, 
     * CardEmulationListener.EVT_CARD_DEACTIVATED,
     * CardEmulationListener.EVT_CARD_ACTIVATED,
     * CardEmulationListener.EVT_FIELD_OFF.
     * 
     * <li>For ReaderService:
     * ReaderListener.EVT_TARGET_DISCOVERED.
     * 
     * <li>For ConnectivityService:
     * ConnectivityListener.EVT_STAND_BY.
     * 
     * </ul>
     * @param event event to be activated
     * (editor's note: add exceptions)
     */
    public void deactivateEvent(byte event);

    /**
     * return the activation state of an event.
     * 
     * @param event event to inquire activation state 
     * @return true if the event is activated for this listener, else false
     */
    public boolean getEventNotificationStatus(byte event);
}

