package uicc.hci.framework;

/**
 * This interface is the base interface for any HCI service.
 * <P> An HCI service instance is a permanent JCRE Entry Point Object which allows to
 * register and to unregister listeners in order to receive events and HCIMessage objects.
 * 
 **/
public interface HCIService {

    /**
     * Provides the means to tget access to tge HCI resources
     * <P>  
     * A Service object allows to manage Listener interfaces and event. An object
     * implementing the HCIService interface is permamnent JCRE entry point object.
     * <P>
     * All events shall be deactivated by default. The Applet must
     * activate any event in order to be notified on the event condition. Event
     * activation is persistent.
     * <P>
     * An Applet shall only be able to register one listener per service
     * at any time. Listener registration is persistent.
     *
     * @param listener listener to be registered with the given service
     * @exception HCIException with reason code XX in case the listener Object registerd to the service
     *            is not implementing the corresponding interface of the HCIService instance.
     * <ul>
     * <li>A ReaderListener can only be registered with a ReaderService,
     * <li>a CardEmulationListener can only be registered with a CardEmulationService
     * <li>a ConnectivityListener can only be registered with a ConnectivityService,
     * </ul>
     *
     * with reson code YY in case a second instance of a specific Listener is registered.
     */
    public void register(HCIListener listener) throws HCIException;

    /** This method unregisters the given listener from the service.
     * 
     * @param listener listener to be unregistered from the given service
     */
    public void unregister(HCIListener listener);

    /** 
     * This method requests the HCI framework to call the listener method 
     * onCallBackTriggering(short event, HCIMessage message) with the event parameter
     * to provide access to the HCIMessage object of the addressed service.
     * 
     * <li>This request is transient. In case of tear down the
     * Applet request is lost.
     * 
     * <li>Due to the event driven nature of the HCI framework any incoming
     * HCI event may trigger the HCIListener before this request is
     * forwarded to the HCIListener.
     * 
     * <li>The HCI framework shall buffer exactly one event per Service. 
     * If this entry point is called several times before the event can be 
     * forwarded to the listener only the last event will be delivered. 
     * 
     * The value of the parameter must not collide with any event constant 
     * which is defined in the HCILister interface(s). Therefore
     * the range of short values < 0 shall be reserved for the use with this method.
     * 
     * </ul>
     * @param event event value to be passed to the HCIListener. The value of the 
     * 			parameter must not collide with any event constant which is defined 
     * 			in the HCILister interface(s). Therefore the range of short values 
     * 			< 0 shall be reserved for the use with this method.  
     * @see uicc.hci.framework.HCIListener
     * note: the notification will go to the corresponding Lister Interface
     * note: throws an Exception if an event is used that is defined in this spec
     **/
    public void requestCallBackNotification(byte event) throws HCIException;

    /**
     * This method is used to configure which events will trigger this service.
     *
     * It is only possible to activate Events for a subclass of the HCIService interface
     * 
     * <ul>
     * <li>Possible values for the parameter event are...
     * <li>For CardEmulationService:
     * CardEmulationListener.EVT_SEND_DATA, 
     * CardEmulationListener.EVT_FIELD_ON, 
     * CardEmulationListener.EVT_CARD_DEACTIVATED,
     * CardEmulationListener.EVT_CARD_ACTIVATED,
     * CardEmulationListener.EVT_FIELD_OFF.
     * 
     * <li>For ReaderService:
     * ReaderListener.EVT_TARGET_DISCOVERED.
     * 
     * <li>For ConnectivityService:
     * ConnectivityListener.EVT_STANDBY.
     * 
     * </ul>
     * @param event event to be activated
     * (editor's note: add exceptions)
     */
    public void activateEvent(byte event) throws HCIException;

    /**
     * This method is used remove an event from the list of events which 
     * cause notification of this service.
     * 
     * <ul>
     * <li>Possible values for the parameter event are...
     * <li>For CardEmulationService:
     * CardEmulationListener.EVT_SEND_DATA, 
     * CardEmulationListener.EVT_FIELD_ON, 
     * CardEmulationListener.EVT_CARD_DEACTIVATED,
     * CardEmulationListener.EVT_CARD_ACTIVATED,
     * CardEmulationListener.EVT_FIELD_OFF.
     * 
     * <li>For ReaderService:
     * ReaderListener.EVT_TARGET_DISCOVERED.
     * 
     * <li>For ConnectivityService:
     * ConnectivityListener.EVT_STAND_BY.
     * 
     * </ul>
     * @param event event to be activated
     * (editor's note: add exceptions)
     */
    public void deactivateEvent(byte event);

    /**
     * return the activation state of an event.  ???? what is this method good for
     * 
     * @param event event to inquire activation state 
     * @return true if the event is activated for this listener, else false
     */
    public boolean getEventNotificationStatus(byte event);
}

