package uicc.connection;

/**
 * The <code>BIPLink</code> interface represents a pure BIP logical connection to a remote entity.
 * 
 * Objects implementing <code>BIPLink</code> interface and returned by the <code>ConnectionServer.getConnection</code> method
 * are <b>Permanent Entry Point Objects</b>.
 *
 */
public interface BIPLink extends Observable
{
    /**
     * It request the opening of the BIP link.
     * 
     * @return the size of the allocated PDU buffer or <code>0</code> if the proactive command was not
     *         successful.
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *    	    <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *              issued.</li>
     *              <li><code>INVALID_STATE</code> if the BIP link is already opened.</li>
     *              <li><code>USER_CANCELLATION</code> if the user cancelled the opening of the BIP link.</li>
     *		    <li><code>UNSUPPORTED_OPERATION</code> if the Terminal Equipment does not support the required mechanisms to perform the action.
     *     		<li><code>RESOURCE_NOT_AVAILABLE</code> if there are not sufficient 
     *                resources available (on the UICC or on the terminal).
     *         </ul> 
     */
    public short open()
    		throws ConnectionException;

    /**
     * Closes the BIP link.
     * 
     * @return <code>true</code> if the proactive command has been successfully executed by the Terminal Equipment 
     *         <code>false</code> otherwise.
     *         
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *         	    <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *					issued.</li>
     *              <li><code>USER_CANCELLATION</code> if the user cancelled the closing of the BIP link</li>     
     *         </ul> 
     */
    public boolean close()
    		throws ConnectionException;

    /**
     * Sends data on the BIP link without any additional on-card reliable level.
     * 
     * <p>Note: The applet execution is resumed once the data has been sent.</p>
     * 
     * @param baData byte array containing the data to be sent
     * @param sOffset the starting offset of the data to be sent in the <code>baData</code> byte array
     * @param sLength the length of the data to be sent
     * 
     * @return <code>true</code> if the proactive command has been successfully executed by the Terminal Equipment;
     *         <code>false</code> otherwise.
     * 
     * @exception ArrayIndexOutOfBoundsException if <code>sOffset</code> and <code>sLength</code> are causing access
     *         outside of <code>baData</code> array bounds
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *              issued.</li>
     *              <li><code>INVALID_STATE</code> if the BIP link is not opened.</li>
     *		    <li><code>USER_CANCELLATION</code> if the user cancelled the connecting of the BIP link</li> 
     *         </ul>
     * @exception NullPointerException if <code>baData</code> is equal to <code>null</code>
     */
    public boolean send(byte[] baData, short sOffset, short sLength)
    		throws ArrayIndexOutOfBoundsException,
    		       ConnectionException,
    		       NullPointerException;
}
