
package uicc.connection;

/**
 * The <code>SCP81Connection</code> interface represents an SCP81 connection to
 * a remote entity.
 *
 * Objects implementing <code>SCP81Connection</code>interface and returned by the
 * <code>ConnectionServer.getConnection</code> method are <b>Permanent Entry Point Objects</b>.
 * 
 */
public interface SCP81Connection extends Observable {
/**
     * Request to open the SCP81 connection.
     * <p>
     * The associated (directly or indirectly) Security Domain of the calling application will handle the PSK TLS security of the communication.
     * <p>
     * The calling application will be notified once the connection is opened or once the setup failed. This method is non-blocking.
     *
     * Note:
     * <ul>
     * <li><em>On invocation of this method, the Security Domain shall register this request.
     * Once resources are available, the RTE/OPEN tries to open the connection, manage
     * the underlying protocol layers and performs the TLS handshake. The observer is then notified with
     * the </em><code>SCP81ConnectionStatus</code><em> event.
     * </ul>
     * <p>
     *
     * @param triggeringParameters this buffer contains the administration session triggering parameters
     * 		  as defined in table 4-3 of Amendment B of Global Platform Card Specification version 2.2. Tag '81' and
     *            Length shall be present. Connection parameters defined in Tag '84' may be used if TCP/IP transport is
     *            provided by the Bearer Independent Protocol of TS 102 223 [4]. Note: If a direct IP connection as specified in TS 102 483 [YY] is used these parameters will be ignored.
     * @param parameterOffset the starting offset of the parameters in the <code>triggeringParameters</code> byte array
     * @param parameterLength the length of the parameters
     * @param contentType this buffer contains the Content-Type header value to be used
     * when sending an HTTP POST request through this connection.
     * @param contentTypeOffset the starting offset of the content-type in the <code>contentType</code> byte array
     * @param contentTypeLength the length of the content-type
     *
     * @throws ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>INVALID_STATE</code> if the connection is already opened.
     *              <li><code>INVALID_PARAMETERS</code> if the triggering parameters are not correct or if
     *                   the Content-Type header value defined in the parameters is reserved (for example for
     *                   RAM/RFM over HTTP or for SCWS remote administration).
     *         		<li><code>SCP81_KEY_NOT_FOUND</code>if this requested SCP81 key could no be found
     *                    in the associated (directly or indirectly) Security Domain of the calling application. </li>
     *         </ul>
     *
     * @throws ArrayIndexOutOfBoundsException if <code>parameterOffset</code> and <code>parameterLength</code> are causing access
     *         outside of <code>triggeringParameters</code> array bounds or
     * if <code>contentTypeOffset</code> and <code>contentTypeLength</code> are causing access
     *         outside of <code>contentType</code> array bounds.
     *
     * @throws NullPointerException if <code>triggeringParameters</code>
     * or <code>contentType</code> are equal to <code>null</code>
     */
    public void open(byte[] triggeringParameters, short parameterOffset, short parameterLength,
                    byte[] contentType, short contentTypeOffset, short contentTypeLength)
    		throws ConnectionException;

    /**
     * Close the SCP81 Connection. The calling application will be notified once the connection is closed. This method is non-blocking.
     *
     * @throws ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>INVALID_STATE</code> if the connection is not open.
     *         </ul>
     */
    public void close()
    		throws ConnectionException;

    /**
     * Sends data on the SCP81 Connection.
     *
     * <p>Note: The applet execution is resumed once the HTTP POST request containing the data has been sent.</p>
     * <p>Note: The chunked transfer coding may be used if necessary.</p>
     *
     * @param baData byte array containing the data to be sent
     * @param sOffset the starting offset of the data to be sent in the <code>baData</code> byte array
     * @param sLength the length of the data to be sent
     *
     * @throws ArrayIndexOutOfBoundsException if <code>sOffset</code> and <code>sLength</code> are causing access
     *         outside of <code>baData</code> array bounds
     * @throws ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>INVALID_STATE</code> if
     *                         <ul>
     *                               <li> the connection is not open
     *                               <li> or if data has been sent through the HTTP POST request and the related HTTP POST
     *                                  response has not yet been delivered to the applet with the DataReceived event
     *                               <li> or if an HTTP POST response has been delivered to the applet but all received have not
     *                                  been received or discarded.
     *                         </ul>
     *         </ul>
     * @throws NullPointerException if <code>baData</code> is equal to <code>null</code>
     */
    public void send(byte[] baData, short sOffset, short sLength)
    		throws ArrayIndexOutOfBoundsException,
    		       ConnectionException,
    		       NullPointerException;
}
