/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.me.tools.fs;

import com.sun.tck.me.utils.Closables;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class DirFileSystem implements FileSystem {
    private Closables manager = new Closables();
    private File root;
    private File pwd;

    public DirFileSystem(File root) {
        this.root = root;
        this.pwd = root;
    }

    @Override
    public InputStream openToRead(String path) throws IOException {
        return manager.add(new FileInputStream(createFile(path, false)));
    }

    private File createFile(String path, boolean createParent) {
        int pos = 0;
        while (path.charAt(pos) != '/') {
            pos ++;
        };
        path = path.substring(pos).replace('/', File.separatorChar);
        File file = new File((pos > 0) ? root : pwd, path);
        if (createParent) {
            file.getParentFile().mkdirs();
        }
        return file;
    }

    @Override
    public OutputStream openToWrite(String path) throws IOException {
        return manager.add(new FileOutputStream(createFile(path, true)));
    }

    @Override
    public FileType getType(String path) {
        File file = createFile(path, false);
        if (file.isDirectory()) {
            return FileType.DIR;
        } else if (file.isFile()) {
            return FileType.FILE;
        } else if (!file.exists()) {
            return FileType.DOES_NOT_EXIST;
        } else {
            return FileType.UNKNOWN;
        }
    }

    private static final String[] EMPTY = new String[0];

    @Override
    public String[] list(final String path, final FileSystem.Filter fs) {
        final File dir = createFile(path, false);
        final FileSystem current = this;
        String[] retVal;
        if (fs == null) {
            retVal = dir.list();
        } else {
            retVal = dir.list(new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                return fs.accept(current, path + "/" + name);
            }
            });
        }
        return (retVal == null) ? EMPTY : retVal;
    }


    @Override
    public void close() throws IOException {
        manager.close();
    }

    @Override
    public void switchTo(String path) {
        pwd = new File(root, path.replace('/', File.separatorChar));
    }
}
