/**
 * %W% %E%
 * @Copyright
 */

package com.sun.tck.jc.interview.lib;

import java.io.File;
import java.util.Map;
import java.util.StringTokenizer;

import com.sun.interview.ErrorQuestion;
import com.sun.interview.Interview;

public interface Validator {

    static final String prefix = "com.sun.tck.jc.interview.lib.Validator";
    
    public boolean accept(String tag, Object value, Map map);

    public ErrorQuestion getErrorQuestion(String tag, Object o, Map map);
    
    public static final Validator TrueValidator = new Validator() {

            public String toString() {
                return prefix + "TrueValidator";
            }
            
            public boolean accept(String tag, Object value, Map map) {
                return true;
            }

            public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
                return null;
            }
        };

    public static final Validator NotEmptyListValidator = new Validator() {
            public boolean accept(String tag, Object list, Map map) {
                return ((list != null) && (list instanceof Object[])
                        && (((Object[])list).length > 0));
            }
            
            public String toString() {
                return prefix + "NotEmptyListValidator";
            }
            
            public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
                return null;
            }
        };

    public static final Validator ConfigName = new Validator() {
            
            public String toString() {
                return prefix + "ConfigName";
            }
            
            public boolean accept(String tag, Object o, Map map) {
                if ((o == null) || !(o instanceof String) || o.equals("")) {
                    return false;
                }
                String s = (String)o;
                
                if (!Character.isUnicodeIdentifierStart(s.charAt(0))) {
                    return false;
                }
                
                for (int i = 1; i < s.length(); i++) {
                    if (!Character.isUnicodeIdentifierPart(s.charAt(i))) {
                        return false;
                    }
                }
                
                return true;
            }
            
            public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
                return null;
            }
            
        };
    
    public static final Validator Empty = new Validator() {

            public String toString() {
                return prefix + "Empty";
            }
            
            public boolean accept(String tag, Object o, Map map) {
                return ((o == null)
                        || "".equals((o instanceof File) ? ((File)o).getPath() : o));
            }

            public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
                return null;
            }
        };

    public static final Validator NotEmpty = CompositeInterview.not(Empty);

    public static final Validator isDirectoryValidator = new Validator() {
            public String toString() {
                return prefix + "isDirectoryValidator";
            }
            
            public boolean accept(String tag, Object o, Map map) {
                return ((o != null) && !o.equals("")
                        && (((o instanceof File) && ((File)o).isDirectory())
                            || ((o instanceof String)
                                && (new File ((String)o)).isDirectory())));
            }

            public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
                return null;
            }
        };

    public static class VariableListValidator implements Validator {
        private ErrorQuestion formatError;
        private String tag;
        private Object variables;
        
        public VariableListValidator(Interview interview, String tag) {
            this.tag = tag;
            
            this.formatError = new ErrorQuestion(interview, tag) {
                    public Object[] getTextArgs() {
                        return new String[] {
                            getWrongVariables(variables)
                        };
                    }
                };
        }

        
        public String toString() {
            return prefix + "VariableListValidator." + tag;
        }
            
        public boolean accept(String tag, Object o, Map map) {
            return ((o != null) && (o instanceof String)
                    && getWrongVariables(o).equals(""));
        }

        private String getWrongVariables(Object value) {
            if ((value == null) || !(value instanceof String)) {
                return "";
            }
            StringBuffer retVal = new StringBuffer();
            StringBuffer current = new StringBuffer();
            boolean isInsideString = false;
            for (StringTokenizer e = new StringTokenizer((String)value, " \n\t\"", true);
                 e.hasMoreTokens();) {
                String entry = e.nextToken();
                if (entry.equals("\n")) {
                    return "\n";
                } else if (entry.equals("\"")) {
                    isInsideString = ! isInsideString;
                    current.append('\"');
                } else if (entry.equals(" ") || entry.equals("\t")) {
                    if (isInsideString) {
                        current.append('\"');
                    } else {
                        processWord(retVal, current);
                    }
                } else {
                    current.append(entry);
                }
            }
            if (isInsideString) {
                retVal.append(current.toString());
            } else {
                processWord(retVal, current);
            }
            
            return retVal.toString();
        }

        void processWord(StringBuffer retVal, StringBuffer buff) {
            String word = buff.toString();
            int pos = word.indexOf("=");
            if ((buff.length() != 0)
                && ((pos < 0) || (pos == (word.length() - 1)))) {
                retVal.append(word);
                retVal.append(' ');
            }
            buff.delete(0, buff.length());
        }
        
        public ErrorQuestion getErrorQuestion(String tag, Object o, Map map) {
            variables = o;
            return formatError;
        }
    }

    public static class FileListValidator implements Validator {

        private ErrorQuestion notFound;
        private String tag;
        private Object files;
        
        public FileListValidator(Interview interview, String tag) {
            this.tag = tag;
            
            this.notFound = new ErrorQuestion(interview, tag) {
                    public Object[] getTextArgs() {
                        return new String[] {
                            createNonExistedFilesList(files)
                        };
                    }
                };
        }

        private String createNonExistedFilesList(Object o) {
            if (o instanceof File) {
                return ((File)o).getAbsolutePath();
            }
            if ((o == null) || !(o instanceof File[])) {
                return null;
            }
            StringBuffer retVal = new StringBuffer();
            File[] list = (File[])o;
            for (int i = 0; i < list.length; i++) {
                if (!list[i].exists()) {
                    retVal.append(list[i].getAbsolutePath());
                    retVal.append(' ');
                }
            }
            return retVal.toString();
        }
        
        public String toString() {
            return prefix + "FileListValidator." + tag;
        }

        public boolean accept(String tag, Object o, Map map) {
            if (o instanceof File) {
                return isFileExist((File)o);
            } else {
                return (o instanceof File[]) && areFilesExist((File[])o);
            }
        }

        private boolean isFileExist(File file) {
            return file.exists();
        }

        private boolean areFilesExist(File[] files) {
            for (int i = 0; i < files.length; i++) {
                if (!isFileExist(files[i])) {
                    return false;
                }
            }
            return true;
        }
            
        public ErrorQuestion getErrorQuestion(String tag, final Object o, Map map) {
            this.files = o;
            return accept(tag, o, map) ? null : notFound;
        }
    }
}
        
