/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.terminal;

import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FatalException.Scope;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.security.GeneralSecurityException;
import java.util.LinkedHashMap;
import java.util.Set;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class CardStore {
    public static final int NUMBER_OF_GSM_KEYS = 4;
    public static final int ICCID_OFFSET = 1;
    public static final int DEFAULT_GSM_KEYS_OFFSET = 14;
    private static CardStore instance;
    private File stateStorage;
    private Format format = Format.LONG;

    public void setFormat(Format format) {
//TODO-Logging        System.out.println("CardStore.setFormat(" + format + ")");
        this.format = format;
    }

    public static enum Format {
        LONG(ICCID_OFFSET, DEFAULT_GSM_KEYS_OFFSET, 3),
        SHORT(0, 1, 2);

        private Format(int iccidOffset, int gsmKeysOffset, int keysPerVersion) {
            this.iccidOffset = iccidOffset;
            this.gsmKeysOffset = gsmKeysOffset;
            this.keysPerVersion = keysPerVersion;
        }
        private int iccidOffset;
        private int gsmKeysOffset;
        private int keysPerVersion;

        public int getICCIDOffset() {
            return iccidOffset;
        }

        public String getKID(String[] keys, int id) {
            return keys[this.gsmKeysOffset + (id - 1) * keysPerVersion + 1];
        }

        public String getKIC(String[] keys, int id) {
            return keys[this.gsmKeysOffset + (id - 1) * keysPerVersion];
        }

        public int getKeyCount(int length) {
            return (length - gsmKeysOffset)/keysPerVersion;
        }
    }

    public static synchronized CardStore getDefaultStore() {
        if (instance == null) {
            instance = new CardStore();
        }
        return instance;
    }

    private LinkedHashMap<String,Card> cards = new LinkedHashMap<String, Card>();
    private File keyList;

    public void setKeyList(String name) throws Exception {
        this.keyList = new File(name);
    }

    public void init() throws Exception {
        File defList = new File("keys.txt");
        if ((keyList == null) && defList.exists()) {
            this.keyList = defList;
        }
        setKeyList(new FileInputStream(keyList));
    }

    public void close() throws Exception {
    }

    public void setKeyList(InputStream strm) throws Exception {
        BufferedReader in = new BufferedReader(new InputStreamReader(strm));
        try {
            String line;
            while ((line = in.readLine()) != null) {
                String[] keys = line.split("[\n\t ]+");
                int iccidOffset = format.getICCIDOffset();
                String iccid = canonizeICCID(keys[iccidOffset]);
                Card card = new Card(CardState.restore(stateStorage, iccid), iccid);
                configure(card, keys);
                cards.put(iccid, card);
            }
        } finally {
            in.close();
        }
    }

    private void configure(Card card, String[] keys) throws GeneralSecurityException {
        int length = format.getKeyCount(keys.length);
        for (int id = 1; id <= length; id++) {
            card.addKey(id, format.getKID(keys, id), format.getKIC(keys, id));
        }
    }

    private static String canonizeICCID(String st) {
        if (st.length() < 20) {
            return st.toLowerCase();
        } else {
            return st.substring(0, st.length() - 1).toLowerCase();
        }
    }

    public Card findCard(String id) throws FatalException {
        Card retVal = null;
        id = canonizeICCID(id);
        for (String iccid : cards.keySet()) {
            if (iccid.endsWith(id)) {
                if (retVal != null) {
                    throw new FatalException(Scope.Card, "CONFLICT: " + retVal.state.iccid + " and " + iccid);
                }
                retVal = cards.get(iccid);
            }
        }
        if (retVal == null) {
            throw new FatalException(Scope.Card, "can not find key for " + id);
        }
        return retVal;
    }

    public Set<String> getCardIDs() {
        return cards.keySet();
    }

    @Override
    public String toString() {
        StringBuilder retVal = new StringBuilder("(CardStore");
        for (String iccid : cards.keySet()) {
            retVal.append("\n    ").append(iccid).append("=").append(cards.get(iccid));
        }
        retVal.append(")");
        return retVal.toString();
    }

    public void setStorageDir(File stateStorage) {
        this.stateStorage = stateStorage;
    }
}
