/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.tlv;

import java.util.ArrayList;
import com.sun.tck.me.utils.Utils;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class TLVBuffer {

    public TLVBuffer(String string) {
        this(Utils.parse(string));
    }

    private static class Ref {
        public byte[] ref;
        public int upperBound;
        public Ref(byte[] ref, int upperBound) {
            this.ref = ref;
        }
        public void setPos(int val) {
            this.upperBound = Math.max(upperBound, val);
        }
    }

    private Ref data;
    private Position counter;
    private ArrayList<Position> pointers = new ArrayList<Position>();
    private Position start;
    private Position end;
    private boolean isExpandable;

    public TLVBuffer(int length) {
        this(new byte[length], 0, length);
        isExpandable = true;
    }

    public Position createPosition() {
        return createPosition(getPos());
    }

    public Position createPosition(int pos) {
        Position retVal = new Position(this, pos);
        pointers.add(retVal);
        return retVal;
    }

    public TLVBuffer(byte[] data) {
        this(data, 0, data.length);
        isExpandable = false;
    }

    private TLVBuffer(TLVBuffer parent, Position start, Position end) {
        this.data = parent.data;
        this.start = start;
        this.end = end;
        counter = createPosition(start.pos);
        this.isExpandable = parent.isExpandable;

    }

    public TLVBuffer(byte[] data, int offset, int length) {
        this.data = new Ref(data, offset);
        start = createPosition(offset);
        end = createPosition(offset + length);
        counter = createPosition(offset);
        isExpandable = false;
    }

    public byte[] getBuffer() {
        return data.ref;
    }

    public int getPos() {
        return counter.pos;
    }

    public int start() {
        return start.pos;
    }

    public int end() {
        return end.pos;
    }

    public void reset() throws EncodingException {
        setPosTo(start());
    }

    public TLVBuffer createChild(Position start, int length) throws EncodingException {
        TLVBuffer retVal = new TLVBuffer(this, start, 
                createPosition(start.pos +length));
        retVal.pointers = pointers;
        retVal.check(getPos());
        retVal.counter.pos = getPos();
        this.counter.pos = retVal.end();
        return retVal;
    }

    public TLVBuffer createChild(int length) throws EncodingException {
        return createChild(createPosition(), length);
    }

    public void guarantee(int upperPos) {
        data.setPos(upperPos);
        if (upperPos <= data.ref.length) {
            return;
        }
        // Increase size of the array
        int newLength = Math.max((int)(data.ref.length * 1.5), upperPos + 128);
        byte[] increased = new byte[newLength];
        System.arraycopy(data.ref, 0, increased, 0, data.ref.length);
        data.ref = increased;
    }

    public int write(int b) {
        guarantee(counter.pos + 1);
        data.ref[counter.pos++] = (byte)b;
        return counter.pos;
    }

    public int write(byte[] data) {
        return write(data, 0, data.length);
    }

    public int write(byte[] data, int offset, int length) {
        guarantee(counter.pos + length);
        System.arraycopy(data, offset, this.data.ref, counter.pos, length);
        counter.pos += length;
        return counter.pos;
    }

    public int write(long data, int length) {
        guarantee(counter.pos + length);
        for (int i = length - 1; i >= 0; i--) {
            this.data.ref[counter.pos + i] = (byte)data;
            data >>= 8;
        }
        counter.pos += length;
        return counter.pos;
    }

    public int skip(int length) throws EncodingException {
        guarantee(getPos() + length);
        check(getPos() + length);
        counter.pos += length;
        return counter.pos;
    }

    private void check(int pos) throws EncodingException {
        if ((pos < start.pos) || (pos > end.pos)) {
            throw new EncodingException(pos + " is not in range from "
                    + start.pos + " inclusive to " + end.pos + " exclusive");
        }
    }

    public void setPosTo(int pos) throws EncodingException {
        check(pos);
        counter.pos = pos;
    }

/*    public byte[] toByteArray(Position pos) {
        int start = (pos == null) ? 0 : pos.pos;
        byte[] retVal = new byte[getPos() - start];
        System.arraycopy(data, start, retVal, 0, retVal.length);
        return retVal;
    }
*/
    public static class Position {
        public int pos;
        public TLVBuffer buff;
        public Position(TLVBuffer buff, int pos) {
            this.buff = buff;
            this.pos = pos;
        }

        public void write(int val) throws EncodingException {
            this.buff.check(pos);
            this.buff.guarantee(pos + 1);
            this.buff.data.ref[pos++] = (byte)val;
        }

        public int read(int val) throws EncodingException {
            this.buff.check(pos);
            return this.buff.data.ref[pos++] & 0xFF;
        }
    }


    public void expand(int increment) throws EncodingException {
        if (increment < 0) {
            throw new EncodingException("Increment should be positive");
        }
        guarantee(data.upperBound + increment);
        // copy current position to the variable,
        // because the end.pos is changed during update
        int at = end.pos;
        System.arraycopy(data.ref, at, data.ref, at + increment, 
                data.ref.length - at - increment);
        for (Position p : pointers) {
            // TODO binary search
            if (p.pos >= at) {
                p.pos += increment;
            }
        }
    }

    public byte[] readArray(int length) throws EncodingException {
        check(getPos() + length);
        byte[] retVal = new byte[length];
        System.arraycopy(data.ref, counter.pos, retVal, 0, length);
        counter.pos += length;
        return retVal;
    }

    public int read() throws EncodingException {
        check(getPos());
        int retVal = 0xff & data.ref[counter.pos++];
        return retVal;
    }

    public long read(int length) throws EncodingException {
        check(getPos() + length);
        long retVal = 0;
        for (int i = 0; i < length; i++) {
            retVal = (retVal << 8) | read();
        }
        return retVal;
    }

    public int available() {
        return end() - counter.pos;
    }

    @Override
    public String toString() {
        return "TLVBuffer:" + Utils.canonize(data.ref, start(), Math.min(65, end() - start()));
    }
}
