/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.gsm;

import com.sun.tck.me.utils.User;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.DeploymentException;
import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FatalException.Scope;
import com.sun.javacard.cjck.userinterface.OutOfMemoryOnCardException;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.tck.bvtool.terminal.Card;
import com.sun.tck.bvtool.etsi.data.ResponseHeader;
import com.sun.tck.bvtool.etsi.data.sms.SMS_PP_TLV;
import com.sun.tck.bvtool.etsi.gp.GPCardService;
import com.sun.tck.bvtool.etsi.tlv.ConstantBundle;
import com.sun.tck.bvtool.etsi.tlv.Encodable;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.RawTLV;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer;
import com.sun.tck.bvtool.etsi.tlv.TLVPrototypeFactory;
import com.sun.tck.me.utils.Utils;
import java.util.List;
import static com.sun.tck.bvtool.etsi.gsm.GSMConstants.TERMINAL_PROFILE_INS;
import static com.sun.tck.bvtool.etsi.gsm.GSMConstants.TERMINAL_RESPONSE_INS;
import static com.sun.tck.bvtool.etsi.gsm.GSMConstants.SIM_CLA;
import static com.sun.tck.bvtool.etsi.gsm.GSMConstants.FETCH_INS;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class SMSChannel {
    private static final byte ZERO = (byte)0x00;
    public static final String RESPONSE_APDU_PATH = "SMS-TPDU/response/response-apdus";
    public static final String RESPONSE_HEADER = "SMS-TPDU/response/command-header";

    public static final int SW_OK = 0x9000;

    private GPCardService service;
    private Card card;

    private User user = new User(User.NOT_INTERACTIVE);

    public SMSChannel(GPCardService service) {
        this.service = service;
    }

    public void setUser(User.UserConfirmation confirm) {
        confirm = (confirm == null) ? User.NOT_INTERACTIVE : confirm;
        this.user = new User(confirm);
        if (card != null) {
            card.setUser(this.user);
        }
    }

    public void setCard(Card card) {
        this.card = card;
        card.setUser(this.user);
    }

    public ResponseAPDU initCard() throws Exception {
//        user.checkYes("init card?");
        user.log("TERMINAL PROFILE");
        CommandAPDU apdu = new CommandAPDU(SIM_CLA, TERMINAL_PROFILE_INS,
                (byte)0, (byte)0, Utils.parse("130180020000000000"), 0);
        ResponseAPDU answer = service.sendAPDU(apdu);
        if (isProactiveCommand(answer)) {
            int sw = answer.sw();
            answer = service.sendAPDU(new CommandAPDU(new byte[] {
                SIM_CLA, FETCH_INS, (byte)0x00, (byte)0x00, (byte)sw
            }));
            ProactiveCommandTLV tlv = new ProactiveCommandTLV();
            tlv.decode(service.getFactory(), new TLVBuffer(answer.getResponseData()));
            return service.sendAPDU(createTerminalResponseOK(tlv));
        }
        return answer;
    }

    /**
     * Sends list of the apdu to the CardManager via SMS.
     * @param apdus
     * @return
     * @throws Exception
     */
    public List<ResponseAPDU> sendSMS(CommandAPDU... apdus)
            throws Exception {
        checkInitialization(card);
        ProactiveCommandTLV tlv;
        while (!process(tlv = sendSingleSMS(apdus))) {
            card.logCardInfo();
            user.checkYes("Resending TLV: Continue?");
        }
        return (List<ResponseAPDU>)tlv.get(RESPONSE_APDU_PATH);
    }

    private ConstantBundle responseCodes = new ConstantBundle(false, ResponseHeader.class);

    public void powerReset() {
        if (card != null) {
            this.card.isInitalized = false;
        }
    }

    private boolean process(ProactiveCommandTLV apdu) throws CardProxyException {
        ResponseHeader header = (ResponseHeader)apdu.get(RESPONSE_HEADER);
        if (header == null) {
            throw new DeploymentException("Can not find Response Header");
        }
        this.user.log("Response-Code:" + responseCodes.find(header.status()));
        switch (header.status()) {
            case ResponseHeader.PoR_OK:
                card.confirmCounter();
                return true;
            case ResponseHeader.TAR_Unknown:
                card.confirmCounter();
                throw new FatalException(Scope.Card, "TAR is not supported");
            case ResponseHeader.CNTR_Low:
                card.increaseCounter();
                break;
            case ResponseHeader.CNTR_High:
                card.decreaseCounter();
                break;

            case ResponseHeader.CNTR_Blocked:
            case ResponseHeader.RC_CC_DS_Failed:
            case ResponseHeader.Ciphering_Error:
            case ResponseHeader.Undefined_Security_Error:
                card.changeKeyVersion();
                break;

            case ResponseHeader.Insufficient_Memory:
                card.confirmCounter();
                throw new OutOfMemoryOnCardException();

            case ResponseHeader.More_Time:
                card.confirmCounter();
                sleep();
                break;
            case ResponseHeader.Insufficient_Security_Level:
                card.increaseSecurityLevel();
                break;
            default:
                throw new DeploymentException("Unknown status:" + header.status());
        }
        return false;
    }

    private Object monitor = new Object();

    private void sleep() {
        try {
            synchronized (monitor) {
                monitor.wait(2000L);
            }
        } catch (InterruptedException e) {
            return;
        }
    }

    private ProactiveCommandTLV fetch(Card card, int le)
            throws CardProxyException, EncodingException {
        user.log("FETCH PROACTIVE COMMAND");
        ResponseAPDU answer = service.sendAPDU(new CommandAPDU(new byte[] {
            GSMConstants.SIM_CLA, GSMConstants.FETCH_INS,
            ZERO, ZERO, (byte)le
        }));

        ProactiveCommandTLV tlv = new ProactiveCommandTLV();
        byte[] data = answer.getResponseData();
        log("GET SMS RESPONSE:" + Utils.canonize(data));
        tlv.decode(new TLVPrototypeFactory(card), new TLVBuffer(data));
        log("TERMINAL RESPONSE OK");
        service.sendAPDU(createTerminalResponseOK(tlv));
        return tlv;
    }

    private ProactiveCommandTLV sendSingleSMS(CommandAPDU... apdus)
            throws CardProxyException {
        try {
            int counter = card.getCounter();
            card.logCardInfo();
            if (counter > 0x1000) {
                user.checkYes("!!!WARNING: Counter is too high. It might block the card prematurely. Are you sure?");
            }
            int length = 104;//255 - service.getHeaderLength();
            List<SMS_PP_TLV> smsList = SMS_PP_TLV.create(card, length, apdus);

            for (SMS_PP_TLV sms : smsList) {
                user.log("SEND SMS: " + sms.getShortDescription());

                ResponseAPDU answer = service.sendAPDU(
                        new CommandAPDU(
                        (byte)0x80, (byte)0xC2, (byte)0x00, (byte)0x00,
                        sms.toByteArray(), (byte)0));
               if (isProactiveCommand(answer)) {
//                    user.checkYes("PROACTIVE COMMAND: Fetch?");
                    return fetch(card, answer.sw() & 0xFF);
                }
            }
        } catch (EncodingException e) {
            throw new DeploymentException("Can not create or parse TLV", e);
        }
        throw new DeploymentException("there is no expected proactive command");
    }

    private void checkInitialization(Card card) throws Exception {
        if (card.isInitalized) {
            return;
        }
        ResponseAPDU apdu = initCard();
        check(apdu, SW_OK);
        card.isInitalized = true;
    }

    private static boolean compare(ResponseAPDU apdu, int expected, int mask) {
        int sw = apdu.sw();
        return ((sw & mask & 0xFFFF) == (expected & 0xFFFF));
    }

    private static void check(ResponseAPDU apdu, int expected) throws EncodingException {
        check(apdu, expected, 0xFFFF);
    }

    /**
     * Checks that the SW returned by the card is as expected. If the SW is
     * not as expected, then the EncodingException is thrown.
     * @param apdu apdu returned by the card
     * @param expected expected SW
     * @param mask mask being applied to SW returned by the card prior checks
     * @throws EncodingException If the SW is not as expected
     */
    private static void check(ResponseAPDU apdu, int expected, int mask) throws EncodingException {
        if (!compare(apdu, expected, mask)) {
            throw new EncodingException("Incorrect sw:0x" + Integer.toHexString(apdu.sw()));
        }
    }

    /**
     * returns true if the given response apdu indicates that the
     * proactive command is ready to fetch.
     */
    private static boolean isProactiveCommand(ResponseAPDU answer) {
        return 0x9100 == (0xFF00 & answer.sw());
    }

    private static CommandAPDU createTerminalResponseOK(ProactiveCommandTLV command) throws EncodingException {
        TLVBuffer data = new TLVBuffer(256);
        ((Encodable)command.get("command-details")).encode(data); // Command details
        data.write(0x82028281L, 4);                 // Device identities
        new RawTLV((byte)(GSMTags.ResultTag | 0x80), "00").encode(data); // result
        byte[] bytes = new byte[data.getPos()];
        System.arraycopy(data.getBuffer(), 0, bytes, 0, bytes.length);
        return new CommandAPDU(SIM_CLA, TERMINAL_RESPONSE_INS, ZERO, ZERO, bytes, ZERO);
    }

    private void log(Object msg) {
        service.out.println(msg);
    }
}
