/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.gp;

import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.invoke.CardServicesPool;
import com.sun.javacard.cjck.userinterface.AppletID;

import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.CommunicationService;
import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.tck.bvtool.etsi.gsm.StatusCodes;
import com.sun.tck.bvtool.etsi.security.KeyDescr;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.TLV;
import com.sun.tck.bvtool.etsi.tlv.TLVEncoder;
import com.sun.tck.bvtool.etsi.tlv.TLVPrototypeFactory;
import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import com.sun.tck.me.utils.CommandLineAdapter;
import com.sun.tck.me.utils.Utils;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import javax.smartcardio.Card;
import javax.smartcardio.CardException;
import javax.smartcardio.CardNotPresentException;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class GPCardService implements CJCKCardService, CardService, CommunicationService {
    boolean isInSession;
    boolean isInGPAction;
    private byte GET_RESPONSE_CLA_BYTE = (byte)0x00;
    private StatefulCardTerminal terminal;


    public CommunicationService getCommunicationService(int i) throws CardProxyException {
        return this;
    }

    public void setCardTerminal(StatefulCardTerminal terminal) {
        this.terminal = terminal;
        cad = new CardReader(terminal);
        this.card = terminal.getInsertedCard();
    }

    protected StatefulCardTerminal getCardTerminal() {
        return terminal;
    }

    public static class CardReader {
        private StatefulCardTerminal terminal;
        private Card card;

        public CardReader(StatefulCardTerminal terminal) {
            this.terminal = terminal;
        }

        public CardReader() {
        }

        public void init(PrintWriter out) throws FatalException {
            if (terminal == null) {
                this.terminal = CardServicesPool.getAvailableTerminals().get(0);
            }
            try {
            terminal.waitForCardPresent(600000L);
            } catch (CardException ex) {
                throw new FatalException(FatalException.Scope.CAD,
                        I18n.getString("card.exception.waitforpresent", ex));
            }
            card = terminal.connectToCard("*");
            out.println("ATR: bytes: 0x" + Utils.canonize(card.getATR().getBytes())
                    + " historical bytes: 0x" +  Utils.canonize(card.getATR().getHistoricalBytes()));
            out.flush();
        }

        public ResponseAPDU sendAPDU(CommandAPDU capdu) throws FatalException {
            javax.smartcardio.CommandAPDU scApdu = new javax.smartcardio.CommandAPDU(capdu.getBytes());
            try {
                javax.smartcardio.ResponseAPDU scAnswer = card.getBasicChannel().transmit(scApdu);
                return new ResponseAPDU(scAnswer.getBytes());
            } catch (CardNotPresentException ex) {
                throw new FatalException(FatalException.Scope.Card,
                        I18n.getString("cardnopresent.exception.transmit", ex));
            } catch (CardException ex) {
                //Additional check if terminal itself is alive
                try {
                    if (!terminal.isCardPresent()) {
                        throw new FatalException(FatalException.Scope.Card,
                                I18n.getString("card.exception.transmit", ex));
                    }
                } catch (CardException ex2) {
                    //we throw FatalException with original exception: ex
                }
                throw new FatalException(FatalException.Scope.CAD,
                        I18n.getString("card.exception.transmit", ex));
            }
        }

        public void powerUp() throws FatalException {
            card = terminal.connectToCard("*");
        }

        public void powerDown() throws FatalException {
            try {
                card.disconnect(true);
            } catch (CardException ex) {
                throw new FatalException(FatalException.Scope.Card,
                        I18n.getString("card.exception.disconnect", ex));
            }
        }

        public void release() throws Exception {
            //Does nothing
        }
    }

    private static final byte GP_CLA            = (byte)0x80;
    private static final byte JC_CM_CLA         = (byte)0x00;
    private static final byte DELETE_INS        = (byte)0xE4;
    private static final byte INSTALL_INS       = (byte)0xE6;
    private static final byte LOAD_INS          = (byte)0xE8;
    private static final byte SELECT_INS        = (byte)0xa4;
    private static final byte SELECT_BY_DF_NAME = (byte)0x04;
    private static final byte AID_TAG           = (byte)0x4f;
    public static final byte APPLICATION_SPECIFIC_PARAMETERS_TAG = (byte)0xc9;
    public static final byte SYSTEM_SPECIFIC_PARAMETERS_TAG = (byte)0xEF;
    public static final byte VOLATILE_DATA_SPACE_LIMIT_TAG = (byte)0xC7;
    public static final byte NON_VOLATILE_DATA_SPACE_LIMIT_TAG = (byte)0xC8;

    public static final byte SECURITY_DOMAIN_PRIVILEDGE = (byte)0x80;
    public static final byte DAP_VERIFICATION_PRIVILEDGE = (byte)0xC0;
    public static final byte DELEGATED_MANAGEMENT_PRIVILEDGE = (byte)0xA0;
    public static final byte CARD_LOCK_PRIVILEDGE = (byte)0x10;
    public static final byte CARD_TERMINATE_PRIVILEDGE = (byte)0x08;
    public static final byte DEFAULT_SELECTED_PRIVILEDGE = (byte)0x04;
    public static final byte CVM_MANAGEMENT_PRIVILEDGE = (byte)0x02;
    public static final byte MANDATED_DAP_VERIFICATION_PRIVILEDGE = (byte)0xC1;



    private static final int SW_OK              = 0x9000;
    private static final int HEADER_LENGTH      = 37;
    private static final int APDU_LENGTH        = 254;
    private static final int APDU_LENGTH_NET    = APDU_LENGTH - HEADER_LENGTH;
    public PrintWriter out;
    protected PrintWriter ref;
    private CardReader cad = new CardReader();
    private KeyDescr key;
    protected TLVPrototypeFactory parser;
    protected com.sun.tck.bvtool.terminal.Card card;

    public void setCad(CardReader cad) {
        this.cad = cad;
    }


    public GPCardService() {
    }

    public TLVPrototypeFactory getFactory() {
        return parser;
    }

    public void startTest(String workingDir, int numberOfExecutions)
            throws CardProxyException {
        cad.init(out);
        resetCard();
    }

    public void resetCard() throws CardProxyException {
        powerUp();
        // reset the card
        sendAPDU(new CommandAPDU(Utils.parse("D08111222000112233445566778899aabbccddeeff00112233445566778899aabbccddeeff")));
        sendAPDU(new CommandAPDU(Utils.parse("D080010000")));
        sendAPDU(new CommandAPDU(Utils.parse("D08000003A11FF050001000005404142434445464748494A4B4C4D4E4F404142434445464748494A4B4C4D4E4F404142434445464748494A4B4C4D4E4F0215")));
    }


    private ResponseAPDU send(String apdu) throws CardProxyException {
        return send(new CommandAPDU(Utils.parse(apdu)));
    }

    private ResponseAPDU send(CommandAPDU apdu) throws CardProxyException {
        ResponseAPDU answer = sendApduToCardManager(apdu);
        if (!isOk(answer)) {
            throw new CardProxyException("Unknown SW:0x" + Integer.toHexString(answer.sw() & 0xFFFF));
        }
        return answer;
    }

    public void stopTest() throws CardProxyException {
        try {
            cad.release();
        } catch (Exception e) {
            e.printStackTrace(out);
        }
    }

    public void powerUp() throws CardProxyException {
        isInSession = false;
        out("GP.powerUp()");
            cad.powerUp();
        }

    public void powerDown() throws CardProxyException {
        out("GP.powerDown()");
        try {
            cad.powerDown();
        } catch (Exception ex) {
            ex.printStackTrace();
            throw new CardProxyException(I18n.getString("power.down.exception", ex));
        }
        if (resetDelay > 0) {
            synchronized (this) {
                try {
                    this.wait(resetDelay);
                } catch (Exception e) {

                }
            }
        }
    }

    public String convertPackage(String packageName, AppletID packageAid, 
            int majorVersion, int minorVersion,
            String classRootDir, String exportRootDir,
            boolean isExportMap, String outputRootDir,
            AppletProperties[] apa) throws CardProxyException {
        throw new UnsupportedOperationException(I18n.getString("not.supported.yet"));
    }

    public boolean loadClassFileApplets(String[] packageNames, 
            AppletProperties[] appletProperties,
            String classRootDir, String outputRootDir) throws CardProxyException {
        throw new UnsupportedOperationException(I18n.getString("not.supported.yet"));
    }

    public boolean loadCapFileApplets(String[] capfiles,
            AppletProperties[] appletProperties,
            String outputRootDir) throws CardProxyException {
        throw new UnsupportedOperationException(I18n.getString("not.supported.yet"));
    }

    public void selectInstaller() throws CardProxyException {
        //reset();
        out("GP.selectInstaller");
        if (!isInSession) {
            CommandAPDU apdu = new CommandAPDU(JC_CM_CLA, SELECT_INS, SELECT_BY_DF_NAME,
                    (byte)0x00, installerAid, (byte)0);
            ResponseAPDU answer = sendAPDU(apdu);
            if (!isOk(answer)) {
                throw new CardProxyException(I18n.getString("can.not.select.installer"));
            }
            out("START SESSION");
            answer = send("80 50 00 00 08 11 22 33 44 55 66 77 88 00");
            ExternalAuthenticator auth = new ExternalAuthenticator("4041424344454647 48494A4B4C4D4E4F",
                    "4041424344454647 48494A4B4C4D4E4F");
            CommandAPDU extAuth;
            try {
                extAuth = auth.createAnswer(Utils.parse("1122334455667788"),
                        answer.getResponseData());
            } catch (Exception e) {
                e.printStackTrace(out);
                throw new CardProxyException(I18n.getString("can.not.create.external.authenticate", e));
            }
            send(extAuth);
//            send("84 82 00 00 10 6C AB F3 4A CF AA 6C CB F3 46 3E BD 51 AE 8A 55");
            isInSession = true;
        }
    }

    public void startSession() throws CardProxyException {
        byte[] hostChallenge = Utils.parse("11 22 33 44 55 66 77 88 00");
        ResponseAPDU answer = send("80 50 00 00 08 11 22 33 44 55 66 77 88 00");
        byte[] data = answer.getBytes();
        byte[] challenge = new byte[24];
        System.arraycopy(data, 12, challenge, 0, 8);
        System.arraycopy(hostChallenge, 0, challenge, 8, 8);
        challenge[16] = (byte)0x80; //80 00 00 00 00 00 00 00' - Padding
        send(new CommandAPDU((byte)0x84, (byte)0x82, (byte)0x00, (byte)0x00,
                createExternalAuthenticate(challenge), (byte)0x00));
        //6C AB F3 4A CF AA 6C CB F3 46 3E BD 51 AE 8A 55 ");

    }
    private byte[] createExternalAuthenticate(byte[] request) throws CardProxyException {
        byte[] retVal = new byte[16];
        try {
            key.checksum.update(request, 0, request.length);
            byte[] hostCryptogram = key.checksum.calculateCheckSum();
            System.arraycopy(hostCryptogram, 0, retVal, 0, 8);
        } catch (GeneralSecurityException e) {
            throw new CardProxyException(I18n.getString("can.not.create.response", e));
        }
        return retVal;
    }

    public boolean installCAPFile(AppletProperties[] applets, String catFile,
            String outputDir) throws CardProxyException {
        isInGPAction = true;
        try {
            selectInstaller();
            GPCapFileInfo info = GPCapFileInfo.createData(out, catFile);
            for (AppletProperties apl : applets) {
                if (apl.getPackageAID() == null) {
                    apl.setPackageAID(info.packageAID);
                }
            }
            if (card != null) {
                card.notifyRegisteredCapFile(info.packageAID, applets);
            }
            return load(info) && install(info, applets);
        } catch (Exception e) {
            if (e instanceof CardProxyException) {
                throw (CardProxyException)e;
            }
            e.printStackTrace(out);
            throw new CardProxyException(I18n.getString("io.exception.cap.installation", e));
        } finally {
            isInGPAction = false;
        }
    }

    public boolean load(GPCapFileInfo info) throws CardProxyException, EncodingException {
        out("GP.load " + info);
        if (!installForLoad(info)) {
            out("!! install for load fails.");
            return false;
        }
        List<CommandAPDU> list = this.dataToLoadApdus(info.data);
        out("list.length=" + list.size());
        // TODO install(for load) apdu
        List<ResponseAPDU> answers = sendApduListToCardManager(list.toArray(new CommandAPDU[list.size()]));
        GPConstants data = GPConstants.getLoad();
        for (ResponseAPDU answer : answers) {
            out("STATUS: " + data.findName(answer.sw()));
            if (!isOk(answer)) {
                return false;
            }
        }
        return true;
    }

    public boolean installForLoad(GPCapFileInfo info) throws CardProxyException {
        out("GP.installForLoad " + info);
        AppletID aid = info.packageAID;
        byte[] aidBytes = aid.getBytes();
        byte[] data = new byte[aidBytes.length + 5];
        int offset = 0;
        offset = encodeArrayAsLV(data, offset, aidBytes); // Length of Load File AID and Load File AID
        data[offset++] = (byte)0x00;                      // Length of Security Domain AID,
        data[offset++] = (byte)0x00;                      // Length of Load File Data Block Hash
        data[offset++] = (byte)0x00;                      // Length of load parameters field
        data[offset++] = (byte)0x00;                      // Length of Load Token
        CommandAPDU apdu = new CommandAPDU(GP_CLA, INSTALL_INS,
                (byte)0x02, (byte)0x00, data, (byte)0x01);
        ResponseAPDU answer = sendApduToCardManager(apdu);
        out("STATUS: " + GPConstants.getInstall().findName(answer.sw()));
        return isOk(answer);
    }

    private static final byte[] EMPTY = new byte[0];
    private static byte[] nullToEmpty(byte[] data) {
        return (data == null) ? EMPTY : data;
    }

    public boolean installForInstall(String name, AppletID packageID, AppletProperties applet)
            throws EncodingException, CardProxyException {
        // Load File AID is package aid assert defined interface cap file;
        // Executable Module AID is the applet AID as it is defined in the cap file;
        // Application AID is the AID of the applet instance it is the same as Executable Module AID in TCK testing.
        out("GP.installForInstall " + name + applet);
        /*
         * Data Field for INSTALL [for install]
         * Mandatory 1 Length of Executable Load File AID
         * Mandatory 5-16 Executable Load File AID
         * Mandatory 1 Length of Executable Module AID
         * Mandatory 5-16 Executable Module AID
         * Mandatory 1 Length of Application AID
         * Mandatory 5-16 Application AID
         * Mandatory 1 Length of Application Privileges
         * Mandatory 1 Application Privileges
         * Mandatory 1 Length of install parameters field
         * Mandatory 2-n Install parameters field
         * Mandatory 1 Length of Install Token
         * Conditional 0-n Install Token
         */
        TLVEncoder out = new TLVEncoder(1024);
        byte[] packAidBytes = packageID.getBytes();
        byte[] appletAidBytes = applet.getAID().getBytes();
        encodeArrayAsLV(out, packAidBytes);       // Length of Executable Load File AID and Executable Load File AID
        encodeArrayAsLV(out, appletAidBytes);     // Length of Executable Module AID and Executable Module AID
        encodeArrayAsLV(out, appletAidBytes);     // Length of Application AID and Application AID
        out.out().write(0x01);                    // Length of Application Privileges
        out.out().write(applet.getPriviledges()); // Application Privileges

        out.startLV("Install parameters field");
        encodeInstallParameters(out, applet);     // Length of install parameters field and Install parameters field
        out.end("Install parameters field");

        out.out().write(0x00);                    // Length of Install Token

        CommandAPDU apdu = new CommandAPDU(
                GP_CLA, INSTALL_INS, (byte)0x04, (byte)0x00,
                out.toByteArray(), (byte)0x01);
        ResponseAPDU answer = sendApduToCardManager(apdu);
        out("STATUS: " + GPConstants.getInstall().findName(answer.sw()));
        return isOk(answer);
    }

    protected void encodeInstallParameters(TLVEncoder out, AppletProperties applet) throws EncodingException {
        byte[] applicationParams = nullToEmpty(applet.getParms());
        byte[] gpParams = applet.getGPParameters();
        if (gpParams != null) {
            out.out().write(gpParams);
        } else {
            out.startTag(APPLICATION_SPECIFIC_PARAMETERS_TAG);  // Application Specific Parameters TAG
            out.out().write(applicationParams);         // Application Specific Parameters length and value
            out.end();
//        if (systemParams != null) {
//            out.write(SYSTEM_SPECIFIC_PARAMETERS_TAG);
//            encodeArrayAsLV(out, systemParams);
//        }
            encodeNonZeroShortAsTLV(out, VOLATILE_DATA_SPACE_LIMIT_TAG, applet.getVolatileSpace());
            encodeNonZeroShortAsTLV(out, VOLATILE_DATA_SPACE_LIMIT_TAG, applet.getNonVolatileSpace());
        }
    }

    private void encodeNonZeroShortAsTLV(TLVEncoder out, byte tag, int value) throws EncodingException {
        if (value != 0) {
            out.startTag(tag);
            out.out().write(value, 2);
            out.end();
        }
    }

    String wA = "A0:00:00:00:62:03:01:0A:01:01:02:01:04:05:01:81";

    public boolean installForMakeSelectable(String name, AppletProperties applet) throws CardProxyException {
        out("GP.installForMakeSelectable " + name + " " + applet);
        byte[] aidBytes = applet.getAID().getBytes();
        byte[] data = new byte[7 + aidBytes.length];
        int pos = 0;
        data[pos++] = (byte)0x00;                   // Length = '00'
        data[pos++] = (byte)0x00;                   // Length = '00'
        pos = encodeArrayAsLV(data, pos, aidBytes); // Length and data of the Application AID
        data[pos++] = (byte)0x01;                   // Length of Application Privileges
        data[pos++] = (byte)0x00;                   // Application Privileges
        data[pos++] = (byte)0x00;                   // Length = '00'
        data[pos++] = (byte)0x00;                   // Length of Install Token
        CommandAPDU apdu = new CommandAPDU(GP_CLA, INSTALL_INS,
                (byte)0x08, (byte)0x00, data, (byte)0x01);
        ResponseAPDU answer = sendApduToCardManager(apdu);
        out("STATUS: " + GPConstants.getInstall().findName(answer.sw()));
        return isOk(answer);
    }

    public boolean deleteObject(AppletID aid, boolean deleteRelated) throws CardProxyException {
        out("GP.deleteObject " + aid  + " deleteRelated=" + deleteRelated);
        selectInstaller();
        byte p2 = (byte)(deleteRelated ? 0x80 : 0x00);
        byte[] aidBytes = aid.getBytes();
        byte[] data = new byte[aidBytes.length + 2];
        data[0] = AID_TAG;
        encodeArrayAsLV(data, 1, aidBytes);
        CommandAPDU apdu = new CommandAPDU(GP_CLA, DELETE_INS, (byte)0x00, p2,
                data, (byte)0x00);
        ResponseAPDU answer = sendApduToCardManager(apdu);
        out("STATUS: " + GPConstants.getDelete().findName(answer.sw()));
        return isOk(answer);

    }

    private int encodeArrayAsLV(byte[] out, int offset, byte[] data) {
        out[offset++] = (byte)data.length;
        System.arraycopy(data, 0, out, offset, data.length);
        return offset + data.length;
    }

    private void encodeArrayAsLV(TLVEncoder out, byte[] data) {
        out.out().write(data.length);
        out.out().write(data);
    }

    public static boolean isOk(ResponseAPDU answer) {
        return StatusCodes.isOkOrWarning(answer.sw());
//        return (answer.sw() == SW_OK) || ((answer.sw() & 0xFF00) == 0x6100);
    }

    public int getHeaderLength() {
        return 0;//150;//76; 15- works on G&D and 76 works on Gemalto
    }

    private ArrayList<CommandAPDU> dataToLoadApdus(TLV tlv) throws EncodingException {
        byte[] data = tlv.toByteArray();
        return dataToLoadApdus(data, 0 , data.length);
    }

    public ArrayList<CommandAPDU> dataToLoadApdus(byte[] data, int offset, int length) {
        ArrayList<CommandAPDU> retVal = new ArrayList<CommandAPDU>();
        int blockNumber = 0;
        int limit = offset + length;
        int apduLengthNet = APDU_LENGTH - getHeaderLength();//255 - 71;//APDU_LENGTH - getHeaderLength();
        for (int pos = offset; pos < limit;) {
            int blockLength = Math.min(apduLengthNet, data.length - pos);
            byte[] apduData = new byte[blockLength];
            System.arraycopy(data, pos, apduData, 0, blockLength);
            pos += blockLength;
            byte controlParameter = (pos == data.length) ? (byte)0x80 : (byte)0;
            retVal.add(new CommandAPDU(GP_CLA, LOAD_INS, controlParameter, (byte)blockNumber++, apduData,
                    (byte)((pos < limit) ? 0 : 1)));
        }
        return retVal;
    }

    public boolean install(GPCapFileInfo info, AppletProperties[] appletPropertiesArray) 
            throws EncodingException, CardProxyException {
        boolean retVal = true;
        for (AppletProperties appletProperties : appletPropertiesArray) {
            retVal = installForInstall(info.toString(), info.packageAID, appletProperties) && retVal;
            retVal = installForMakeSelectable(String.valueOf(info), appletProperties) && retVal;
        }
        return retVal;
    }

    public void reset() throws CardProxyException {
        powerDown();
        powerUp();
    }

    public ResponseAPDU sendAPDU(CommandAPDU capdu) throws CardProxyException {
        if (!isInGPAction) {
            isInSession = false;
        }
        out("  CAD:<<< " + capdu);
        ResponseAPDU retVal = cad.sendAPDU(capdu);
//            if ((retVal.sw() && 0xFF00) == 0x) {
//
//            }
        out("  CAD:>>> " + retVal);
        out.flush();
        return retVal;
    }

    public Hashtable getProperties() {
        throw new UnsupportedOperationException(I18n.getString("not.supported.yet"));
    }

    private int resetDelay = 0;

    protected Object[] getConfigurables() {
        return new Object[] { this };
    }

    public void init(String[] args, PrintWriter out, PrintWriter ref) {
        this.out = out;
        this.ref = ref;
        if (args == null) {
            return;
        }
        CommandLineAdapter adapter = new CommandLineAdapter(getConfigurables());
        adapter.init(args);
        out("(DRIVER type=" + cad.getClass().getSimpleName() + ")");
    }

    public boolean deleteAppletInstances(AppletProperties[] applets) throws CardProxyException {
        boolean retVal = true;
        for (AppletProperties applet : applets) {
            boolean status = deleteObject(applet.getAID(), false);
            if (status && (card != null)) {
                card.notifyDeletedCapFile(applet.getAID());
            }
            retVal = status && retVal;
        }
        return retVal;
    }

    public boolean deletePackage(AppletID packageID, String packageName) throws CardProxyException {
        boolean retVal = deleteObject(packageID, false);
        if (retVal && (card != null)) {
            card.notifyDeletedCapFile(packageID);
        }
        return retVal;
    }

    public boolean deletePackageAndInstances(AppletID packageID, String packageName) throws CardProxyException {
        boolean retVal = deleteObject(packageID, true);
        if (retVal && (card != null)) {
            card.notifyDeletedCapFile(packageID);
        }
        return retVal;
    }

    public List<ResponseAPDU> sendApduListToCardManager(CommandAPDU... apdus) throws CardProxyException {
        ArrayList<ResponseAPDU> retVal = new ArrayList<ResponseAPDU>();
        for (CommandAPDU apdu : apdus) {
            retVal.add(sendApduToCardManager(apdu));
        }
        return retVal;
    }

    public ResponseAPDU sendApduToCardManager(CommandAPDU apdu) throws CardProxyException {
        ResponseAPDU answer = sendAPDU(apdu);
        int sw = answer.sw();
        if ((sw & 0xFF00) != 0x6100) {
            return answer;
        } else {
            ArrayList<ResponseAPDU> list = new ArrayList();
            int length = 0; //0xFF & sw;
            while (((sw & 0xFF00) == 0x6100)
                   || ((sw & 0xFF00) == 0x6200)) {
                list.add(answer);
                length += answer.getLe();
                answer = sendAPDU(getGetResponseAPDU(apdu.getCLA(), sw));
                sw = answer.sw();
            }
            list.add(answer);
            length += answer.getLe();
            byte[] data = new byte[length + 2];
            int offset = 0;
            for (ResponseAPDU current : list) {
//                answer = (ResponseAPDU)e.next();
                for (int i = 0; i < answer.getLe(); i++) {
                    data[offset++] = (byte)current.getData(i);
                }
            }
            data[offset++] = (byte)(sw >> 8);
            data[offset++] = (byte)sw;
            return new ResponseAPDU(data);
        }
   }

     private CommandAPDU getGetResponseAPDU(byte original_cla, int sw) {
        return new CommandAPDU(new byte[] {
                                   (byte)((GET_RESPONSE_CLA_BYTE & 0xF0) | (original_cla & 0x0F)),
                                   (byte)0xc0,
                                   (byte)0x00, (byte)0x00,
                                   (byte)sw
                               });
    }

     private byte[] installerAid = Utils.parse("A000000003000000");


    private AppletProperties parse(String[] list, int offset) {
        AppletProperties retVal = new AppletProperties("unknown.package", null, "unknown.class");
        retVal.setAID(new AppletID(Utils.parse(list[offset++])));
        if (offset >= list.length) {
            return retVal;
        } else if (list[offset].startsWith("app:")) {
            retVal.setParms(Utils.parse(list[offset].substring(4)));
        } else if (!list[offset].isEmpty()) {
            retVal.setGPParameters(Utils.parse(list[offset]));
        }
        offset++;

        if ((list.length > offset) && (list[offset].length() > 0)) {
            retVal.setPriviledges(Integer.parseInt(list[offset++], 16));
        }
        return retVal;
    }

    public boolean install(String[] args) throws Exception {
        String capFile = args[0];
        AppletProperties[] applets = new AppletProperties[args.length - 1];
        for (int i = 0; i < applets.length; i++) {
            applets[i] = parse(args[i + 1].split(","), 0);
        }
        boolean retVal = installCAPFile(applets, capFile, ".");
        StringBuilder line = new StringBuilder("install(capFile=").append(capFile);
        for (AppletProperties applet : applets) {
            line.append(" ").append(applet.getAID());
        }
        line.append(")");
        out(line + (retVal ? " is SUCCESSFUL" : " is FAILED"));
        return retVal;
    }

    public void installApplet(String[] args) throws Exception {
        if (args.length < 2) {
            throw new IllegalArgumentException(I18n.getString("aid.not.defined"));
        }
        String val = args[0];
        AppletID parent = (Utils.isHexString(val)
                           ? new AppletID(Utils.parse(val))
                           : GPCapFileInfo.createData(out, val).packageAID);
        AppletProperties applet = parse(args, 1);
        boolean status = installForInstall(val, parent, applet);
        out("INSTALL(for INSTALL) " + (status ? "PASSED" : "FAILED"));
        if (status) {
            status = installForMakeSelectable(val, applet);
            out("INSTALL(for MAKE SELECTABLE) " + (status ? "PASSED" : "FAILED"));
        }
    }

    public void delete_Package(String[] args) throws IOException, CardProxyException {
        AppletID aid = new AppletID(Utils.parse(args[0]));
        boolean status = deletePackage(aid, "no-name");
        out("deletePackage(" + aid + ") returns " + status);
    }

    public void delete_Applet(String[] args) throws IOException, CardProxyException {
        AppletProperties applet = new AppletProperties("no-name", null, "no-name");
        AppletID aid = new AppletID(Utils.parse(args[0]));
        applet.setAID(aid);
        boolean status = deleteAppletInstances(new AppletProperties[] {
            applet
        });
        out("deleteApplet(" + aid + ") returns " + status);
    }

    public void delete(String[] args) throws IOException, CardProxyException {
        AppletID aid = new AppletID(Utils.parse(args[0]));
        boolean status = deletePackageAndInstances(aid, "no-name");
        out("deletePackageAndInstances(" + aid + ") returns " + status);
    }

    public ResponseAPDU select(String aid) throws CardProxyException {
        byte[] bytes = Utils.parse(aid);
        return sendAPDU(new CommandAPDU((byte)0x00, (byte)0xa4, (byte)0x04, (byte)0x00, bytes, 0x7f));
    }

    public void setDriver(String driver) throws Exception {
        CardReader cad;
        if (driver.equalsIgnoreCase("java")) {
            cad = new CardReader();
        } else {
            String name = "com.sun.tck.bvtool.etsi.gp." + driver;
            cad = (CardReader)Class.forName(name).newInstance();
        }
        setCad(cad);
    }

    public static void main(String[] args) throws Exception {
        if (args.length < 2) {
            System.out.println("Usage: <installer AID> <command> [<cap file>] [config file]" );
            return;
        }
        PrintWriter out = new PrintWriter(new OutputStreamWriter(System.out));
        try {
            GPCardService service = new GPCardService();
            service.setDriver("JPCSC");
            service.init(new String[] {args[0]}, out, out);
            service.startTest(".", 10);

            String[] methArgs = new String[args.length - 2];
            System.arraycopy(args, 2, methArgs, 0, methArgs.length);
            String command = args[1];
            invoke(service, command, methArgs);
        } finally {
            out.flush();
        }
    }

    public static Object invoke(Object instance, String name, String[] args)
            throws Exception {
        try {
            Method meth = instance.getClass().getMethod(name, args.getClass());
            return meth.invoke(instance, (Object)args);
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t instanceof Exception) {
                throw (Exception)t;
            } else {
                throw e;
            }
        } catch (NoSuchMethodException e) {
            // Ignore. Looking for the method without parameters.
        } catch (Exception e) {
            throw e;
        }
        Method meth = instance.getClass().getMethod(name, args.getClass());
        return meth.invoke(instance);
    }

    public void out(String msg) {
        if (out == null) {
            System.out.println(msg);
        } else {
            out.println(msg);
            out.flush();
        }
    }
}
