/*
 * %W% %E% Jonathan Gibbons
 * 
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved. 
 * Use is subject to license terms.
 */

package com.sun.jck.lib;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import com.sun.javatest.TestEnvironment;
import com.sun.javatest.TestFilter;
import com.sun.javatest.finder.HTMLTestFinder;
import com.sun.javatest.util.I18NResourceBundle;
import com.sun.javatest.util.StringArray;

public class JCKTestFinder extends HTMLTestFinder
{
    public void init(String[] args, File testSuiteRoot,
		     TestEnvironment env) throws Fault {
	setMode(DIR_WALK);
	decodeAllArgs(args);
	setRoot(testSuiteRoot);
	// we don't need env, so don't set it
    }

    public void setVerify(boolean f) {
	verify = f;
    }

    public void setRoot(File testSuiteRoot) throws Fault {
	int mode = getMode();

	if (testSuiteRoot.isDirectory()) {
	    File f = new File(testSuiteRoot, TESTSUITE_HTML);
	    if (!(f.exists() && !f.isDirectory() && f.canRead())) {
		throw new Fault(i18n, "jckFinder.badRootDir", 
				new Object[] {TESTSUITE_HTML, testSuiteRoot.getPath()});

	    }
	    if (mode == WEB_WALK)
		// force the test suite root to be the HTML file
		testSuiteRoot = f;
	}
	else {
	    String name = testSuiteRoot.getName();
	    if (!name.equals(TESTSUITE_HTML)) {
		throw new Fault(i18n, "jckFinder.badRootFile");
	    }
	    if (mode == DIR_WALK)
		testSuiteRoot = new File(testSuiteRoot.getParent());
	}

	super.setRoot(testSuiteRoot);
    }

    protected int decodeArg(String[] args, int i) throws Fault {
	if (args[i].equals("-verify")) {
	    verify = true;
	    return 1;
	}
	else if (args[i].equals("-allowEntry")) {
	    String e = args[i+1];
	    validEntries.put(e.toLowerCase(), e);
	    return 2;
	}
	else if (args[i].equals("-allowKeyword")) {
	    String k = args[i+1];
	    validKeywords.put(k.toLowerCase(), k);
	    return 2;
	}
	else
	    return super.decodeArg(args, i);
    }

    /**
     * Handle a test description entry read from a file.
     * Valid entries are put in the dictionary; others are ignored.
     * (Note you can specify additional entries to be considered valid, by
     * means of the args to init.) Error messages may optionally be generated
     * if bad entries are found.
     * @param entries   The dictionary of the entries being read
     * @param name	The name of the entry that has been read
     * @param value	The value of the entry that has been read
     */
    protected void processEntry(Map entries, String name, String value) {
	// this should be moved to JCKTestSuite
	boolean valid = (validEntries.get(name.toLowerCase()) != null);

	if (verify) {
	    if (!valid) {
		error(i18n, "jckFinder.unknownEntry", 
		      new Object[] {name, getCurrentFile()} );
	    }
	    if (name.equalsIgnoreCase("keywords")) {
		String[] keys = StringArray.split(value);
		boolean foundRuntime = false;
		boolean foundCompiler = false;
		if (keys != null) {
		    for (int i = 0; i < keys.length; i++) {
			String key = keys[i];
			if (validKeywords.get(key.toLowerCase()) == null) {
			    error(i18n, "jckFinder.unknownKeyword", 
				  new Object[] {key, getCurrentFile()} );
			}
			if (key.equals("runtime"))
			    foundRuntime = true;
			else if (key.equals("compiler"))
			    foundCompiler = true;
		    }
		}
		if (!(foundRuntime || foundCompiler)) {
		    error(i18n, "jckFinder.needCompOrRuntimeKeyword", getCurrentFile());
		}
	    }
	}

	if (valid) {
	    if (name.equalsIgnoreCase("remote")) {
		// allow multiple env values, separate them by \n
		String prevValue = (String)(entries.get(name));
		if (prevValue != null)
		    value = prevValue + "\n" + value.trim();
	    }
	    super.processEntry(entries, name, value);
	}
    }

    protected Map normalize(Map entries) {
	if (verify) {
	    for (int i = 0; i < requiredEntries.length; i++) {
		if (entries.get(requiredEntries[i]) == null) 
		    error(i18n, "jckFinder.requiredEntryNotFound", 
			  new Object[] {requiredEntries[i], getCurrentFile()});
		    
	    }
	}
	return entries;
    }

    private static Map initTable(String[] entries) {
	Map map = new HashMap();
	for (int i = 0; i < entries.length; i++)
	    map.put(entries[i].toLowerCase(), entries[i]);
	return map;
    }

    private boolean verify;
    private String[] requiredEntries = getRequiredEntries();
    private Map validEntries = initTable(getValidEntries());
    private Map validKeywords = initTable(getValidKeywords());

    private static final String TESTSUITE_HTML = "testsuite.html";
    private static I18NResourceBundle i18n = I18NResourceBundle.getBundleForClass(JCKTestFinder.class);

    public static String[] getRequiredEntries() {
	return new String[] {
	    // required
	    "keywords",
	    "source",
	    "title"
	};
    }

    public static String[] getValidEntries() {
	return new String[] {
	    // required
	    "keywords",
	    "source",
	    "title",
	    // optional
	    "context",
	    "executeArgs",
	    "executeClass",
	    "executeNative",
	    "id",		// defined and used internally by JavaTest
	    "remote",
	    "rmicClass",	// temp for backwards compatible for Kestrel
	    "rmicClasses",
	    "selectIf",
	    "executeIf",	// temp for backwards compatible
	    "timeout"
	};
    }

    public static String[] getValidKeywords() {
	return new String[] {
	    // approved
	    "compiler",
	    "runtime",
	    "positive",
	    "negative",
	    "idl_inherit",
	    "idl_tie",
	    "interactive",
	    "jniinvocationapi",
	    "only_once",
	    "rmi_iiop",
	    "rmi_v11",
	    // will eventually be superceded/deprecated
	    "serial"
	};
    }

}
