/*
 * %W% %E% Jonathan Gibbons
 * 
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved. 
 * Use is subject to license terms.
 */

package com.sun.jck.lib;

import java.util.ResourceBundle;
import com.sun.javatest.TestDescription;
import com.sun.javatest.TestEnvironment;
import com.sun.javatest.TestFilter;
import com.sun.javatest.util.I18NResourceBundle;

/**
 * Filter to select JCK tests based on test-specific dependencies
 */
public class JCKTestFilter extends TestFilter
{
    public JCKTestFilter(TestEnvironment env, int product, int mode) {
	this.env = env;
	this.product = product;
	this.mode = mode;
    }

    public String getName() {
	return i18n.getString("jckFilter.name");
    }

    public String getDescription() {
	return i18n.getString("jckFilter.description");
    }

    public String getReason() {
	return i18n.getString("jckFilter.reason");
    }

    public boolean accepts(TestDescription td) throws Fault {
	// compiler tests must have compiler keyword; runtime tests must have runtime keyword
	String keywords = td.getParameter("keywords");
	String reqdKeyword = (product == JCKTestSuite.RUNTIME ? "runtime" : "compiler");
	if (!containsKeyword(keywords, reqdKeyword))
	    return false;

	// if test is to be executed, do runtime test selection;
	// if precompiling, always select it
	switch (mode) {
	case JCKTestSuite.PRECOMPILE:
	    return true;

	case JCKTestSuite.DEVELOPER:
	case JCKTestSuite.CERTIFY:
	    try {
		String s = td.getParameter("selectIf");
		if (s == null || s.length() == 0) {
		    s = td.getParameter("executeIf");  // deprecated
		    if (s == null || s.length() == 0)
			return true;
		}
		
		Expr e = Expr.parse(s);
		return e.evalBoolean(env);
	    }
	    catch (Expr.Fault e) {
		throw new Fault(i18n, "jckFilter.error", 
				new Object[] {td.getRootRelativeURL(), e.getMessage()});
	    }
	default:
	    throw new IllegalStateException();
	}
    }

    public boolean equals(Object o) {
	if (o == this)
	    return true;

	if (!(o instanceof JCKTestFilter))
	    return false;

	JCKTestFilter other = (JCKTestFilter) o;
	return (this.env == env && product == other.product && mode == other.mode);
    }

    private boolean containsKeyword(String all, String reqd) {
	int reqdLen = reqd.length();
	int allLen = (all == null ? 0 : all.length());
	for (int i = 0; i <= allLen - reqdLen; i++) {
	    if (all.regionMatches(true, i, reqd, 0, reqdLen)) {
		if ( (i == 0 || isBoundaryCh(all.charAt(i-1))) 
				&& (i+reqdLen == all.length() || isBoundaryCh(all.charAt(i+reqdLen))) )
		    return true;
	    }
	}
	return false;
    }

    private static boolean isBoundaryCh(char c) {
	return !(Character.isUnicodeIdentifierStart(c)
		 || Character.isUnicodeIdentifierPart(c));
    }

    private TestEnvironment env;
    private int product;
    private int mode;
    private static I18NResourceBundle i18n = I18NResourceBundle.getBundleForClass(JCKTestFilter.class);
}
