/*
 * %W% %E%
 * 
 * @Copyright
 */

package com.sun.jck.lib;

import java.io.File;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import com.sun.javatest.Command;
import com.sun.javatest.Status;
import com.sun.javatest.util.DirectoryClassLoader;
import com.sun.javatest.util.WriterStream;
// NOTE: This class also uses (explicitly) the following classes
//	javasoft.sqe.harness.Status
//	javasoft.sqe.harness.Test
//	javasoft.sqe.javatest.Status
//	javasoft.sqe.javatest.Test

/**
 * ExecJCKTestSameJVMCmd executes a JCK test in the same Java Virtual
 * Machine as the caller of this class.
 *
 * <p> JCK tests may be "standard" tests, which implement the Test interface,
 * and which return a result via the value returned from the run method,
 * or they may be a "simple" which implements a standard simple method
 * which an integer return code.
 *
 * <p> It can use either a private class loader or the system class loader.
 * A private class loader will be created if the -loadDir option is given;
 * otherwise the system class loader will be used.  A private class
 * loader minimizes the interference between tests, but you may be
 * restricted from using private class loaders if you are running the
 * harness inside a web browser.
 *
 * <p> If the the <code>-repeat</code> option is provided, then the test will be
 * run multiple times in the same JVM.  <code>Status.error()</code> will be
 * returned (and the remainder of the iterations will not be performed) if any
 * repetition of the test returns an error, or if the status return type changes
 * between iterations.  The returned status after each iteration will be
 * included in the log. If this option is not given, the test will be run once.
 *
 * @see javasoft.sqe.javatest.lib.ExecJCKTestOtherJVMCmd
 * @see javasoft.sqe.javatest.lib.ExecStdTestSameJVMCmd
 *
 * @author Jonathan J Gibbons
 * @version 01/03/03
 */
public class ExecJCKTestSameJVMCmd extends Command
{
    /**
     * entry point for stand-alone debugging.
     */
    public static void main(String[] args) {
	Command c = new ExecJCKTestSameJVMCmd();
	PrintWriter log = new PrintWriter(new OutputStreamWriter(System.err));
	PrintWriter ref = new PrintWriter(new OutputStreamWriter(System.out));
	Status s = c.run(args, log, ref);
	log.flush();
	ref.flush();
	s.exit();
    }

    /**
     * The method that that does the work of the command.
     * @param args	[-loadDir <em>dir</em>] <em>executeClass</em> <em>executeArgs</em>
     * @param log	A stream to which to report messages and errors
     * @param ref	A stream to which to write reference output
     * @return		The result of the command
     */
    public Status run(String[] args, PrintWriter log, PrintWriter ref) {
	int repeat = 1;
	boolean finalizeWhenDone = false;
	String className = null;
	String[] executeArgs = { };
	ClassLoader loader = getClassLoader();

	int i = 0;

	for (; i < args.length && args[i].startsWith("-"); i++) {
	    if ("-loadDir".equalsIgnoreCase(args[i]) && i+1 < args.length) {
		// -loadDir is optional; if given, a new class loader will be created
		// to load the class to execute;  if not given, the system class loader
		// will be used.
		loader = new DirectoryClassLoader(new File(args[++i]));
	    } 
	    else if ("-repeat".equalsIgnoreCase(args[i]) && i+1 < args.length) {
		// -repeat is optional; if given, the test will be run that
		// number of times (in the same JVM)
		try {
		    if ((repeat = Integer.parseInt(args[++i])) < 1)
			return Status.error(I18n.getString("incorrect.repetitions.number", String.valueOf(repeat)));
		}
		catch (NumberFormatException e) {
		    return Status.error(I18n.getString("incorrect.repetitions.number", String.valueOf(repeat)));
		}
	    }
	    else if ("-finalizeWhenDone".equalsIgnoreCase(args[i])) {
		finalizeWhenDone = true;
	    }
	    else
		throw new Error(I18n.getString("argument.not.recognized", getClass().getName(), args[i]));
	}

	// Next must come the executeClass
	if (i < args.length) {
	    className = args[i];
	    i++;
	} else
	    return Status.failed(I18n.getString("executeclass.not.specified"));

	// Finally, any optional args
	if (i < args.length) {
	    executeArgs = new String[args.length - i];
	    System.arraycopy(args, i, executeArgs, 0, executeArgs.length);
	}

	try {
	    Class c;
	    try {
		if (loader == null)
		    c = Class.forName(className);
		else
		    c = loader.loadClass(className);
	    }
	    catch (ClassNotFoundException e) {
		return Status.failed(I18n.getString("cant.load.test", e));
	    }
	    catch (VerifyError e) {
		return Status.failed(I18n.getString("class.load.verification.error", className, e));
	    }
	    catch (LinkageError e) {
		return Status.failed(I18n.getString("class.load.linking.error", className, e));
	    }
	    catch (ThreadDeath e) {
		throw (ThreadDeath)(e.fillInStackTrace());
	    }
	    catch (Throwable e) {
		String t = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
		PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
		e.printStackTrace(ps);
		ps.close();
		return Status.error(I18n.getString("class.load.unexpected.error", t, className, e));
	    }
	    
	    Status prevStatus = null;
	    for (int j = 0; j < repeat; j++) {
		if (repeat > 1)
		    log.println("iteration: " + (j+1));
		
		Status currStatus;
		if (javasoft.sqe.javatest.Test.class.isAssignableFrom(c)) {
		    currStatus = executeStandardTest(c, executeArgs, log, ref);
		}
		else if (javasoft.sqe.harness.Test.class.isAssignableFrom(c)) {
		    currStatus = executeOldTest(c, executeArgs, log, ref);
		}
		else
		    currStatus = executeSimpleTest(c, executeArgs, log);
		
		if (repeat > 1)
		    log.println("   " + currStatus);
		
		if ((prevStatus != null) && (currStatus.getType() != prevStatus.getType()))
		    currStatus = Status.error(I18n.getString("return.status.changed", String.valueOf(j+1)));
		
		if (!currStatus.isError())
		    prevStatus = currStatus;
		else
		    return currStatus;
	    }

	    return prevStatus;
	}
	finally {
	    if (finalizeWhenDone) {
		loader = null;
		System.gc();
		System.runFinalization();
	    }
	}

    }

    private Status executeStandardTest(Class c, String[] executeArgs,
				       PrintWriter log, PrintWriter ref) {
	javasoft.sqe.javatest.Test t;

	try {
	    t = (javasoft.sqe.javatest.Test)(c.newInstance());
	}
        catch (ClassCastException e) {
	    return Status.error(I18n.getString("interface.not.found"));
	}
	catch (InstantiationException e) {
	    return Status.error(I18n.getString("test.instantiate.error",e));
	}
	catch (IllegalAccessException e) {
	    return Status.error(I18n.getString("test.illegal.access",e));
	}
	catch (ThreadDeath e) {
	    throw (ThreadDeath)(e.fillInStackTrace());
	}
	catch (VerifyError e) {
	    return Status.failed(I18n.getString("class.instantiate.verification.error", c.getName(), e));
	}
	catch (LinkageError e) {
	    return Status.failed(I18n.getString("class.instantiate.linking.error", c.getName(), e));
	}
	catch (Throwable e) {
	    String tt = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.printStackTrace(ps);
	    ps.close();
	    return Status.error(I18n.getString("class.instantiate.unexpected.error", tt, c.getName(), e));
	}

	try {
	    javasoft.sqe.javatest.Status s = t.run(executeArgs, log, ref);
	    return new Status(s.getType(), s.getReason());	    
	}
	catch (ThreadDeath e) {
	    throw (ThreadDeath)(e.fillInStackTrace());
	}
	catch (VerifyError e) {
	    return Status.failed(I18n.getString("class.execute.verification.error", c.getName(), e));
	}
	catch (LinkageError e) {
	    return Status.failed(I18n.getString("class.execute.linking.error", c.getName(), e));
	}
	catch (Throwable e) {
	    String tt = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.printStackTrace(ps);
	    ps.close();
	    return Status.error(I18n.getString("class.execute.unexpected.error", tt, c.getName(), e));
	}
    }

    private Status executeOldTest(Class c, String[] executeArgs,
				  PrintWriter log, PrintWriter ref) {
	javasoft.sqe.harness.Test t;

	try {
	    t = (javasoft.sqe.harness.Test)(c.newInstance());
	}
        catch (ClassCastException e) {
	    return Status.error(I18n.getString("interface.not.found"));
	}
	catch (InstantiationException e) {
	    return Status.error(I18n.getString("test.instantiate.error",e));
	}
	catch (IllegalAccessException e) {
	    return Status.error(I18n.getString("test.illegal.access",e));
	}
	catch (ThreadDeath e) {
	    throw (ThreadDeath)(e.fillInStackTrace());
	}
	catch (Throwable e) {
	    String tt = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.printStackTrace(ps);
	    ps.close();
	    return Status.error(I18n.getString("class.instantiate.unexpected.error", tt, c.getName(), e));
	}

	try {
	    javasoft.sqe.harness.Status s = t.run(executeArgs,
			Deprecated.createPrintStream(new WriterStream(log)),
			Deprecated.createPrintStream(new WriterStream(ref)));
	    return new Status(s.getType(), s.getReason());
	}
	catch (ThreadDeath e) {
	    throw (ThreadDeath)(e.fillInStackTrace());
	}
	catch (Throwable e) {
	    String tt = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.printStackTrace(ps);
	    ps.close();
	    return Status.error(I18n.getString("class.execute.unexpected.error", tt, c.getName(), e));
	}
    }

    private Status executeSimpleTest(Class c, String[] executeArgs,
				       PrintWriter log) {
	Method runMethod;

	try {

	    Class[] runParamTypes = {String[].class, PrintStream.class};
	    Method[] mm = c.getDeclaredMethods();

	    if (false) {
		log.println("class: " + c);
		for (int ii = 0; ii < mm.length; ii++)
		    log.println("method " + ii + ": " + mm[ii]);
		log.println("runParamTypes[0]: " + runParamTypes[0]);
		log.println("runParamTypes[1]: " + runParamTypes[1]);
	    }

	    runMethod = c.getMethod("run", runParamTypes);

	    if (false)
		log.println("method: " + runMethod);
	}
        catch (NoSuchMethodError e) {
	    return Status.failed(I18n.getString("method.not.found", "run", e));
	}
        catch (NoSuchMethodException e) {
	    return Status.failed(I18n.getString("method.not.found", "run", e));
	}

	try {
	    Object[] runArgs = {executeArgs, Deprecated.createPrintStream(new WriterStream(log))};
	    Object result = runMethod.invoke(null, runArgs);

	    if (false)
		log.println("result: " + result);

	    switch (((Integer)result).intValue()) {
	    case 0:
		return Status.passed("");
	    case 2:
		return Status.failed("");
	    default:
		return Status.failed(I18n.getString("unexpected.exit.code",result));
	    }
	}
        catch (ClassCastException e) {
	    return Status.failed(I18n.getString("method.noninteger.result", "run"));
	}
	catch (IllegalAccessException e) {
	    return Status.failed(I18n.getString("test.illegal.access",e));
	}
        catch (InvocationTargetException e) {
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.getTargetException().printStackTrace(ps);
	    ps.close();
	    return Status.failed(I18n.getString("method.thrown.exception", "run", e));
	}
	catch (ThreadDeath e) {
	    throw (ThreadDeath)(e.fillInStackTrace());
	}
	catch (Throwable e) {
	    String tt = (e instanceof Exception ? "exception" : e instanceof Error ? "error" : "throwable");
	    PrintStream ps = Deprecated.createPrintStream(new WriterStream(log));
	    e.printStackTrace(ps);
	    ps.close();
	    return Status.error(I18n.getString("class.execute.unexpected.error", tt, c.getName(), e));
	}
    }
}
