/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.referenceimpl;

import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketAddress;
import java.net.InetSocketAddress;
import java.net.BindException;
import java.net.ConnectException;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Properties;

import com.sun.javacard.apduio.Apdu;
import com.sun.javacard.apduio.CadClient;
import com.sun.javacard.apduio.TLP224Exception;
import com.sun.javacard.apdutool.ParseException;
import com.sun.javacard.apdutool.ScriptCommand;
import com.sun.javacard.apdutool.ScriptParser;

import com.sun.javacard.cjck.scripts.ScriptFailException;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.DeploymentException;
import com.sun.javacard.cjck.userinterface.FrameworkException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.userinterface.CommunicationService;
import com.sun.javacard.referenceimpl.ProcessManager.Descriptor;
import com.sun.javacard.cjck.I18n;

/**
 * A socket-based interface to Sun's C-JCRE
 * (C language Java Card Runtime Environment).
 * <br>This is an example of a CJCKCardService which a licensee must implement.
 * <em>This code is provided as an example.  Do not assume that this code is totally correct.</em>
 * The {@link CJCKCardService} interface is the definitive specification.
 * <br>The C-JCRE is a Java Card simulator which does simulation at the byte code level.
 *
 * @version Version:1.62 Date:02/16/06
 *
 * @author Eric Nellen (eric.nellen@eng.sun.com)
 */

public class CJCRECardService implements CJCKCardService {
    
    protected boolean crefRunning = false;
    protected int basePort = 0;
    protected boolean isMainCommunicationService = false;
    
    public static final String OPT_DEBUG_LEVEL      = "DEBUG_LEVEL";
    public static final String OPT_RI               = "RI";
    public static final String OPT_RI_RUN           = "RI_RUN";
    public static final String OPT_COMVERTER_PREFIX = "-PC";
    public static final String OPT_CREF_PREFIX      = "-PV";
    public static final String OPT_APDUTOOL_PREFIX  = "-PA";
    public static final String OPT_SCRIPTGEN_PREFIX = "-PS";
    public static final String OPT_TRANSPORT_PROTOCOL = "TP";
    public static final String T0_PROTOCOL_OPTION = "t0";
    public static final String T1_PROTOCOL_OPTION = "tdual";
    
    public static final Boolean INT_SUPPORT = new Boolean("true");
    public static final Integer MAJOR_VERSION = new Integer(CJCKCardService.MAJOR_VERSION);
    public static final Integer MAX_NVRAM = new Integer(4096);
    public static final Integer MAX_STACK = new Integer(256);
    public static final Integer MAX_TRANSIENT_RAM = new Integer(128);
    public static final Integer MINOR_VERSION = new Integer(CJCKCardService.MINOR_VERSION);
    public static final String API_VERSION = "2.2.1";
    public static final String BUILD_DATE = "10/08/03";
    public static final String BUILD_VERSION = "@(#)";
    public static final String MYNAME = "CJCRECardService";
    public static final String PROXY_NAME = MYNAME;
    public static final String PROXY_VERSION = "2.2.1";
    public static final String RELEASE_VERSION = "2.2.1";
    public static final long SLEEP_TIME = 50; // milliseconds we sleep each iteration
    
    protected String classpath = "";
    protected static final String CONVERTER_CLASSNAME = "com.sun.javacard.converter.Converter";
    protected static final String DEFAULT_CONFIG_FILE_NAME = "cjcretemp.cfg";
    protected static final String DEFAULT_HOST_NAME = "localhost"; // "127.1";
    protected static final String FILE_SEPARATOR = System.getProperty("file.separator");
    protected static final String JAVA_HOME = System.getProperty("java.home");
    protected static ArrayList JAVA_COMMAND;
    static {
        JAVA_COMMAND = new ArrayList();
        JAVA_COMMAND.add(JAVA_HOME
                + FILE_SEPARATOR + "bin" + FILE_SEPARATOR + "java");
        JAVA_COMMAND.add("-classpath");
    }
    protected static final String JAVA_VERSION = System.getProperty("java.version");
    protected static final String JCRE_OPTION_RESTART = "-i";  // -i <filename>
    protected static final String JCRE_OPTION_SAVE = "-o";  // -o <filename>
    protected static final String JCRE_STATE_FILENAME = "JCRE-EEPROM-SAVE-";
    protected static final String JCRE_STATE_FILENAME_SUFFIX = ".dat";
    protected static final String REFIMPL_NAME = "C-JCRE";
    protected static final String SCRIPTGEN_CLASSNAME = "com.sun.javacard.scriptgen.Main";
    protected static final String SCRIPT_COMMENT = "//";
    protected static final String SCRIPT_FILE_SUFFIX = ".script.txt";
    protected static final int DEFAULT_DEBUG_LEVEL = Logger.MESSAGE;
    protected static final int DEFAULT_PORT = 9025;
    protected static final int MAX_ATTEMP = 2000;
    protected static final int CREF_CONNECT_ATTEMPTS = 100;
    protected static final int CREF_RESTART_ATTEMPTS = 10;
    protected static final int SW_RETURN = 0x9000;
    protected static String[] EXEC_ENVIRONMENT;
    protected boolean isCleanTempFiles = true;
    protected static final  byte[] SELECT_INSTALLER_APDU = {
        (byte)0x00, (byte)0xA4, (byte)0x04, (byte)0x00, (byte)0x09, (byte)0xA0,
        (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x62, (byte)0x03, (byte)0x01,
        (byte)0x08, (byte)0x01, (byte)0x7F
    };
    
    protected BufferedInputStream bin;
    protected BufferedOutputStream bos;
    
    protected Socket sock;
    protected String workingDir;
    
    protected int port = 0;
    
    protected boolean poweredUp = false;
    protected int recvApduNum;
    protected int sendApduNum;
    
    protected ArrayList converter_options = new ArrayList();
    protected ArrayList cref_options = new ArrayList();
    protected ArrayList apdutool_options = new ArrayList();
    protected ArrayList scriptgen_options = new ArrayList();
    
    protected CadClient cad;
    protected Logger log;
    protected ProcessManager manager;
    protected Properties serviceProperties = new Properties();
    protected int proxyStartCount = 0;
    protected long taskTimeout = 0 /* FOREVER */;
    protected Descriptor currentProcDescriptor;
    protected byte transportProtocol;
    protected String transportProtocolName = "T0";
    
    // parameters from init method
    protected String[] args;
    protected PrintWriter out;
    protected PrintWriter ref;
    
    protected CJCRECardService contactlessService;
    /**
     * Creates a new C-JCRE CardService using a socket.
     * This is an example of a CJCKCardService that a licensee would implement.
     * The service cannot be used immediately. The actual initialization
     * is triggered by an invocation of <tt>initialize</tt>.
     */
    public CJCRECardService() {
    }
    
    
    /**
     * Creates a new C-JCRE card service using a socket.
     */
    public CJCRECardService(int port)
    throws java.io.IOException {
        this.port = port;
    }
    
    
    /**
     * Prepare to start a test.  Initialize the Proxy.
     * Called by the cJCK to notify the Card Service that a test is starting, and
     * specify the path of the working directory.
     * <P>Note:<ul>
     * <li>The Card Service should prepare the working directory for the test.
     * This includes deleting any intermediate files that may have been left over
     * from a prior test execution.
     * <li>The cJCK assumes that no test applets exist in either the ROM image or EEPROM.
     * </ul>
     * @param   workingDir
     *     The <tt>working directory</tt> to be used for this test.
     * @param   numberOfExecutions
     *     Indicates the number of times that the proxy will be powered on.
     * @exception FrameworkException
     *     Thrown if unable to initialize or set the working directory.
     */
    public void startTest(String workingDir, int numberOfExecutions)
    throws FrameworkException {
        this.workingDir = workingDir;
        log.log(I18n.getString("start.test", workingDir), Logger.DEBUG);
        if (contactlessService != null) {
            contactlessService.workingDir = workingDir;
        }
        cleanupWorkingDir();
    }
    
    /**
     * Stop the test.  Terminate the Proxy.
     * Called by the cJCK to notify the Card Service that a test is stopping.
     * @exception FrameworkException
     *     Thrown if unable to terminate the Proxy.
     */
    public void stopTest() throws FrameworkException {
        log.log(I18n.getString("stop.test"), Logger.DEBUG);
        if (script_file != null) {
            script_file.close();
        }
        
        if (isCleanTempFiles) {
            for (Iterator e = tempFiles.iterator(); e.hasNext();) {
                try {
                    (new File((String)e.next())).delete();
                } catch (Throwable t) {
                }
            }
        }
        for (Iterator e = tempFiles.iterator(); e.hasNext();) {
            try {
                (new File((String)e.next())).delete();
            } catch (Throwable t) {
            }
        }
        if (getCurrentProcDescriptor() != null) {
            getCurrentProcDescriptor().close(true);
        }
        manager.closeAllProcesses();
    }
    
    
    /* Power-up the Proxy.
     *
     * @exception FrameworkException
     *     Thrown if reset failed.
     */
    public void powerUp() throws FrameworkException {
        if (poweredUp) {
            log.log(I18n.getString("already.power.up"), Logger.DEBUG);
            return;
        }
        if (script_file != null) {
            script_file.println(I18n.getString("power.up"));
        }
        log.log(I18n.getString("powerup.up.start"), Logger.DEBUG_TRACE);
        startProxy();
        if (cad == null) {
            return;
        }
        /**/
    }
    
    
    /* Power-down the Proxy.
     *
     * @exception FrameworkException
     *     Thrown if reset failed.
     */
    public void powerDown() throws FrameworkException {
        
        log.log(I18n.getString("power.down"), Logger.DEBUG);
        if (!poweredUp) {
            log.log(I18n.getString("not.powered.up.yet"), Logger.DEBUG);
            return;
        }
        
        if (script_file != null) {
            script_file.println(I18n.getString("power.down.1"));
        }
        if (sendApduNum > recvApduNum) {
            log.log(I18n.getString("proxy.hung.failed", new Integer(sendApduNum)), Logger.DEBUG);
            recvApduNum = sendApduNum;
        }
        
        if (cad == null) {
            return;
        }
        try {
            if (cad != null) {
                powerDownCad();
            }
        } catch (TLP224Exception e) {
            log.log(I18n.getString("power.down.exception", e), Logger.DEBUG);
        } catch (IOException e) {
            try {
                powerDownCad();
            } catch (TLP224Exception e1) {
            } catch (IOException e1) {}
            log.log(I18n.getString("power.down.exception", e), Logger.DEBUG);
        }
        
        stopCref();
        
        log.log(I18n.getString("proxy.has.powered.down"), Logger.DEBUG);
        poweredUp = false;
        
    }
    
    public void stopCref() throws FrameworkException {
        closeSocket(sock);
        stopProxy(); // stop the Proxy
    }
    /* Reset the Proxy.
     *
     * @exception FrameworkException
     *     Thrown if reset failed.
     */
    public void reset() throws FrameworkException {
        
        log.log(I18n.getString("reset.cjcre.process"), Logger.DEBUG);
        powerDown();
        powerUp();
    }
    
    
    public String convertPackage(String packageName,
            AppletID packageAid,
            int majorVersion,
            int minorVersion,
            String classRootDir,
            String exportRootDir,
            boolean isExportMap,
            String outputDir,
            AppletProperties[] apa)
            throws FrameworkException {
        log.log(I18n.getString("convert.package"), Logger.DEBUG_TRACE);
        log.log(I18n.getString("preparing.convert.package", new Object[] {
            packageName, new Integer(majorVersion),
                    new Integer(minorVersion), this.converter_options}),
                            Logger.DEBUG_TRACE);
                    log.log(I18n.getString("convert.package.info", new Object[] {classRootDir,
                            exportRootDir, isExportMap ? Boolean.TRUE : Boolean.FALSE,
                            outputDir}),
                                    Logger.DEBUG_TRACE);
                            ArrayList cmd = (ArrayList)JAVA_COMMAND.clone();
                            cmd.add(classpath);
                            cmd.add(CONVERTER_CLASSNAME);
                            cmd.add("-exportpath");
                            cmd.add(exportRootDir);
                            if (isExportMap) {
                                cmd.add("-exportmap");
                            }
                            cmd.add("-classdir");
                            cmd.add(classRootDir);
                            
                            for (int i = 0; i < apa.length; i++) {
                                AppletProperties ap = apa[i];
                                cmd.add("-applet");
                                cmd.add(convertAID(ap.getAID(), "0x", ":"));
                                cmd.add(ap.getClassName());
                            }
                            
                            cmd.add("-d");
                            cmd.add(outputDir);
                            cmd.add("-out");
                            cmd.add("JCA");
                            cmd.add("EXP");
                            cmd.add("CAP");
                            
                            cmd.addAll(converter_options);
                            
                            cmd.add(packageName);
                            cmd.add(convertAID(packageAid, "0x", ":"));
                            cmd.add(Integer.toString(majorVersion) + "."
                                    + Integer.toString(minorVersion));
                            doExec("Converter", cmd, taskTimeout, 0);
                            
                            String capFile = outputDir + FILE_SEPARATOR
                                    + packageName.replace('.', File.separatorChar) + FILE_SEPARATOR
                                    + "javacard" + FILE_SEPARATOR
                                    + packageName.substring(packageName.lastIndexOf(".") + 1,
                                    packageName.length()) + ".cap";
                            
                            if (!(new File(capFile)).exists()) {
                                throw new FrameworkException(I18n.getString("capfile.creation.exception"));
                            }
                            return capFile;
    }
    
    /**
     * This method converts an AID
     * to a formatted string (e.g.,  0x0:0x1:0x2:0x3:0x4 ).
     */
    String convertAID(AppletID aid, String prefix, String suffix)
    throws FrameworkException {
        
        StringBuffer newAID = new StringBuffer();
        byte [] aidba = aid.getBytes();
        String token;
        
        if (aidba.length > 0) {
            newAID.append(prefix);
            newAID.append(Integer.toHexString((int)aidba[0] & 0xFF));
        }
        
        for (int i = 1; i < aidba.length; i++) {
            newAID.append(suffix);
            newAID.append(prefix);
            newAID.append((Integer.toHexString((int)aidba[i] & 0xFF)));
        }
        return newAID.toString();
    }
    
    
    
    /**
     * Loads the specified applets from their class files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support class file format.
     * <br>How the loading is done and where the applets are loaded
     * is at the discretion of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param packageNames {@link AppletProperties} array with the
     * package names. The array contains names of the all packages,
     * which are required for the test execution.
     * @param appletProperties []an {@link AppletProperties} array
     *     which specifies the applets to be loaded.
     * @param classRootDir the root directory of the class hierarchy.
     * The path /java/lang/javacard is appended to this root,
     * yielding the directory in which the Converter will look for classes.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @exception DeploymentException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadClassFileApplets(String[] packages,
            AppletProperties[] apa,
            String classDir,
            String outputDir)
            throws DeploymentException {
        
        throw new DeploymentException(I18n.getString("loadclassfileapplet.not.applicable"));
    }
    
    
    /**
     * Loads the specified applets from their cap files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support CAP file format.
     * <br>How the loading is done and where the applets are loaded
     * is at the discretion of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param capfiles {@link AppletProperties} array with the
     * cap file names. The array contains names of the all cap files,
     * which are required for the test execution.
     * @param appletProperties []an {@link AppletProperties} array
     *     which specifies the applets to be loaded.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @exception DeploymentException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadCapFileApplets(String[] capfiles,
            AppletProperties[] apa,
            String outputDir)
            throws DeploymentException {
        throw new DeploymentException(I18n.getString("loadcapfileapplet.not.applicable"));
    }
    
    /**
     * Installs a CAP file into the Proxy.
     * Assumes that powerUp() has already been called.
     * @param appletPropertiesArray {@link AppletProperties}
     *    specifies the applets in the CAP file.
     * @param aCAPFileName is a fully-qualified file name including the path name.
     * @param outputRootDir the root directory for output.
     * Any intermediate or other files produced by the Converter must be created in
     * the directory at the outputRootDir and the path implied by the package name.
     * @exception DeploymentException
     *   thrown if the Proxy is unable to load the CAP file.
     *   The exception message should be appropriately detailed and descriptive.
     */
    public boolean installCAPFile(AppletProperties[] appletPropertiesArray,
            String aCAPFileName,
            String outputDir)
            throws DeploymentException {
        int errorCount;
        try {
        // Run scriptgen to create the script.
        
        String scriptFileName = outputDir + FILE_SEPARATOR
                + aCAPFileName.substring(aCAPFileName.lastIndexOf(FILE_SEPARATOR))
                + SCRIPT_FILE_SUFFIX;
        
        tempFiles.add(scriptFileName);
        ArrayList cmd = (ArrayList)JAVA_COMMAND.clone();
        cmd.add(getClassPath());
        cmd.add(SCRIPTGEN_CLASSNAME);
        cmd.addAll(scriptgen_options);
        cmd.add("-o");
        cmd.add(scriptFileName);
        cmd.add(aCAPFileName);
        doExec("scriptgen", cmd, taskTimeout, 1);
        
        if (!poweredUp) {
            throw new DeploymentException(I18n.getString("invoke.installcapfile.before.powerup"));
        }
        boolean retVal = true;
        
        Apdu selectInstallerApdu = convertToApdu(new CommandAPDU(SELECT_INSTALLER_APDU));
        try {
            sendInternalAPDU(selectInstallerApdu, 0x9000);
            
            log.log(I18n.getString("send.create.applet.command"), Logger.DEBUG_TRACE);
            processScriptFile(scriptFileName);
        } catch (ScriptFailException e) {
            log.log(e.toString(), Logger.ERROR);
            return false;
        }
        
        for (int i = 0; i < appletPropertiesArray.length; i++) {
            try {
                log.log(I18n.getString("try.install.capfile.name", appletPropertiesArray[i].getClassName()),
                        Logger.DEBUG);
                sendInternalAPDU(selectInstallerApdu, 0x9000);
                
                byte[] aid = appletPropertiesArray[i].getAID().getBytes();
                byte[] command = new byte[aid.length + 8];
                command[0] = (byte)0x80;
                command[1] = (byte)0xB8;
                command[2] = (byte)0x00;
                command[3] = (byte)0x00;
                command[4] = (byte)(aid.length + 2);
                command[5] = (byte)aid.length;
                System.arraycopy(aid, 0, command, 6, aid.length);
                command[command.length - 2] = (byte)0x00;
                command[command.length - 1] = (byte)0x7F;
                Apdu cApdu = convertToApdu(new CommandAPDU(command));
                retVal = ((sendInternalAPDU(cApdu).sw() != SW_RETURN)
                          ? false : retVal);
            } catch (ScriptFailException e) {
                log.log(e.toString(), Logger.ERROR);
                retVal = false;
            }
        }
        sendInternalAPDU(selectInstallerApdu, 0x9000);
        return retVal;
        } catch (Exception e) {
            e.printStackTrace();
            e.printStackTrace(out);
            throw new DeploymentException("Unknown");
        }
    }
    
    
    /*
     * Sends a script file to the Proxy
     */
    public void processScriptFile(String scriptFileName)
    throws FrameworkException, DeploymentException, ScriptFailException {
        Apdu apdu;
        InputStream inputStream;
        ScriptCommand cmd;
        ScriptParser parser;
        int apduCount = 0;
        
        try {
            inputStream = new BufferedInputStream(
                    new FileInputStream(scriptFileName));
            parser = new ScriptParser(inputStream);
        } catch (IOException e) {
            log.log(e.toString(), Logger.ERROR);
            throw new FrameworkException(I18n.getString("open.script.file.exception", e));
        }
        try {
            while ((cmd = parser.getScriptCommand()) != null) {
                switch (cmd.getType()) {
                    case ScriptCommand.POWER_UP:
                        powerUp();
                        break;
                    case ScriptCommand.POWER_DOWN:
                        powerDown();
                        break;
                    case ScriptCommand.DELAY:
                        try {
                            Thread.sleep(((Integer)cmd.getData()).intValue());
                        } catch (InterruptedException e) {
                        }
                        break;
                    case ScriptCommand.ECHO:
                        log.log(SCRIPT_COMMENT + cmd.getData(), Logger.MESSAGE);
                        break;
                    case ScriptCommand.APDU:
                        sendInternalAPDU((Apdu)cmd.getData(),
                                         SW_RETURN);
                        break;
                    default:
                        throw new ParseException(I18n.getString("unknown.script.command"));
                }
            }
        } catch (ScriptFailException e) {
            throw e;
        } catch (Exception e1) {
            e1.printStackTrace();
            e1.printStackTrace(out);
            log.log(I18n.getString("install.failed.with.exception", e1), Logger.ERROR);
            throw new DeploymentException(e1.toString());
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
            }
        }
    }
    
    protected HashSet tempFiles = new HashSet();
    
    protected void startCref() throws FrameworkException {
        ArrayList command = new ArrayList();
        command.add(getRunCommand());
        command.addAll(cref_options);
        if (proxyStartCount > 1) {
            // on 2nd and subsequent starts, restore EEPROM
            // use the EEPROM save file from the previous execution
            command.add(JCRE_OPTION_RESTART);
            command.add(getSaveFileName(proxyStartCount - 1));
        }
        
        if (log.isTraceEnabled()) {
            command.add("-z");
        }
        
        command.add(JCRE_OPTION_SAVE);
        command.add(getSaveFileName(proxyStartCount));
        setCrefBasePort(command);
        tempFiles.add(workingDir + File.separator + getSaveFileName(proxyStartCount));
        log.log(I18n.getString("start.proxy.command", command), Logger.DEBUG_TRACE);
        String processName = REFIMPL_NAME + "(#" + proxyStartCount + ")";
        Throwable t = null;
        for (int i = 0; i < CREF_RESTART_ATTEMPTS; i++) {
            try {
                currentProcDescriptor = manager.start(processName,
                        (String[])command.toArray(new String[command.size()]),
                        EXEC_ENVIRONMENT, workingDir);
                this.crefRunning = true;
                return;
            } catch (Throwable th) {
                log.log(I18n.getString("cref.start.failed", th), Logger.DEBUG_TRACE);
                t = th;
                setCrefBasePort(command);
                try {
                    Thread.sleep(SLEEP_TIME);
                } catch (InterruptedException ex) {}
            }
        }
        throw new FrameworkException(I18n.getString("cref.start.failed", t));
    }
    
    public Descriptor getCurrentProcDescriptor() {
        return currentProcDescriptor;
    }
    
    public boolean isCrefRunning() {
        return crefRunning;
    }
    
    public CadClient getCad() throws FrameworkException {
        if (cad == null) {
            cad = createCadClient(DEFAULT_HOST_NAME, getPort());
        }
        return cad;
    }
    
    public void powerDownCad() throws TLP224Exception, IOException {
        log.log("powerDownCad()", Logger.DEBUG_TRACE);
        cad.powerDown();
    }
    /**
     * Start the Proxy.
     *
     * @exception FrameworkException
     *     if C-JCRE could not be created or socket could not be opened.
     */
    protected void startProxy() throws FrameworkException {
        proxyStartCount++; // increment the proxy start count
        log.log(I18n.getString("start.proxy", new Integer(proxyStartCount)),
                Logger.DEBUG_TRACE);
        Throwable t = null;
        for (int i = 0; i < CREF_RESTART_ATTEMPTS; i++) {
            if (!isCrefRunning()) {
                startCref();
                log.log(I18n.getString("refimpl.started.separate.process"), Logger.DEBUG_TRACE);
            }
            // create cad client
            try {
                cad = getCad();
            } catch (FrameworkException ex) {
                if (isCrefRunning()) {
                    stopProxy();
                }
                continue;
            }
            log.log("Attempt #" + i, Logger.DEBUG_TRACE);
            try {
                cad.powerUp();
                log.log(I18n.getString("proxy.has.powered.up"), Logger.DEBUG);
                poweredUp = true;
                return;
            } catch (Throwable e) {
                t = e;
                log.log(I18n.getString("cjcre.powerup.exception", e), Logger.DEBUG);
                if (e instanceof SocketException) {
                    crefRunning = false;
                    cad = null;
                }
            }
            try {
                currentProcDescriptor.checkProcess(SLEEP_TIME);
            } catch (CardProxyException ex) {
                t = ex;
                crefRunning = false;
                cad = null;
            }
        }
        throw new FrameworkException(I18n.getString("cref.start.failed", t));
    }
    
    
    /**
     * exec's a command.  Also calls writeShellScript to create a shell script.
     * @param pName is the name of the process manager
     * @param prog is the command
     * @param scriptFileName is the name of a shell script file which will be created
     * @param timeOut is the number of milliseconds to wait for
     * completion.  If equal to NOWAIT_TIMEOUT (zero), it means do not
     * wait for completion, return immediately.
     * @param maxErrors is the maximum number of errors allowable.
     */
    void doExec(String pName, ArrayList prog, long timeOut, int maxErrors)
    throws FrameworkException {
        log.log(I18n.getString("doexec.exec", prog), Logger.DEBUG_TRACE);
        reportProcess(manager.execute(pName, (String[])prog.toArray(new String [prog.size()]),
                EXEC_ENVIRONMENT));
    }
    
    /* Get the JCRE EEPROM save file name
     * @param n the execution number
     *
     */
    String getSaveFileName(int n) {
        return JCRE_STATE_FILENAME + n + JCRE_STATE_FILENAME_SUFFIX;
    }
    
    
    /* Stop the Proxy.
     *
     * @exception FrameworkException
     *     Thrown if stop failed.
     */
    protected void stopProxy() throws FrameworkException {
        log.log(I18n.getString("start.stop.proxy"), Logger.DEBUG_TRACE);
        
        if (getCurrentProcDescriptor() != null) {
            log.log(I18n.getString("start.terminate.proxy"), Logger.DEBUG_TRACE);
            reportProcess(manager.waitUntilDone(getCurrentProcDescriptor()));
            log.log(I18n.getString("terminate.proxy.done"), Logger.DEBUG_TRACE);
        } else {
            log.log(I18n.getString("proxy.already.terminated"), Logger.DEBUG_TRACE);
        }
        
        cad = null;
        if (contactlessService != null) {
            contactlessService.cad = null;
        }
        log.log(I18n.getString("proxy.stopped"), Logger.DEBUG_TRACE);
        closeStream(bin);
        closeStream(bos);
        this.crefRunning = false;
    }
    
    public static void closeStream(Object in) {
        if (in == null) {
            return;
        }
        try {
            if (in instanceof InputStream) {
                ((InputStream)in).close();
            } else if (in instanceof OutputStream) {
                ((OutputStream)in).close();
            }
        } catch (IOException e) {
        }
    }
    
    private void reportProcess(ProcessManager.Status status) {
        log.log(I18n.getString("process.end.with.status", new Object[] {status.name,
                new Integer(status.status), status.out, status.err}),
                        Logger.MESSAGE);
    }
    
    /* Clean up the working directory.
     *
     * @exception FrameworkException
     *     Thrown if reset failed.
     */
    private void cleanupWorkingDir() throws FrameworkException {
        log.log(I18n.getString("clean.work.dir"), Logger.DEBUG_TRACE);
        // clean up the working directory -- delete the save files
        File dir = new File(workingDir);
        String[] list = dir.list();
        if (list != null) {
            for (int i = 0; i < list.length; i++) {
                if (list[i].startsWith(JCRE_STATE_FILENAME)) {
                    try {
                        File f = new File(dir, list[i]);
                        if (f.exists() && f.delete()) {
                            log.log(I18n.getString("deleted.eeprom.save.file", list[i]), Logger.DEBUG);
                        }
                    } catch (Exception e) {
                        log.log(I18n.getString("unable.delete.eeprom.save.file", list[i], e), Logger.DEBUG);
                    }
                } else {
                    log.log(I18n.getString("do.not.delete.file", list[i]), Logger.DEBUG);
                }
            }
        }
        proxyStartCount = 0; // reset the proxy start count
    }
    
    public ResponseAPDU sendAPDU(CommandAPDU capdu) throws FrameworkException {
        // verify that the Proxy has been started
        if (!this.poweredUp) {
            throw new FrameworkException(I18n.getString("invoke.sendapdu.before.powerup"));
        }
        return sendCommandAPDU(capdu);
    }

    private Apdu convertToApdu(CommandAPDU capdu) {
        Apdu apdu = new Apdu();
        apdu.command = new byte[] {
            (byte)capdu.getCLA(), (byte)capdu.getINS(),
            (byte)capdu.getP1(), (byte)capdu.getP2()
        };
        apdu.Lc = capdu.getLc();
        apdu.setDataIn(capdu.getDataIn());
        apdu.Le = capdu.getLe();
        return apdu;
    }


    public ResponseAPDU sendCommandAPDU(CommandAPDU capdu) throws FrameworkException {
        return sendInternalAPDU(convertToApdu(capdu));
        /*
        log.log(I18n.getString("sending.apdu",  new Integer(++sendApduNum),
                apdu.toString()),
                Logger.DEBUG);
        
        addApduToScript(capdu.getBytes());
        try {
            cad.exchangeApdu(apdu);
        } catch (CadTransportException e) {
            e.printStackTrace();
            try {
                cad.powerDown(false);
            } catch (CadTransportException e1) {
            } catch (IOException e1) {
            }
            throw new FrameworkException(I18n.getString("cjcre.responseapdu.cad.powerdown",  e.getMessage()));
        } catch (IOException e) {
            log.log(e.toString(), Logger.ERROR);
            try {
                cad.powerDown(false);
            } catch (CadTransportException e1) {
            } catch (IOException e1) {
            }
            throw new FrameworkException(I18n.getString("cjcre.responseapdu.cad.powerdown",  e.getMessage()));
        }
        
        log.log(I18n.getString("receive.apdu", new Integer(++recvApduNum), apdu), Logger.MESSAGE);
        
        // convert the Apduio apdu to an opencard rapdu
        int le = apdu.getLe();
        byte[] respByte = new byte[le + 2];
        if (le > 0) {
            byte[] dataOut = apdu.getDataOut();
            if (dataOut != null) {
                System.arraycopy(dataOut, 0, respByte, 0, le);
            }
        }
        respByte[respByte.length - 2] = apdu.sw1sw2[0];
        respByte[respByte.length - 1] = apdu.sw1sw2[1];
        return new ResponseAPDU(respByte);
        */
    }
    
    public void sendInternalAPDU(Apdu capdu, int sw)
        throws FrameworkException {
        ResponseAPDU resp = sendInternalAPDU(capdu);
        if (resp.sw() != sw) {
            throw new ScriptFailException(I18n.getString("incorrect.sw.answer",
                    Integer.toHexString(sw),
                    Integer.toHexString(resp.sw())));
        }
    }
    private void addApduToScript(byte [] buffer) {
        if ((script_file != null) && (buffer != null)) {
            StringBuffer line = new StringBuffer();
            for (int i = 0; i < buffer.length; i++) {
                line.append(" 0x");
                line.append(Integer.toHexString(0xFF & buffer[i]));
            }
            line.append(";");
            script_file.println(line.toString());
        }
    }
    
    public ResponseAPDU sendInternalAPDU(Apdu apdu)
        throws FrameworkException {
        log.log(I18n.getString("sending.apdu",  new Integer(++sendApduNum),
                               apdu.toString()),
                Logger.DEBUG);
        addApduToScript(apdu.getCommandApduBytes());
        try {
            cad.exchangeApdu(apdu);
        } catch (TLP224Exception e) {
            e.printStackTrace();
            try {
                cad.powerDown();
            } catch (TLP224Exception e1) {
            } catch (IOException e1) {
            }
            throw new FrameworkException(I18n.getString("cjcre.responseapdu.cad.powerdown",  e.getMessage()));
        } catch (IOException e) {
            log.log(e.toString(), Logger.ERROR);
            try {
                cad.powerDown();
            } catch (TLP224Exception e1) {
            } catch (IOException e1) {
            }
            throw new FrameworkException(I18n.getString("cjcre.responseapdu.cad.powerdown",  e.getMessage()));
        }
        
        log.log(I18n.getString("receive.apdu", new Integer(++recvApduNum), apdu), Logger.MESSAGE);
        
        // convert the Apduio apdu to an ResponseAPDU
        int le = apdu.getLe();
        byte[] respByte = new byte[le + 2];
        if (le > 0) {
            byte[] dataOut = apdu.getDataOut();
            if (dataOut != null) {
                System.arraycopy(dataOut, 0, respByte, 0, le);
            }
        }
        respByte[respByte.length - 2] = apdu.sw1sw2[0];
        respByte[respByte.length - 1] = apdu.sw1sw2[1];
        return new ResponseAPDU(respByte);
    }
    
    
    /**
     * Creates a cad client.
     * @exception FrameworkException
     *   if the socket open failed.
     */
    CadClient createCadClient(String hostName, int port)
        throws FrameworkException {
        CadClient cad;
        
        log.log(I18n.getString("createcadclient.open.connection", hostName, new Integer(port)),
                Logger.DEBUG);
        
        try {
            closeSocket(sock);
            
            log.log(I18n.getString("createcadclient.try.create.cadclient"), Logger.DEBUG_TRACE);
            
            sock = openSocket(hostName, port);
            log.log(I18n.getString("createcadclient.opened.socket", sock), Logger.DEBUG_TRACE);
            bin = new BufferedInputStream(sock.getInputStream());
            bos = new BufferedOutputStream(sock.getOutputStream());
            cad = new CadClient(bin, bos);
        } catch (java.io.IOException e) {
            throw new FrameworkException(I18n.getString("socket.connection.failed.exception",
                    new Integer(port), e.getMessage()));
        }
        
        log.log(I18n.getString("createcadclient.created"), Logger.DEBUG_TRACE);
        return cad;
    }
    
    /**
     * Opens a socket.
     *
     * @param hostname the name of the host computer
     * @param port the port number
     *
     * @exception FrameworkException
     *   if the socket open failed.
     */
    private Socket openSocket(String hostName, int port)
    throws FrameworkException {
        Socket socket = new Socket();
        SocketAddress crefAddr = new InetSocketAddress(DEFAULT_HOST_NAME, port);
        Throwable t = null;
        SocketAddress cadLocalAddr = new InetSocketAddress(DEFAULT_HOST_NAME, getCadLocalPort());
        for (int i = 0; i < CREF_CONNECT_ATTEMPTS; i++) {
            try {
                socket = new Socket();
                try {
                    log.log("Trying to bind socket to " + cadLocalAddr, Logger.DEBUG_TRACE);
                    socket.bind(cadLocalAddr);
                } catch (BindException e) {
                    log.log("Failed to bind socket to " + cadLocalAddr
                            + " BindException: " + e.getMessage(),Logger.DEBUG_TRACE);
                    cadLocalAddr = new InetSocketAddress(DEFAULT_HOST_NAME, getCadLocalPort());
                    t = e;
                    continue;
                }
                log.log("Trying to connect socket to " + crefAddr, Logger.DEBUG_TRACE);
                socket.connect(crefAddr);
                socket.setTcpNoDelay(true);
                log.log(I18n.getString("opensocket.open.socket.on.port", new Integer(port)),
                        Logger.DEBUG);
                return socket;
            } catch (UnknownHostException e) {
                log.log(I18n.getString("opensocket.unknown.host.name", hostName),
                        Logger.DEBUG_TRACE);
                t = e;
            } catch (ConnectException ex) {
                log.log("Failed to connect socket to " + crefAddr
                        + " ConnectException: " + ex.getMessage(),
                        Logger.DEBUG_TRACE);
                t = ex;
            } catch (IOException e) {
                log.log(I18n.getString("open.socket.exception", e), Logger.DEBUG_TRACE);
                t = e;
            }
            try {
                getCurrentProcDescriptor().checkProcess(SLEEP_TIME);
            } catch (CardProxyException ex) {
                t = ex;
                crefRunning = false;
                break;
            }
        }
        throw new FrameworkException(I18n.getString("can.not.open.socket.on.port",
                new Integer(port), t));
    }
    
    
    /**
     * Closes the socket.
     */
    void closeSocket(Socket sock) {
        if (sock != null && !sock.isClosed()) {
            String socketName = sock.toString();
            try {
                sock.shutdownOutput();
                sock.close();
            } catch (Exception e) {
                log.log(I18n.getString("closesocket.got.exception", e), Logger.DEBUG);
            }
            sock = null;
            log.log(I18n.getString("socket.closed")+ " " + socketName, Logger.DEBUG_TRACE);
        } else {
            log.log(I18n.getString("socket.already.closed"), Logger.DEBUG_TRACE);
        }
        
    }
    
    
    
    /* Retrieve the Proxy properties.
     * Can return null.
     */
    public Hashtable getProperties() {
        Hashtable ht = new Hashtable();
        ht.put(PROXY_NAME_PROPERTY, PROXY_NAME);
        ht.put(PROXY_VERSION_PROPERTY, PROXY_VERSION);
        ht.put(API_VERSION_PROPERTY, API_VERSION);
        ht.put(MAX_STACK_PROPERTY, MAX_STACK);
        ht.put(MAX_TRANSIENT_RAM_PROPERTY, MAX_TRANSIENT_RAM);
        ht.put(MAX_NVRAM_PROPERTY, MAX_NVRAM);
        ht.put(INT_SUPPORT_PROPERTY, INT_SUPPORT);
        return ht;
    }
    
    protected void banner() {
        log.log(I18n.getString("release.info", new Object[] {RELEASE_VERSION,
                MAJOR_VERSION,
                MINOR_VERSION,
                BUILD_VERSION,
                BUILD_DATE}),
                        Logger.DEBUG);
                log.log(I18n.getString("java.info", JAVA_VERSION), Logger.DEBUG);
                log.log(I18n.getString("cjckcardservice.release.info",
                        com.sun.javacard.cjck.userinterface.CJCKCardService.RELEASE_VERSION,
                        com.sun.javacard.cjck.userinterface.CJCKCardService.BUILD_VERSION),
                        Logger.DEBUG);
    }
    
    public String getClassPath() {
        String ri = serviceProperties.getProperty("RI");
        log.log(I18n.getString("cjcrecardservice.classpath", ri), Logger.DEBUG);
        return classpath + ((ri == null) ? "" : File.pathSeparator + ri +
                File.separator + "lib" + File.separator +
                "scriptgen.jar");
    }
    
    public String getRunCommand() throws FrameworkException {
        String command = null;
        if (serviceProperties != null) {
            command = serviceProperties.getProperty(OPT_RI_RUN);
            if (command == null) {
                command = serviceProperties.getProperty(OPT_RI);
                if (command != null) {
                    command = command + File.separator + "bin"
                            + File.separator;
                    command = getExecutableName(command);
                }
            }
        }
        if (command == null) {
            throw new IllegalArgumentException(I18n.getString("undefined.cref"));
        }
        checkExecutable(command);
        return command;
    }
    
    private void checkExecutable(String name) {
        if ((new File(name)).exists()) {
            return;
        }
        if ((File.separatorChar == '\\') // Win32
        && ((new File(name + ".exe")).exists()
        ||(new File(name + ".bat")).exists())) {
            return;
        }
        throw new IllegalArgumentException(I18n.getString("name.not.exist", name));
    }
    
    private PrintWriter script_file;
    /**
     * initializes the CardService instance.
     * @param args specifies initialization options.
     * These options are defined in the jte file with prefix -s
     * @param out specifies PrintWriter for output.
     */
    public void init(String[] args, PrintWriter out, PrintWriter ref) {
        this.args = args;
        this.out = out;
        this.ref = ref;
        log = new Logger(0, out, ref);
        if (args == null) {
            return;
        }
        for (int i = 0; i < args.length; i++) {
            if (args[i].startsWith(OPT_COMVERTER_PREFIX)) {
                converter_options.add(args[i].substring(OPT_COMVERTER_PREFIX.length()));
            } else if (args[i].startsWith(OPT_CREF_PREFIX)) {
                cref_options.add(args[i].substring(OPT_CREF_PREFIX.length()));
            } else if (args[i].startsWith(OPT_APDUTOOL_PREFIX)) {
                apdutool_options.add(args[i].substring(OPT_APDUTOOL_PREFIX.length()));
            } else if (args[i].startsWith(OPT_SCRIPTGEN_PREFIX)) {
                scriptgen_options.add(args[i].substring(OPT_SCRIPTGEN_PREFIX.length()));
            } else if (args[i].equals("-classpath") && (i < (args.length - 1))) {
                classpath = args[++i];
            } else if (args[i].equals("-script_file") && (i < (args.length - 1))) {
                try {
                    script_file =
                            new PrintWriter(
                            new BufferedOutputStream(
                            new FileOutputStream(args[++i])));
                } catch (IOException e) {
                    e.printStackTrace();
                }
            } else if (args[i].startsWith("-") && (i < (args.length - 1))) {
                serviceProperties.setProperty(((args[i].length() > 1) ?
                    args[i].substring(1) : ""),
                        args[++i]);
            }
        }
        
        /*String protocolName = (String)serviceProperties.get(OPT_TRANSPORT_PROTOCOL);
        if (protocolName == null || protocolName.toUpperCase().equals("TDUAL")) {
            transportProtocol = CadDevice.PROTOCOL_T1;
            transportProtocolName = T1_PROTOCOL_OPTION;
        } else if (protocolName.toUpperCase().equals("T0")) {
            transportProtocol = CadDevice.PROTOCOL_T0;
            transportProtocolName = T0_PROTOCOL_OPTION;
        } else {
            transportProtocol = CadDevice.PROTOCOL_T1;
            transportProtocolName = T1_PROTOCOL_OPTION;
        }**/
        String s = (String)serviceProperties.get(OPT_DEBUG_LEVEL);
        int level = DEFAULT_DEBUG_LEVEL;
        try {
            if (s != null) {
                level = Integer.parseInt(s);
            }
        } catch (Exception e) {
            log.log(e.toString(), Logger.MESSAGE);
        }
        log.setDebugLevel(level);
        manager = new ProcessManager(log);
        banner();
    }
    
    
    /**
     * deletes list of the applets.
     * @param applets specifies list of the applets for deletion
     * @return true if the all applets are deleted successfully, false otherwise.
     */
    public boolean deleteAppletInstances(AppletProperties[] applets)
    throws DeploymentException {
        log.log(I18n.getString("delete.applet.ins.log"), Logger.DEBUG_TRACE);
        for (int i = 0; i < applets.length; i++) {
            log.log(I18n.getString("deleting.applet", applets[i].getClassName()), Logger.DEBUG);
        }
        if (applets.length > 7) {
            log.log(I18n.getString("allowed.applet.deletion.per.one.request"), Logger.MESSAGE);
            return false;
        }
        int size = 6;
        for (int i = 0; i < applets.length; i++) {
            size += 1 + applets[i].getAID().getBytes().length;
        }
        byte[] data = new byte[size];//0x80, 0xc4, 0x0#, 0x00,
        data[0] = (byte)0x80;
        data[1] = (byte)0xC4;
        data[2] = (byte)applets.length;
        data[3] = (byte)0x00;
        data[4] = (byte)(size - 6);
        int pos = 5;
        for (int aidNumber = 0; aidNumber < applets.length; aidNumber++) {
            byte[] aid = applets[aidNumber].getAID().getBytes();
            data[pos++] = (byte)aid.length;
            for (int i = 0; i < aid.length; i++) {
                data[pos++] = aid[i];
            }
        }
        data[pos] = (byte)0x7F;
        selectInstaller();
        try {
            ResponseAPDU answer = sendAPDU(new CommandAPDU(data));
            int sw = answer.sw();
            return (sw == 0x9000);
        } catch (FrameworkException ex) {
            throw new DeploymentException(ex.getMessage());
        }
    }
    
    /**
     * deletes given applet/library package.
     * @param packageID specifies AID of the given package.
     * @param packageName specifies full qualified name of the given package.
     * @return true if the package are deleted successfully, false otherwise.
     */
    public boolean deletePackage(AppletID packageID, String packageName)
    throws DeploymentException {
        log.log(I18n.getString("package.deletion.log"), Logger.DEBUG_TRACE);
        
        return deletePackageInternal(packageID, packageName, false);
    }
    
    private boolean deletePackageInternal(AppletID packageID, String packageName,
            boolean isInstance)
            throws DeploymentException {
        try {
            log.log(I18n.getString("deletion.package.name", packageName), Logger.DEBUG);
            byte aid_bytes[] = packageID.getBytes();
            byte[] data = new byte[7 + aid_bytes.length];
            data[0] = (byte)0x80;
            data[1] = (byte)(isInstance ? 0xC2 : 0xC0);
            data[2] = (byte)0x00;
            data[3] = (byte)0x00;
            data[4] = (byte)(aid_bytes.length + 1);
            data[5] = (byte)aid_bytes.length;
            System.arraycopy(aid_bytes, 0, data, 6, aid_bytes.length);
            data[data.length - 1] = (byte)0x7F;
            selectInstaller();
            ResponseAPDU answer = sendAPDU(new CommandAPDU(data));
            int sw = answer.sw();
            if (sw == 0x6F00) {
                throw new DeploymentException(I18n.getString("unknown.answer"));
            }
            return (sw == 0x9000);
        } catch (FrameworkException ex) {
            throw new DeploymentException(ex.getMessage());
        }
    }
    
    
    protected void selectInstaller() throws DeploymentException {
        try {
            ResponseAPDU answer = sendAPDU(new CommandAPDU(SELECT_INSTALLER_APDU));
            if (answer.sw() != 0x9000) {
                throw new DeploymentException(I18n.getString("can.not.select.installer"));
            }
        } catch (FrameworkException ex) {
            throw new DeploymentException(ex.getMessage());
        }
    }
    
    public boolean deletePackageAndInstances(AppletID packageID,
            String packageName)
            throws DeploymentException {
        log.log(I18n.getString("package.ins.deletion.log"), Logger.DEBUG_TRACE);
        
        return deletePackageInternal(packageID, packageName, true);
    }
    
    /**
     * This method gets free local port for CadClient
     * returned value will never be equal to port for C-JCRE
     * because it should be divisible by 4 without remainder
     * and C-JCRE base port should be divisible by 4 with remainder 1.
     */
    protected synchronized int getCadLocalPort() {
        int freePort = 0;
        ServerSocket cadTmpSocket;
        for (int i = 0; i < MAX_ATTEMP; i++) {
            try {
                cadTmpSocket = new ServerSocket(0);
                freePort = cadTmpSocket.getLocalPort();
                if (freePort % 4 != 0) {
                    try {
                        cadTmpSocket.close();
                        cadTmpSocket = null;
                    } catch (IOException ex) {}
                    continue;
                }
                try {
                    cadTmpSocket.close();
                } catch (IOException ex){}
                log.log("getCadLocalPort() returned port #" + freePort, Logger.DEBUG_TRACE);
                return freePort;
            } catch (IOException e1) {
                continue;
            }
        }
        return (DEFAULT_PORT - 1);
    }
    
    /**
     * This method sets base port for C-JCRE startup command
     * Note: We need to ensure that CadClient local port, base port of C-JCRE
     * and contactless port of C-JCRE are different.
     * So we require that CadClient local port should be divisible by 4,
     * base port should be divisible by 4 with remainder 1 and contacless port
     * is base port + 1
     *
     * @param crefCommand C-JCRE starup command
     */
    protected synchronized void setCrefBasePort(ArrayList crefCommand) throws FrameworkException {
        int nextPort;
        ServerSocket baseSocket;
        if (crefCommand.contains("-p")) {
            crefCommand.remove("-p");
            crefCommand.remove(Integer.toString(this.basePort));
        }
        crefCommand.add("-p");
        for (int i = 0; i < MAX_ATTEMP; i++) {
            try {
                baseSocket = new ServerSocket(0);
            } catch (IOException ex) {
                continue;
            }
            basePort = baseSocket.getLocalPort();
            if (basePort % 4 != 1) {
                try {
                    baseSocket.close();
                } catch (IOException ex){}
                continue;
            }
            //If this is not T0 protocol implementation we need one more free port for cref
            if (transportProtocolName != T0_PROTOCOL_OPTION) {
                
                ServerSocket nextSocket;
                try {
                    nextSocket = new ServerSocket(basePort + 1);
                    log.log("Base port: " + basePort + ", next port: "
                            + nextSocket.getLocalPort(), Logger.DEBUG_TRACE);
                } catch (IOException ex) {
                    log.log(I18n.getString("io.exception", ex.getMessage()), Logger.DEBUG_TRACE);
                    try {
                        baseSocket.close();
                    } catch (IOException iex){}
                    continue;
                }
                
                try {
                    nextSocket.close();
                } catch (IOException ex) {
                    log.log(I18n.getString("socket.next.close.failed", nextSocket), Logger.DEBUG_TRACE);
                }
            }
            try {
                baseSocket.close();
            } catch (IOException ex) {
                log.log(I18n.getString("socket.base.close.failed", baseSocket), Logger.DEBUG_TRACE);
            }
            try {
                Thread.sleep(SLEEP_TIME);
            } catch (InterruptedException ex) {}
            log.log("CREF will be started on port #" + basePort, Logger.DEBUG_TRACE);
            crefCommand.add(Integer.toString(this.basePort));
            return;
        }
        throw new FrameworkException(I18n.getString("no.free.ports"));
    }
    
    /**
     * This method returns name of executable for CJCRE
     */
    protected String getExecutableName(String dir) throws FrameworkException {
        String[] guess = new String[] {
            dir + "cref_" + System.getProperty("os.arch") + "_" + transportProtocolName,
                    dir + "cref_" + transportProtocolName,
                    dir + "cref_" + transportProtocolName + ".exe",
                    dir + "cref",
                    dir + "cref.exe"
        };
        for (int i = 0; i < guess.length; i++) {
            if (new File(guess[i]).exists()) {
                return guess[i];
            }
        }
        throw new FrameworkException("Cref implementation does not exist.");
    }
    
    /**
     *  Get a CommunicationService instance for specified i/o interface. For backward compatibility
     *  this method returns <code>this</code> for contacted i/o interface and throw an exception for
     *  contactless i/o interface
     *
     *  @param  iface   The I/O interface, that the requested CommunicationService will work over.
     *
     *  @return     this    if <code>iface == CardService.CONTACTED_INTERFACE</code>
     *
     *  @exception  FrameworkException  if <code>iface != CardService.CONTACTED_INTERFACE</code>
     */
    public CommunicationService getCommunicationService(int iface) throws FrameworkException {
        if (iface == CardService.CONTACTED_INTERFACE) {
            if (contactlessService == null) {
                isMainCommunicationService = true;
            }
            return this;
        }
        if (iface == CardService.CONTACTLESS_INTERFACE) {
            if (contactlessService == null) {
                ContactlessService clessService = new ContactlessService();
                clessService.isMainCommunicationService = !this.isMainCommunicationService;
                clessService.setContactedService(this);
                clessService.init(args, out, ref);
                contactlessService = clessService;
            }
            return contactlessService;
        }
        throw new FrameworkException(I18n.getString("unsupported.io.interface", new Integer(iface)));
    }
    
    int getPort() {
        return this.basePort;
    }

    public void setCardTerminal(StatefulCardTerminal terminal) {
        // do nothing
    }
    
}
