/*
 * %W% %E%
 * @Copyright
 *
 */

package com.sun.javacard.referenceimpl;

import com.sun.javacard.cjck.userinterface.BiometricDataProvider;
import com.sun.javacard.cjck.userinterface.NoSuchBioTypeException;

/**
 *
 * @author Dmitri Trounine
 */
public class CJCREBiometricDataProvider implements BiometricDataProvider {
    
    private static final byte[] SunRID = {(byte) 0xa0, (byte) 0x00, (byte) 0x00,
            (byte) 0x00, (byte) 0x62};
            
    private static final byte[] REFERENCE_PASSWORD = { (byte) 1, (byte) 2,
            (byte) 3, (byte) 4, (byte) 5, (byte) 6, (byte) 7, (byte) 8 };
            
    private static final byte[] POSITIVE_PASSWORD = REFERENCE_PASSWORD;
//    private static final byte[] POSITIVE_PASSWORD = { (byte) 1, (byte) 2,
//            (byte) 3, (byte) 4, (byte) 5, (byte) 0, (byte) 7, (byte) 8 };
          
    private static final byte[] NEGATIVE_PASSWORD = { (byte) 8, (byte) 7,
            (byte) 6, (byte) 5, (byte) 4, (byte) 3, (byte) 2, (byte) 1 };
            
    private static final byte[] WRONG_PASSWORD = { (byte) 0, (byte) 0 };
    
    /** Creates a new instance of CJCREBiometricDataProvider */
    public CJCREBiometricDataProvider() {
    }
    
    /**
     * Returns the RIDs of all on-card biometric providers (NOT including the
     * RID of the default biometric provider). Each RID is a byte array,
     * the returned array is the array of RIDs. If there is only default
     * biometric provider, will return zero-length array.
     */
    public byte[][] getAllBiometricProvidersRIDs() {
        return new byte[][] { SunRID };
    }
    
    /**
     * Returns the RID of the default biometric provider
     */
    public byte[] getDefaultBiometricProviderRID() {
        return SunRID;
    }
    
    /**
     * @return true if the given biometric provider supports the given bio type.
     */
    public boolean isSupportedBioType(byte bioType, byte[] RID) {
        if ((RID == null || isSunRID(RID)) && bioType == BioBuilder.PASSWORD) {
            return true;
        }
        return false;
    }
    
    /**
     *  Gets the possible values of the initialization parameter for the given
     *  bio type and biometric provider. If some values are equivalent for 
     *  the OwnerBioTemplate implementation, only one of them must be present
     *  in the returned byte array. The value 0 (BioBuilder.DEFAULT_INITPARAM)
     *  should not be present.
     */
    public byte[] getAllInitParamValues(byte bioType, byte[] RID)
            throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return new byte[0];
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }
    
    /**
     * Gets all bio types supported by the given biometric provider
     */
    public byte[] getSupportedBioTypes(byte[] RID) {
        if (RID == null || isSunRID(RID)) {
            return new byte[] { BioBuilder.PASSWORD };
        }
        return null;
    }
    
    /**
     *  Returns the byte array containing the data of a reference bio template
     *  of the given type for the given biometric provider and the given
     *  initialization parameter for the OwnerBioTemplate instance.
     *  Two successive calls to this method will return the same data.
     *
     *  @param  bioType requested bio type
     *  @paran  RID     the RID of the provider of OwnerBioTemplate
     *                  implementation.
     *  @param  initParam   the provider specific initialization information for
     *                      the OwnerBioTemplate instance. 
     *
     *  @return A byte array of positive length.
     *
     *  @exception  NoSuchBioTypeException  if the requested bio type is not
     *      supported by the given on-card biometric provider.
     */
    public byte[] getReferenceBioTemplate(byte bioType, byte[] RID,
            byte initParam) throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return REFERENCE_PASSWORD;
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }
    
    /**
     * Returns the byte array containing the data of a candidate bio template
     * of the given bio type for the given biometric provider, which
     * beeing matched with the reference bio template returned by the
     * <code>getReferenceBioTemplate(int, byte[])</code> method with the same
     * bioType and RID parameters, will be accepted.
     * Two successive calls to this method will return the same data.
     *
     * @param  bioType requested bio type
     * @paran  RID     the RID of the provider of OwnerBioTemplate
     *                  implementation.
     * @param  initParam   the provider specific initialization information for
     *                      the OwnerBioTemplate instance. 
     *
     * @return A byte array of positive length.
     *
     * @exception NoSuchBioTypeException    if the requested bio type is not
     *      supported by the given biometric provider.
     */
    public byte[] getPositiveCandidate(byte bioType, byte[] RID,
            byte initParam) throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return POSITIVE_PASSWORD;
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }

    /**
     * Returns the byte array containing the data of a candidate bio template
     * of the given bio type for the given biometric provider, which
     * beeing matched with the reference bio template returned by the
     * <code>getReferenceBioTemplate(int, byte[])</code> method with the same
     * bioType and RID parameters, will be rejected.
     * Two successive calls to this method will return the same data.
     *
     * @param  bioType requested bio type
     * @paran  RID     the RID of the provider of OwnerBioTemplate
     *                  implementation.
     * @param  initParam   the provider specific initialization information for
     *                      the OwnerBioTemplate instance. 
     *
     * @return A byte array of positive length.
     *
     * @exception NoSuchBioTypeException    if the requested bio type is not
     *      supported by the given biometric provider.
     */
    public byte[] getNegativeCandidate(byte bioType, byte[] RID,
            byte initParam) throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return NEGATIVE_PASSWORD;
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }

    /**
     * Returns the byte array containing the data which is not a correct 
     * reference template data of the given bioType in the format supported by the
     * given biometric provider.
     *
     * @param  bioType requested bio type
     * @paran  RID     the RID of the provider of OwnerBioTemplate
     *                  implementation.
     * @param  initParam   the provider specific initialization information for
     *                      the OwnerBioTemplate instance. 
     *
     * @return A byte array of positive length or null if no such data exists.
     *
     * @exception NoSuchBioTypeException    if the requested bio type is not
     *      supported by the given biometric provider.
     */
    public byte[] getInvalidReferenceData(byte bioType, byte[] RID,
            byte initParam) throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return WRONG_PASSWORD;
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }
    
    /**
     * Returns the byte array containing the data which is not a correct 
     * candidate template data of the given bioType in the format supported by the
     * given biometric provider.
     *
     * @param  bioType requested bio type
     * @paran  RID     the RID of the provider of OwnerBioTemplate
     *                  implementation.
     * @param  initParam   the provider specific initialization information for
     *                      the OwnerBioTemplate instance. 
     *
     * @return A byte array of positive length or null if no such data exists.
     *
     * @exception NoSuchBioTypeException    if the requested bio type is not
     *      supported by the given biometric provider.
     */
    public byte[] getInvalidCandidateData(byte bioType, byte[] RID,
            byte initParam) throws NoSuchBioTypeException {
        if (RID == null || isSunRID(RID)) {
            switch (bioType) {
            case BioBuilder.PASSWORD :
                return new byte[0];
            default :
                throw new NoSuchBioTypeException("Unsopported bio type :"
                        + bioType);
            }
        }
        return null;
    }
    private boolean isSunRID(byte[] RID) {
        if (RID.length != SunRID.length) {
            return false;
        }
        for (int i = 0; i < RID.length; i++) {
            if (RID[i] != SunRID[i]) {
                return false;
            }
        }
        return true;
    }
}
