/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */


package com.sun.javacard.cjck.userinterface;

import java.util.Hashtable;
import java.lang.System;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.I18n;


/**
 * AppletProperties is an applet information manager.
 * This class is used by the CJCKCardService interface.
 *
 * @version Version:%I% Date:%G%
 *
 * @author Eric Nellen (eric.nellen@eng.sun.com)
 */
public class AppletProperties {
    /////////////////////////////////////////////////////////////////////////////
    // Named constants
    /////////////////////////////////////////////////////////////////////////////

    /**
     * The major version of the AppletProperties class, currently 2.
     */
    public final static int MAJOR_VERSION = 2;

    /**
     * The minor version of the AppletProperties class, currently 2.
     */
    public final static int MINOR_VERSION = 2;

    /**
     * The minimum number of bytes in an AID, currently 5.
     */
    public final static int MIN_AID_LENGTH = 5;

    /**
     * The maximum number of bytes in an AID, currently 16 decimal.
     */
    public final static int MAX_AID_LENGTH = 16;

    /**
     * The maximum number of installation applet data bytes in an
     * applet, currently 32 decimal.
     */
    public final static int MAX_PARMS_LENGTH = 32;

    /**
     * The maximum number of bytes of code in an applet, currently 5120 decimal.
     */
    public final static int MAX_CODE_BYTES = 5120;

    /**
     * The maximum number of bytes of data in an applet, currently 2048 decimal.
     */
    public final static int MAX_DATA_BYTES = 2048;

    /**
     * The maximum number of bytes of stack space used by an applet,
     * currently 128 decimal.
     */
    public final static int MAX_STACK_BYTES = 128;

    /**
     * The maximum number of bytes of RAM used by an applet, currently 128 decimal.
     */
    public final static int MAX_RAM_BYTES = 128;

    public static final byte SECURITY_DOMAIN_PRIVILEDGE = (byte)0x80;
    public static final byte DAP_VERIFICATION_PRIVILEDGE = (byte)0xC0;
    public static final byte DELEGATED_MANAGEMENT_PRIVILEDGE = (byte)0xA0;
    public static final byte CARD_LOCK_PRIVILEDGE = (byte)0x10;
    public static final byte CARD_TERMINATE_PRIVILEDGE = (byte)0x08;
    public static final byte DEFAULT_SELECTED_PRIVILEDGE = (byte)0x04;
    public static final byte CVM_MANAGEMENT_PRIVILEDGE = (byte)0x02;
    public static final byte MANDATED_DAP_VERIFICATION_PRIVILEDGE = (byte)0xC1;


    /////////////////////////////////////////////////////////////////////////////
    // Instance data
    /////////////////////////////////////////////////////////////////////////////

    String packageName;
    AppletID packageAID;
    String className;
    AppletID AID;
    byte[] parmArray;
    int codeBytes;
    int dataBytes;
    int maxStack;
    int maxRam;
    Hashtable ht;
    private byte priviledges;
    private int volatileSpace;
    private int nonVolatileSpace;
    private byte[] gpParameters;

    /**
     * This constructor requires a valid applet package and class name.
     * <br>The AID and parms default to null; they must be set explicitly
     * using {@link #setAID(AppletID)}
     * and {@link #setParms(byte[])}.
     * <br>The Package Name and Package ID are derived from the class name.
     * <br>All other properties are optional and default to -1.
     * The other properties may be set using the
     * <code>set*()</code> methods of this class.
     * <br>Example of a packageAndClassName:
     * <br><pre><code>"com.sun.javacard.HelloWorld.HelloWorld"</code></pre>
     * @param packageName the name of the Applet's package.
     * @param className the name of the Applet's class.
     * @param packageAID the AppletID of the Applet's package.
     * @see #setAID(AppletID)
     * @see #setParms(byte[])
     */
    public AppletProperties(String packageName, AppletID packageAID, String className) 
        throws IllegalArgumentException {

            if ((packageAID != null)
                && ((packageAID.getBytes().length > MAX_AID_LENGTH)
                    || (packageAID.getBytes().length < MIN_AID_LENGTH))) {
                throw new IllegalArgumentException(I18n.getString("aid.out.range"));
	    }
	    this.packageName = packageName;
	    this.className = className;
	    this.packageAID = packageAID;
	    AID = null;
	    parmArray = null;
	    codeBytes = -1;
	    dataBytes = -1;
	    maxStack = -1;
	    maxRam = -1;
	    ht = new Hashtable();
    }

    /**
     * Sets the applet's AID attribute using an AppletID.
     * @param AID the Applet ID
     * @exception IllegalArgumentException
     * if the length of AID is less than {@link #MIN_AID_LENGTH} or
     * greater than {@link #MAX_AID_LENGTH}
     */
    public void setAID(AppletID AID) throws IllegalArgumentException {
        if (AID.getBytes().length > MAX_AID_LENGTH ||
            AID.getBytes().length < MIN_AID_LENGTH) {
            throw new IllegalArgumentException(I18n.getString("aid.out.range"));
        }
        this.AID = AID;
    }
    public void setPackageAID(AppletID packAID) throws IllegalArgumentException {
        if (packAID.getBytes().length > MAX_AID_LENGTH ||
            packAID.getBytes().length < MIN_AID_LENGTH) {
            throw new IllegalArgumentException(I18n.getString("aid.out.range"));
        }
	this.packageAID = packAID;
    }
	
			
    /**
     * Sets the installation applet data (parms) using a byte array.
     * @param parms the parameter bytes associated with this application.
     * @exception IllegalArgumentException
     * parms array is longer than {@link #MAX_PARMS_LENGTH}
     */
    public void setParms(byte [] parms) throws IllegalArgumentException {
        // verify the params
        if (parms.length > MAX_PARMS_LENGTH) {
            throw new IllegalArgumentException(I18n.getString("parameter.too.long"));
        }
        if (parms.length > 0) {
            this.parmArray = new byte[parms.length];
            java.lang.System.arraycopy(parms, 0, parmArray, 0, parms.length);
        } else {
            parmArray = null;
        }
    }

    /**
     * Sets the code bytes attribute.
     * @param n the number of bytes of applet byte code.
     * @exception IllegalArgumentException
     * value of n is not in range 0 &lt; value &lt;= {@link #MAX_CODE_BYTES}
     */
    public void setCodeBytes(int n) throws IllegalArgumentException {
        if (n < 0 || n > MAX_CODE_BYTES) {
            throw new IllegalArgumentException(I18n.getString("codebyte.out.range"));
        } else {
            codeBytes = n;
        }
    }

    /**
     * Sets the data bytes attribute.
     * @param n the number of bytes of applet data.
     * @exception IllegalArgumentException
     * value of n is not in range 0 &lt; value &lt;= {@link #MAX_DATA_BYTES}
     */
    public void setDataBytes(int n) throws IllegalArgumentException {
        if (n < 0 || n > MAX_DATA_BYTES) {
            throw new IllegalArgumentException(I18n.getString("databyte.out.range"));
        } else {
            dataBytes = n;
        }
    }

    /**
     * Sets the max stack bytes attribute.
     * @param n the maximum number of bytes of stack space required by this applet.
     * @exception IllegalArgumentException
     * value of n is not in range 0 &lt; value &lt;= {@link #MAX_STACK_BYTES}
     */
    public void setMaxStack(int n) throws IllegalArgumentException {
        if (n < 0 || n > MAX_STACK_BYTES) {
            throw new IllegalArgumentException(I18n.getString("stackvalue.out.range"));
        } else {
            maxStack = n;
        }
    }

    /**
     * Sets the max RAM bytes attribute.
     * @param n number of bytes of RAM required by this applet.
     * @exception IllegalArgumentException
     * value of n is not in range 0 &lt; value &lt;= {@link #MAX_RAM_BYTES}
     */
    public void setMaxRam(int n) throws IllegalArgumentException {
        if (n < 0 || n > MAX_RAM_BYTES) {
            throw new IllegalArgumentException(I18n.getString("rambyte.out.range"));
        } else {
            maxRam = n;
        }
    }

    /**
     * Sets the value of an arbitrary applet property.
     * A Card Service which implements this interface is not required to
     * do anything with the applet properties.
     * @param propertyName the name of the property, for example, "version".
     * @param propertyValue the value of the property, for example, "1.0".
     */
    public void setProperty(String propertyName, String propertyValue) {
        ht.put(propertyName, propertyValue);
    }

    /**
     * Gets the applet's package name.
     * @return the package name.
     */
    public String getPackageName() {
        return packageName;
    }

    /**
     * Gets the package's AID.
     * @return the package AID.
     */
    public AppletID getPackageAID() {
        return packageAID;
    }

    /**
     * Gets the applet's class name.
     * @return the applet's class name.
     */
    public String getClassName() {
        return className;
    }

    /**
     * Gets the applet's AID.
     * @return the applet's AID.
     */
    public AppletID getAID() {
        return AID;
    }

    /**
     * Gets the installation applet data (parms) as a byte array. These applet data
     * should be used in the installation parameters as defined in the
     * Java Card 2.2 Runtime Environment Specification, Chapter
     * 11.2.1. The method returns null if there are no applet data for
     * the applet.
     * @return byte array containing the applet data.
     */
    public byte [] getParms() {
        return parmArray;
    }


    /**
     * Gets the parameters attribute as a String.
     * @return the string representation of the parameter attribute.
     */
    public String getParmString() {
        return new String(parmArray);
    }


    /**
     * Gets the code bytes value.
     * @return the size in bytes of the applet byte code.
     */
    public int getCodeBytes() {
        return codeBytes;
    }

    /**
     * Gets the data bytes value.
     * @return the size in bytes of the applet data.
     */
    public int getDataBytes() {
        return dataBytes;
    }

    /**
     * Gets the max stack value.
     * @return the maximum number of bytes of stack space required by this applet.
     */
    public int getMaxStack() {
        return maxStack;
    }

    /**
     * Gets the max RAM value.
     * @return the number of bytes of RAM required by this applet.
     */
    public int getMaxRam() {
        return maxRam;
    }

    /**
     * Gets an applet property.
     * @param propertyName is the name of the property, for example, "version".
     * @return the value of the property, for example, "1.0",
     * or null if the property name is not defined.
     */
    public String getProperty(String propertyName) {
        return (String)ht.get(propertyName);
    }

    /**
     * This method produces a header string
     * @return String header line
     */
    private static String toHeader() {
        StringBuffer sb = new StringBuffer();
        sb.append("'class name'\t" + "'AID'");
        return sb.toString();
    }

    /**
     * This method produces a string equivalent to a config file entry.
     * <br>Format:        {class name}{tab}"{AID}"
     * @return String equivalent of config file entry
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append(className + "\t");

        int n;
        byte[] aidBa = AID.getBytes();

        for (int i=0; i<aidBa.length; i++){
            n = aidBa[i] & 0xFF;
            if (i != 0) {
                sb.append(":");
            }
            sb.append("0x");
            sb.append(Integer.toHexString(n));
        }
        return sb.toString();
    }

    public byte getPriviledges() {
        return priviledges;
    }

    public void setPriviledges(int priviledges) {
        this.priviledges = (byte)priviledges;
    }

    public int getNonVolatileSpace() {
        return nonVolatileSpace;
    }

    public void setNonVolatileSpace(int nonVolatileSpace) {
        this.nonVolatileSpace = nonVolatileSpace;
    }

    public int getVolatileSpace() {
        return volatileSpace;
    }

    public void setVolatileSpace(int volatileSpace) {
        this.volatileSpace = volatileSpace;
    }

    public byte[] getGPParameters() {
        return gpParameters;
    }

    public void setGPParameters(byte[] systemParameters) {
        this.gpParameters = systemParameters;
    }
}
