/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.ArrayList;
import java.util.Iterator;

import com.sun.javatest.Status;

import com.sun.javacard.cjck.invoke.ConfigFileReader;
import com.sun.javacard.cjck.invoke.ConfiguredScript;
import com.sun.javacard.cjck.invoke.CustomizableScript;
import com.sun.javacard.cjck.invoke.CardProxyArguments;
import com.sun.javacard.cjck.invoke.CardProxyTest;
import com.sun.javacard.cjck.invoke.Utils;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.userinterface.OnCardException;

/**
 * This class defines standard script for Java Card tests. This class
 * provides access to the applet and package properties using
 * full-qualified or local names. The script resolves these name using
 * applets and package description from
 * <code>ConfigFileReader</code>. If the applet or package with the
 * same qualified name does not found, the the scripts tries to find applet
 * or package with the same local name. If the more than one objects
 * or no objects are found in the ConfigFileReader then methods throws
 * <code>IllegalArgumentException</code>.
 * <p>
 * The following method can throw <code>IllegalArgumentException</code><ul>
 *   <li>{@link #deleteApplets(String[], boolean)}
 *   <li>{@link #deletePackage(String, boolean)}
 *   <li>{@link #deletePackageAndInstances(String, boolean)}
 *   <li>{@link #findApplet(String)}
 *   <li>{@link #findPackageName(String)}
 *   <li>{@link #select(String)}
 * </ul>
 * <p>
 * <b>Note</b> that scripts for Java Card TCK tests should be defined in the
 * other package than classes, which are loaded on the card. Otherwise
 * this test fails in the convert mode and it can broke the Java Card
 * TCK build.
 */
public abstract class StdScript implements ConfiguredScript, CustomizableScript {
    
    public static final byte NOTHING = 0;
    public static final byte APPLETS = 1;
    public static final byte PACKAGES = 2;
    public static final byte ALL = 3;
    
    
    protected CJCKCardService cardService;
    protected ConfigFileReader configFile;
    protected CardProxyArguments args;
    protected PrintWriter out;
    protected PrintWriter ref;
    private boolean isExcluded;
    protected int apduCount;
    private Hashtable excluded_testcases = new Hashtable();

    protected byte DEFAULT_LE = (byte)0x7f;
    protected byte DELETE_POLICY = ALL;

    private byte GET_RESPONSE_CLA_BYTE = (byte)0x00;

    public StdScript() {
    }


    /**
     * sets ConfigFileReader for name resolving
     */
    public void setConfigFileReader(ConfigFileReader configFile) {
        this.configFile = configFile;
    }

    /**
     * returns ConfigFileReader used for name resolving
     */
    public ConfigFileReader getConfigFileReader() {
        return this.configFile;
    }


    /**
     * runs scripts. This methods sets requested resources and executes
     * abstract method runScript. The CardProxyException thrown in
     * runScript will be reported as failure.
     */
    public Status run(CJCKCardService cardService, PrintWriter out,
                      PrintWriter ref) throws CardProxyException {
        init(cardService, out, ref);
        this.apduCount = 0;
        if (isExcluded) {
            return Status.passed(I18n.getString("script.excluded"));
        }
        try {
            cardService.powerUp();
            return runScript();
        } catch (ScriptFailException e) {
            return e.getStatus();
        } finally {
            cardService.powerDown();
        }
    }

    protected void init(CJCKCardService cardService, PrintWriter out,
                        PrintWriter ref) {
        
        this.cardService = cardService;
        this.out = out;
        this.ref = ref;
    }

    /**
     * executes commands defined by subclass.
     */
    public abstract Status runScript() throws CardProxyException;


    /**
     * sends APDU to the card and verifies response. If the response
     * does not equal to the expected response then ScriptFailException
     * will be thrown.
     * @param apdu specifies APDU which will send to the card
     * @param expectedResponse specifies expected response for verification
     * @throws ScriptFailException if the response does not equals
     * to expectedResponse or
     * @throws CardProxyException if CardProxyException is thrown by
     * CJCKCardService implementation. 
     */
    public void sendAPDU(CommandAPDU apdu, short expectedResponse)
        throws CardProxyException {
        this.apduCount++;
        ResponseAPDU answer = sendAPDU(apdu);
        int sw = answer.sw();
        if ((sw & 0xFFFF) != (0xFFFF & expectedResponse)) {

            log(I18n.getString("incorrect.response", new Integer(this.apduCount), 
                               Integer.toHexString(0xFFFF & expectedResponse), 
                               Integer.toHexString(0xFFFF & sw)));
            throw new ScriptFailException(I18n.getString("incorrect.response.1", 
                                          new Integer(this.apduCount)));
        }
    }

    public ResponseAPDU checkExceptionStatus(ResponseAPDU answer) {
        if (answer != null) {
            int status = 0xFFFF & answer.sw();
            int sw1 = extractByte(status, 1);
            int sw2 = extractByte(status, 0);
            if (sw1 == 0x9c) {
                throw OnCardException.create(sw2, answer);
            }
        }
        return answer;
    }

    private static int extractByte(int value, int count) {
        return (value >> (count * 8)) & 0xFF;
    }

    public ResponseAPDU sendAPDU(CommandAPDU apdu) throws CardProxyException {
        this.apduCount++;
        ResponseAPDU answer = cardService.sendAPDU(apdu);
        int sw = answer.sw();
        if ((sw & 0xFF00) != 0x6100) {
            return checkExceptionStatus(answer);
        } else {
            ArrayList list = new ArrayList();
            int length = 0; //0xFF & sw; 
            while (((sw & 0xFF00) == 0x6100)
                   || ((sw & 0xFF00) == 0x6200)) {
                list.add(answer);
                length += answer.getLe();
                answer = cardService.sendAPDU(getGetResponseAPDU(apdu.getCLA(), sw));
                sw = answer.sw();
            }
            list.add(answer);
            length += answer.getLe();
            byte[] data = new byte[length + 2];
            int offset = 0;
            for (Iterator e = list.iterator(); e.hasNext();) {
                answer = (ResponseAPDU)e.next();
                for (int i = 0; i < answer.getLe(); i++) {
                    data[offset++] = (byte)answer.getData(i); 
                }
            }
            data[offset++] = (byte)(sw >> 8);
            data[offset++] = (byte)sw;
            return checkExceptionStatus(new ResponseAPDU(data));
        }
    }

    private CommandAPDU getGetResponseAPDU(byte original_cla, int sw) {
        return new CommandAPDU(new byte[] {
                                   (byte)((GET_RESPONSE_CLA_BYTE & 0xF0) | (original_cla & 0x0F)),
                                   (byte)0xc0,
                                   (byte)0x00, (byte)0x00,
                                   (byte)sw
                               });
    }



    /**
     * deletes applets and verifies status.
     * @param names specifies list of the Applet names or name suffixes
     * for deletion. These names must be described in the configuration 
     * file otherwise IllegalArgumentException will be thrown.
     * @param shouldRemove This parameter is true if the these applets can be
     * deleted from the card.
     * @return true if the applets have been deleted and failed otherwise.
     * @throws ScriptFailException if the shouldRemove is false and applets are
     * deleted only.
     * @throws IllegalArgumentException if at least one of the names
     * does not described in the config file or if more than one
     * applets with given suffix exist in the configuration file.
     */
    public boolean deleteApplets(String[] names, boolean shouldRemove)
        throws CardProxyException {
        AppletProperties[] applets = new AppletProperties[names.length];
        for (int i = 0; i < names.length; i ++) {
            applets[i] = findApplet(names[i]);
        }
        return deleteApplets(applets, shouldRemove);
    }
    

    /**
     * deletes package without instances and verifies status.
     * @param name specifies name of the package or package name suffix.
     * This name must be described in the configuration 
     * file otherwise IllegalArgumentException will be thrown.
     * @param shouldRemove This parameter is true if the the package can be
     * deleted from the card.
     * @return true if the package have been deleted and false otherwise.
     * @throws ScriptFailException if the shouldRemove is false and package is
     * deleted only.
     * @throws IllegalArgumentException if the name
     * does not described in the config file or if the more than one
     * packages with given suffix exist in the configuration file.
     */
    public boolean deletePackage(String name, boolean shouldRemove)
        throws CardProxyException {
        return deletePackage(findPackageName(name), shouldRemove);
    }
 

    /**
     * deletes package with instances and verifies status.
     * @param name specifies name of the package or package name suffix.
     * This name must be described in the configuration 
     * file otherwise IllegalArgumentException will be thrown.
     * @param shouldRemove This parameter is true if the the package can be
     * deleted from the card.
     * @return true if the package have been deleted and false otherwise.
     * @throws ScriptFailException if the shouldRemove is false and package is
     * deleted only.
     * @throws IllegalArgumentException if the name
     * does not described in the config file or if the more than one
     * packages with given suffix exist in the configuration file.
     */
    public boolean deletePackageAndInstances(String name, boolean shouldRemove)
        throws CardProxyException {
        return deletePackageAndInstances(findPackageName(name), shouldRemove);
    }


    /**
     * selects applet with its name and verifies status.
     * @param name specifies full-qualified or local name of the applet used for
     * selection.
     * @param expected expected status of the response.
     * @throws ScriptFailException if the selection APDU returns SW which is not
     * equal to <code>expected</code>.
     * @throws IllegalArgumentException if the name resolving fails.
     */
    public void select(String name, short expected)
        throws CardProxyException {
        select(findApplet(name), expected);
    }
    
    /**
     * selects applet with its name and verifies status.
     * @param name specifies full-qualified or local name of the applet used for
     * selection.
     * @throws ScriptFailException if the selection APDU returns SW which is not
     * equal to <code>0x9000</code>.
     * @throws IllegalArgumentException if the name resolving fails.
     */
    public void select(String name) throws CardProxyException {
        log(I18n.getString("try.select.applet", name));
        select(findApplet(name), (short)0x9b00);
    }

    private boolean deleteApplets(AppletProperties[] applets,
                                  boolean shouldRemove)
        throws CardProxyException {
        log(I18n.getString("start.delete.applet", createAppletList(applets), 
                           shouldRemove ? Boolean.TRUE : Boolean.FALSE));
        boolean status = cardService.deleteAppletInstances(applets);
        if (shouldRemove) {
            if (!status && (DELETE_POLICY >= APPLETS)) {
                throw new ScriptFailException(I18n.getString("applet.deletion.incorrect.rejected"));
            }
        } else {
            if (status) {
                throw new ScriptFailException(I18n.getString("applet.incorrectly.deleted"));
            }
        }
        return status;
    }

    private boolean deletePackage(AppletID packageAID, boolean shouldRemove)
        throws CardProxyException {
        log(I18n.getString("start.delete.package", packageAID, 
                           shouldRemove ? Boolean.TRUE : Boolean.FALSE));
        
        boolean status = cardService.deletePackage(packageAID,
                                                   getPackageName(packageAID));
        if (shouldRemove) {
            if (!status && (DELETE_POLICY >= PACKAGES)) {
                throw new ScriptFailException(I18n.getString("package.deletion.incorrectly.rejected"));
            }
        } else {
            if (status) {
                throw new ScriptFailException(I18n.getString("package.incorrectly.deleted"));
            }
        }
        return status;
    }

    protected String getPackageName(AppletID aid) {
        for (Enumeration e = configFile.getPackages(); e.hasMoreElements();) {
            String name = (String)e.nextElement();
            if (aid.equals(configFile.getPackageAID(name))) {
                return name;
            }
        }
        throw new IllegalArgumentException(I18n.getString("package.does.not.exist", aid));
    }
    
    private boolean deletePackageAndInstances(AppletID packageAID,
                                              boolean shouldRemove)
        throws CardProxyException {
        log(I18n.getString("start.delete.package_and_instances", packageAID,
                           shouldRemove ? Boolean.TRUE : Boolean.FALSE));
        boolean status = cardService.deletePackageAndInstances(packageAID,
                                                               getPackageName(packageAID));
        if (shouldRemove) {
            if (!status && (DELETE_POLICY >= PACKAGES)) {
                throw new ScriptFailException(I18n.getString("package.ins.deletion.incorrectly.rejected"));
            }
        } else {
            if (status) {
                throw new ScriptFailException(I18n.getString("package.ins.incorrectly.deleted"));
            }
        }
        return status;
    }

    private void select(AppletProperties applet, short expected)
        throws CardProxyException {
        log(I18n.getString("start.select.applet", applet.getClassName(), Integer.toHexString(expected)));
        sendAPDU(new CommandAPDU((byte)0x00, (byte)0xA4, (byte)0x04, (byte)0x00,
                                 applet.getAID().getBytes(), DEFAULT_LE),
                 expected);
    }
    
    
    private class NameSet {

        protected void check() throws IllegalArgumentException {
            if (configFile == null) {
                throw new IllegalArgumentException(I18n.getString("can.not.resolve.object.config.undefined"));
            }
        }
        
        public Enumeration keyObjects() {
            return configFile.getAppletVector().elements();
        }
        
        public String getName(Object applet) {
            return ((AppletProperties)applet).getClassName();
        }

        public Object getObject(Object key) {
            return key;
        }

    }
    
    public AppletProperties findApplet(String name) throws CardProxyException {
        NameSet set = new NameSet();
        set.check();
        return (AppletProperties)findObject(name, set);
    }


    public AppletID findPackageName(String name) throws CardProxyException {
        NameSet set = new NameSet() {
                public Enumeration keyObjects() {
                    return configFile.getPackages();
                }
                
                public String getName(Object applet) {
                    return (String)applet;
                }

                public Object getObject(Object key) {
                    return configFile.getPackageAID((String)key);
                }
            };
        set.check();
        return (AppletID)findObject(name, set);
    }


    private Object findObject(String name, NameSet set)
        throws IllegalArgumentException {
        Object key = null;
        String suffix = name.startsWith(".") ? name : ("." + name);
        boolean isAmbiguous = false;
        for (Enumeration e = set.keyObjects(); e.hasMoreElements();) {
            Object current = e.nextElement();
            String objectName = set.getName(current);
            if (name.equals(objectName)) {
                return set.getObject(current);
            }
            if (objectName.endsWith(suffix)) {
                if (key != null) {
                    isAmbiguous = true;
                } else {
                    key = current;
                }
            }
        }
        
        if (isAmbiguous) {
            throw new IllegalArgumentException(I18n.getString("more.object.exist.with.given.suffix"));
        } else if (key == null) {
            throw new IllegalArgumentException(I18n.getString("can.not.resolve.object.by.name"));
        } else {
            Object result = set.getObject(key);
            return result;
        }
    }

    public static int makeInt(byte high, byte low) {
        return (0xFF00 & (high << 8)) | (0xFF & low);
    }
    
    protected void log(Object o) {
        if (out != null) {
            out.println(I18n.getString("stdscript.script.log", getName(), o));
        }
    }

    protected void ref(Object o) {
        log(o);
        if (ref != null) {
           ref.println(o);
        }
    }

    public void setExcludeEntries(String[] entries) {
        String name = getName();
        for (int i = 0; i < entries.length; i++) {
            excluded_testcases.put(entries[i], "OK");
            if (name.equals(entries[i])) {
                isExcluded = true;
                excluded_testcases.clear();
                return;
            }
        }
    }
    
    /**
     * returns true if the current testcase is not excluded by
     * JavaTest and false otherwise.
     * @param name name of the testcase.
     */
    public boolean checkTestCase(String name) {
        return (name == null) || (excluded_testcases.get(name) == null);
    }

    private String name;
    
    public String getName() {
        if (name == null) {
            name = this.getClass().getName();
            int pos = name.lastIndexOf(".");
            name = (pos > 0) ? name.substring(pos + 1) : name;
        }
            
        return this.name;
    }
    
    public void setArguments(String args[]) {
        for (int i = 0; i < args.length; i++) {
            if (args[i].equals("-DeletePolicy")
                && ((i + 1) < args.length)) {
                if (args[i + 1].equals("packages")) {
                    DELETE_POLICY = PACKAGES;
                } else if (args[i + 1].equals("applets")) {
                    DELETE_POLICY = APPLETS;
                } else if (args[i + 1].equals("all")) {
                    DELETE_POLICY = ALL;
                } else if (args[i + 1].equals("nothing")) {
                    DELETE_POLICY = NOTHING;
                }
            }
        }
    }
    
    public void installCapFile(String fileName,
                               String[] applets)
        throws CardProxyException {
        this.installCapFileInt(args.getCapFileDir(), fileName,
                               findAppletProperties(applets));
    }
    
    public void installCapFile(String fileName) throws CardProxyException {
        this.installCapFileInt(args.getCapFileDir(), fileName,
                               this.configFile.getPropertiesByCapFile(fileName));
    }
    
    public void installRefCapFile(String fileName,
                                  String[] applets)
        throws CardProxyException {
        this.installCapFileInt(args.getRefCapFileDir(), fileName,
                               findAppletProperties(applets));
    }

    public void installRefCapFile(String fileName) throws CardProxyException {
        this.installCapFileInt(args.getRefCapFileDir(), fileName,
                               configFile.getPropertiesByCapFile(fileName));
    }

    private AppletProperties[] findAppletProperties(String[] names)
        throws CardProxyException {

        AppletProperties[] retVal = new AppletProperties[names.length];
        for (int i = 0; i < retVal.length; i++) {
            retVal[i] = findApplet(names[i]);
        }
        return retVal;
    }
    
    private void installCapFileInt(String dir, String capFileName,
                                   AppletProperties[] applets)
        throws CardProxyException {
        ConfigFileReader config = getConfigFileReader();
        String name = Utils.getCanonicalName(dir, capFileName);
        
        log(I18n.getString("start.installing.capfile.name", name));
        String info = Utils.fileInfo(name);
        if (info == null) {
            throw new ScriptFailException(I18n.getString("can.not.find.capfile.name", name));
        }

        boolean s = cardService.installCAPFile(applets, name,
                                               this.args.getOutputDir());
        if (!s) {
            throw new ScriptFailException(I18n.getString("installcapfile.failed", name));
        }
    }
    
    public void setCardProxyArguments(CardProxyArguments arguments) {
        this.args = arguments;
        this.GET_RESPONSE_CLA_BYTE = arguments.getGET_RESPONSE_CLA_BYTE();
    }

    private static String createAppletList(AppletProperties[] list) {
        if (list == null) {
            return "null";
        }
        StringBuffer retVal = new StringBuffer("[");
        for (int i = 0; i < list.length; i++) {
            retVal.append(list[i].getClassName());
            retVal.append(", ");
        }
        retVal.append("]");
        return retVal.toString();
    }
}



