/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.Vector;

import com.sun.javatest.Status;

import com.sun.javacard.cjck.scripts.rmi.RemoteRef;
import com.sun.javacard.cjck.scripts.rmi.Parameter;
import com.sun.javacard.cjck.I18n;

public class RemoteMethod {
    private RemoteRef remote;
    private String name;
    private String signature;
    private Vector parameters = new Vector();

    /**
     * creates instance with given remote object reference.
     */
    public RemoteMethod(RemoteRef object) {
        this.remote = object;
    }

    /**
     * creates instance with given remote object reference and method.
     * @param object given remote object reference
     * @param name local name of the method.
     * @param signature method descriptor. Representation of a method descriptor
     * is the same as that described in The Java Virtual Machine Specification.
     */
    public RemoteMethod(RemoteRef object, String name, String signature) {
        this.remote = object;
        setMethod(name, signature);
    }

    /**
     * sets method.
     * @param name local name of the method.
     * @param signature method descriptor. Representation of a method descriptor
     * is the same as that described in The Java Virtual Machine Specification.
     */
    public void setMethod(String name, String signature) {
        this.name = name;
        this.signature = signature;
    }
    
    /**
     * Adds parameter. This method does not verify compatibility with
     * method signature.
     */
    public void addParameter(Parameter arg) {
        parameters.addElement(arg);
    }

    /**
     * Adds parameters. This method does not verify compatibility with
     * method signature.
     */
    public void addParameters(Parameter[] args) {
        for (int i = 0; i < args.length; parameters.addElement(args[i++]));
    }


    /**
     * removes all parameters, which have been added by addParameter
     * or addParameters methods.
     */
    public void clear() {
        parameters.clear();
    }
    
    /**
     * returns data for sending to the Card.
     * This methods returns only raw data without le or lc values.
     */
    public byte [] getData() {
        int length = 4;
        for (Enumeration e = parameters.elements(); e.hasMoreElements();
             length += ((Parameter)e.nextElement()).getBytes().length);
        byte[] retVal = new byte[length];
        int objectID = remote.getRefID();
        short methodID = calculateMethodId(remote.getHashModifier(),
                                         name, signature);
        int offset = 0;
        retVal[offset++] = (byte)(objectID >> 8);
        retVal[offset++] = (byte) objectID;
        retVal[offset++] = (byte)(methodID >> 8);
        retVal[offset++] = (byte) methodID;
        for (Enumeration e = parameters.elements(); e.hasMoreElements();) {
            byte[] data = ((Parameter)e.nextElement()).getBytes();
            System.arraycopy(data, 0, retVal, offset, data.length);
            offset += data.length;
        }
        return retVal;
    }

    /**
     * calculates 16-bit hashcode.
     * @throws ScriptFailException with error status if the SHA-1 algorithm or
     * UTF-8 encoding are unsupported.
     */
    public static short calculateMethodId(String hash_prefix, String name,
                                          String signature) {
        Status reason = null;
        try {
            MessageDigest sha = MessageDigest.getInstance("SHA");
            byte[] name_id = (hash_prefix + name + signature).getBytes("UTF-8");
            sha.update(name_id);
            byte[] data = sha.digest();
            return (short)((0xFF00 & (data[0] << 8)) | (0xFF & (data[1])));
        } catch (UnsupportedEncodingException e) {
            reason = Status.error(I18n.getString("unsupported.utf.encoding"));
        } catch (NoSuchAlgorithmException e1) {
            reason = Status.error(I18n.getString("unsupported.sha.algorithm"));
        }
        throw new ScriptFailException(reason);
    }

    public String toString() {
        return I18n.getString("remotemethod.name.signature", this.name, this.signature);
    }
}
