/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import java.io.PrintWriter;
import java.util.Enumeration;

import com.sun.javatest.Status;
import com.sun.javacard.cjck.invoke.ConfigFileReader;
import com.sun.javacard.cjck.invoke.ConfiguredScript;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.scripts.rmi.Parameter;
import com.sun.javacard.cjck.scripts.rmi.RemoteRef;
import com.sun.javacard.cjck.scripts.rmi.RemoteRefWithClass;
import com.sun.javacard.cjck.scripts.rmi.RemoteRefWithInterfaces;
import com.sun.javacard.cjck.scripts.rmi.ResponseException;
import com.sun.javacard.cjck.I18n;

public abstract class RMIScript extends LCScript {

    public static final int JC_RMI_MAJOR_VERSION   = 0x02;
    public static final int JC_RMI_MINOR_VERSION   = 0x02;
    
    public static final int FCI_TAG                = 0x6F;
    public static final int APPLICATION_DATA_TAG   = 0x6E;
    public static final int JC_RMI_DATA_TAG        = 0x5E;

    public static final int ERROR_TAG              = 0x99;
    public static final int EXCEPTION_SUBCLASS_TAG = 0x83;
    public static final int EXCEPTION_TAG          = 0x82;
    public static final int NORMAL_TAG             = 0x81;
    
    public static final int RMI_NORMAL_RESPONSE    = 0x9000;
    public static final int RMI_SELECTION_SUCCESS  = 0x9000;
    
    public static final int JC_RMI_ERROR_LENGTH    = 6;
    public static final int JC_RMI_DATA_OFFSET     = 3;

    public static final int VOID          = 0x01;
    public static final int BOOLEAN       = 0x02;
    public static final int BYTE          = 0x03;
    public static final int SHORT         = 0x04;
    public static final int INT           = 0x05;
    public static final int REMOTE_REF    = 0x06;
    public static final int ARRAY         = 0x10;
    public static final int BOOLEAN_ARRAY = 0x12;
    public static final int BYTE_ARRAY    = 0x13;
    public static final int SHORT_ARRAY   = 0x14;
    public static final int INT_ARRAY     = 0x15;
    public static final byte NULL_TAG     = (byte)0xff;

    public static final int INTERFACE_MASK = (byte)0x10;
    
    public byte DEFAULT_RMI_LE = (byte)0x7f;
    public int SECURE_MASK = (byte)0x00;
   
    private byte partialSelectionByte = (byte)0;
    
    private boolean isClass;
    private byte invoke_ins;
    private int version;
    
    /**
     * This method selects applet with given name on the given channel.
     * This method returns initial remote reference and initializes the invoke INS.
     * @param name name or suffix of the class. This class name should be
     * unambiguously defined in the configuration file.
     * @param channel the number of the channel for selection.
     * @param isClass If it is true, then remote reference descriptor uses
     * <code>remote_ref_with_class</code>.
     * @throws UnexpectedSWException if the returned SW is not 0x9000.
     * @throws CardProxyException if the CJCKCardService throws
     * CardProxyException during selection.
     */
    public RemoteRef selectWithRMI(String name, int channel, boolean isClass)
        throws CardProxyException, UnexpectedSWException {
        return selectWithRMI(findApplet(name), channel, isClass);
    }

    /**
     * selects applet by the AppletProperties.
     */
    public RemoteRef selectWithRMI(AppletProperties applet, int channel,
                                   boolean isClass)
        throws CardProxyException, UnexpectedSWException {
        log(I18n.getString("start.select.rmi", applet.getClassName(), new Integer(channel), 
                           isClass ? Boolean.TRUE : Boolean.FALSE));
        checkChannel(channel);
        byte CLA = getInterindustryChannelCLA(channel);
        byte p2 = (byte)(partialSelectionByte | (isClass ? 0 : INTERFACE_MASK));
        CommandAPDU apdu = new CommandAPDU(CLA, (byte)0xA4,
                                           (byte)0x04, p2,
                                           applet.getAID().getBytes(),
                                           DEFAULT_RMI_LE);
        this.isClass = isClass;
        ResponseAPDU res = sendAPDU(apdu);
        if (res.sw() != RMI_SELECTION_SUCCESS) {
            throw new UnexpectedSWException(res);
        }
        return parseInitialRemoteRef(res);
    }

    public byte getInvokeInstruction() {
        return this.invoke_ins;
    }
    
    public Parameter invokeRemoteMethod(RemoteMethod method, int channel,
                                        int expected_type)
        throws CardProxyException {
        return invokeRemoteMethod(method, channel, this.invoke_ins, expected_type);
    }
    
    /**
     * sends INVOKE commands on the card. If the answer is exception
     * or error response then ResponseException should be thrown.
     * @param method the method for invocation.
     * @param channel the logical channel number.
     * @param expected_type the expected type of the result.
     */
    public Parameter invokeRemoteMethod(RemoteMethod method, int channel,
                                        int invoke_ins,
                                        int expected_type)
        throws CardProxyException {
        return this.invokeRemoteMethod(method, channel, invoke_ins,
                                       expected_type, (byte)DEFAULT_RMI_LE);
    }
    
    /**
     * sends INVOKE commands on the card. If the answer is exception
     * or error response then ResponseException should be thrown.
     * @param method the method for invocation.
     * @param channel the logical channel number.
     * @param expected_type the expected type of the result.
     * @param expected_le the expected response data length
     */
    public Parameter invokeRemoteMethod(RemoteMethod method, int channel,
                                        int invoke_ins,
                                        int expected_type,
                                        byte expected_le)
        throws CardProxyException {
        log(I18n.getString("start.invoke.remote.method",  
                           new Object[] {method, new Integer(channel), 
                                         Integer.toHexString(invoke_ins), 
                                         Integer.toHexString(expected_type)}));
        checkChannel(channel);
        byte CLA = getSecureProprietaryChannelCLA(channel, true, SECURE_MASK);
        CommandAPDU apdu = new CommandAPDU(CLA,
                                           (byte)invoke_ins,
                                           (byte)JC_RMI_MAJOR_VERSION,
                                           (byte)JC_RMI_MINOR_VERSION,
                                           method.getData(),
                                           expected_le);
        ResponseAPDU resp = sendAPDU(apdu);
        if (resp.sw() != RMI_NORMAL_RESPONSE) {
            throw new UnexpectedSWException(resp);
        }
        TLV answer = new TLV(resp, 0, 0, resp.getLe());
        if (!parseErrorOrException(answer, 0)) {
            if (expected_type == REMOTE_REF) {
                check(answer.getData(0) == NORMAL_TAG,
                      I18n.getString("unknown.tag.in.response"));
                return parseRemoteRefInternal(answer, 1);
            } else {
                check(resp.getLe() > 0,
                      I18n.getString("response.should.contain.onebyte"));
                byte[] data = new byte[resp.getLe() - 1];
                System.arraycopy(resp.getBytes(), 1, data, 0, data.length);
                return Parameter.parseReturnData(data, expected_type);
            }
        }
        return null;
    }
    
    private RemoteRef parseInitialRemoteRef(ResponseAPDU apdu)
        throws CardProxyException {
        TLV rmi = TLV.findTLV(JC_RMI_DATA_TAG,
                              TLV.findTLV(APPLICATION_DATA_TAG,
                                          TLV.findTLV(FCI_TAG, apdu, 0,
                                                      apdu.getLe(), "FCI"),
                                  "APPLICATION_DATA"),
                          "JC_RMI_DATA_TAG");
        check(((rmi.offset + rmi.length) ==  apdu.getLe()),
              I18n.getString("response.incorrectly.extended"));
        this.invoke_ins = (byte)rmi.getData(2);
        this.version = (0xFF00 & (rmi.getData(0) << 8)) | rmi.getData(1);

        if (parseErrorOrException(rmi, JC_RMI_DATA_OFFSET)) {
            log(I18n.getString("rmi.protocal.error"));
            return null;
        } else {
            return parseRemoteRefInternal(rmi, JC_RMI_DATA_OFFSET 
                                          /* w/a of the #4530752 */ + 1);
        }
    }

    
    private RemoteRef parseRemoteRefInternal(TLV rmi, int offset)
        throws CardProxyException {
        try {   
            if (isNullResponse(rmi, offset)) {
                return new RemoteRef();
            } 
            int id = makeInt((byte)rmi.getData(offset),
                             (byte)rmi.getData(offset + 1));
            offset += 2;
            String hash = new String(copyBytes(rmi, offset + 1, rmi.getData(offset)),
                                     "UTF-8");
            offset += 1 + rmi.getData(offset);
                
            if (this.isClass) {
                String pack =  new String(copyBytes(rmi, offset + 1,
                                                    rmi.getData(offset)),
                                          "UTF-8");
                offset += 1 + rmi.getData(offset);
                String name =  new String(copyBytes(rmi, offset + 1,
                                                    rmi.getData(offset)), "UTF-8");
                offset += 1 + rmi.getData(offset);
                check((offset == rmi.length), I18n.getString("response.incorrectly.extended"));
                return new RemoteRefWithClass(id, hash, pack, name); 
            } else {
                int count = rmi.getData(offset++);
                String[][] interfaces = new String[count][2];
                for (int i = 0; i < count; i++) {
                    interfaces[i][0] = new String(copyBytes(rmi, offset + 1,
                                                            rmi.getData(offset)),
                                                  "UTF-8");
                    offset += 1 + rmi.getData(offset);
                    interfaces[i][1] = new String(copyBytes(rmi, offset + 1,
                                                            rmi.getData(offset)),
                                                  "UTF-8");
                    offset += 1 + rmi.getData(offset);
                }
                return new RemoteRefWithInterfaces(id, hash, interfaces); 
            }
        } catch (java.io.UnsupportedEncodingException e) {
            throw new ScriptFailException(Status.error(I18n.getString("unsupported.utf.encoding")));
        }
    }

    private boolean parseErrorOrException(TLV rmi, int offset)
        throws CardProxyException {
        int type = rmi.getData(offset);
        switch (type) {
        case NORMAL_TAG: return false;
        case EXCEPTION_TAG: case EXCEPTION_SUBCLASS_TAG:
            check(rmi.length == (offset + 4), I18n.getString("invalid.exception.structure"));
            throw new ResponseException(type, rmi.getData(offset + 1),
                                        makeInt((byte)rmi.getData(offset + 2),
                                                (byte)rmi.getData(offset + 3)));
        case ERROR_TAG: check(rmi.length == (offset + 3), I18n.getString("invalid.error.structure"));
            throw new ResponseException(ERROR_TAG, 0,
                                        makeInt((byte)rmi.getData(offset + 1),
                                                (byte)rmi.getData(offset + 2)));
        default:
            check(false, I18n.getString("unknown.tag"));
        }
        return false;
    }
    
    public static void check(boolean status, String mes)
        throws CardProxyException {
        if (!status) {
            throw new CardProxyException(mes);
        }
    }
        
    private static byte[] copyBytes(TLV rmi, int offset, int length) {
        debug(I18n.getString("tlv.tag.debug.info", 
                             new Object[] {Integer.toHexString(rmi.tag), 
                                           Integer.toHexString(rmi.length),
                                           Integer.toHexString(offset), 
                                           Integer.toHexString(length)})); 
        byte[] dest = new byte[length];
        for (int i = 0; i < dest.length; i++) {
            dest[i] = (byte)rmi.getData(offset + i);
        }
        return dest;
    }
       
    private static boolean isNullResponse(TLV rmi, int offset) {
        return ((rmi.length == (offset + 2)) &&
                (rmi.getData(offset) == 0xFF) && (rmi.getData(offset + 1) == 0xFF));
    }
           

    public static void debug(Object o) {
        java.util.logging.Logger.global.log(java.util.logging.Level.FINEST, "" + o);
    }
}
