/*
 * %W% %E%
 *
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import com.sun.javacard.cjck.I18n;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class LoggingClient {

    public static final String MESSAGES_RESOURCE = "com/sun/tck/logging/messages.properties";
    public static final String MESSAGE_PREFIX = "Message.value.";
    public static final String FILE_PREFIX = "File.";
    public static final String EXCEPTION_PREFIX = "Exception.";

    public static final String UNKNOWN = "UNKNOWN";

    public final static byte CLA_JCRE_TEST = (byte)0x80;

    /**
     * INS byte: DoTests command
     */
    public final static byte INS_DO_TESTS = (byte)0x20;


    public static final byte NULL_TAG      = (byte)0x00;
    public static final byte BYTE_TAG      = (byte)0x01;
    public static final byte BOOLEAN_TAG   = (byte)0x02;
    public static final byte SHORT_TAG     = (byte)0x03;
    public static final byte EXCEPTION_TAG = (byte)0x05;
    public static final byte REF_TAG       = (byte)0x06;
    public static final byte VALUE_TAG     = (byte)0x07;
    public static final byte ARRAY_MASK    = (byte)0x80;

    public static final short EXCEPTION    = (short)0x05;
    public static final short CARD_EXCEPTION = (short)0x21;
    public static final short CARD_RUNTIME_EXCEPTION = (short)0x22;
    
    public Properties messages = new Properties();

    private boolean isDebug = true;
    
    private PrintWriter out;
    private PrintWriter ref;
    
    public LoggingClient(PrintWriter out, PrintWriter ref) {
        this.out = out;
        this.ref = ref;
        ClassLoader loader = this.getClass().getClassLoader();
        loader = (loader == null) ? ClassLoader.getSystemClassLoader() : loader;
        try {
            messages.load(loader.getResourceAsStream(MESSAGES_RESOURCE));
        } catch (IOException e) {
            ref(I18n.getString("can.not.load.messages", MESSAGES_RESOURCE, e));
        }
    }
    
    public void setDebugMode(boolean isDebug) {
        this.isDebug = isDebug;
    }

    public void ref(String message) {
        out.println(message);
        ref.println(message);
    }
    private String getMessageHeader(int id) {
        return (id == 0) ? UNKNOWN : messages.getProperty(MESSAGE_PREFIX + id);
    }
    
    public void parseLog(byte[] data) {
        ByteArrayInputStream in = new ByteArrayInputStream(data);
        ArrayList messages = new ArrayList();
        HashMap values = new HashMap();
        try {
            while (in.available() > 0) {
                int message_id = readShort(in);
                if (message_id == 0xffff) { // Value Tag
                    values.put(new Integer(in.read()), readParam(in));
                } else {
                    int argCount = in.read();
                    Object[] args = new Object[argCount];
                    Message msg = new Message(getDebugHeader(message_id),
                                              getMessageHeader(message_id), args);
                    messages.add(msg);
                    // ref("messages.add(" + msg + ")");
                    for (int i = 0; i < args.length; i++) {
                        args[i] = readParam(in);
                    }
                }
            }
            for (Iterator i = messages.iterator(); i.hasNext();) { 
                 Message current = (Message)i.next();
                 current.resolve(values);
                 ref("  " + current.toString().replaceAll("\n", "\n  "));
            }
        } catch (Exception e) {
            e.printStackTrace();
            ref(I18n.getString("can.not.parse.log", e));
        }
    }

    private String getDebugHeader(int message_id) {
        StringBuffer retVal = new StringBuffer("Message");        
        if (isDebug) {
            retVal.append(" from ");
            retVal.append(messages.getProperty("Message.file." + message_id));
            retVal.append(':');
            retVal.append(messages.getProperty("Message.line." + message_id));
            retVal.append(":\n");
        } else {
            retVal.append(": ");
        }
        return retVal.toString();
    }

    private Object readParam(ByteArrayInputStream in) throws IOException {
        int tag = in.read();
        int len = 0;
        switch (tag & 0xFF) {
            case NULL_TAG: 
                return null;
            case BYTE_TAG:
                return new Byte((byte)in.read());
            case BOOLEAN_TAG:
                return new Boolean(in.read() != 0);
            case SHORT_TAG: 
                return new Short((short)readShort(in));
            case EXCEPTION_TAG: 
                return  createException((short)readShort(in), // ExceptionType
                                        (short)readShort(in), // Reason
                                        (short)readShort(in), // File
                                        (short)readShort(in)); // line
            case (ARRAY_MASK | BYTE_TAG) & 0xFF:
                return readByteArray(in, readShort(in), true);
            case (ARRAY_MASK | BOOLEAN_TAG) & 0xFF:
                return readBooleanArray(in, readShort(in));
            case (ARRAY_MASK | SHORT_TAG) & 0xFF:
                return readShortArray(in, readShort(in), true);
            case REF_TAG:
                return new Reference(in.read());
        }
        return null;
    }
     
    private String readBooleanArray(InputStream in, int len) 
    throws IOException {
        StringBuffer retVal = new StringBuffer("boolean[");
        retVal.append(Integer.toString(len));
        retVal.append("] {");
        for (int pos = 0; pos < len; pos++) {
            retVal.append((in.read() == 0) ? "false" : "true");
            if (pos < (len - 1)) {
                retVal.append(", ");
            }
        }
        retVal.append('}');
        return retVal.toString();
       
    }

    private String readByteArray(InputStream in, int len, boolean isHex)  
    throws IOException {
        StringBuffer retVal = new StringBuffer("byte[");
        retVal.append(Integer.toString(len));
        retVal.append("] {");
        for (int pos = 0; pos < len; pos++) {
            if (isHex) {
                retVal.append("0x");
                retVal.append(Integer.toHexString(in.read()));
            } else {
                retVal.append(Integer.toString(in.read()));
            }
            if (pos < (len - 1)) {
                retVal.append(", ");
            }
        }
        retVal.append('}');
        return retVal.toString();
    }

    private String readShortArray(InputStream in, int len, boolean isHex) throws IOException {
        StringBuffer retVal = new StringBuffer("short[] {");
        for (int pos = 0; pos < len; pos++) {
            if (isHex) {
                retVal.append("0x");
                retVal.append(Integer.toHexString(readShort(in)));
            } else {
                retVal.append(Integer.toString(readShort(in)));
            }
            if (pos < (len - 1)) {
                retVal.append(", ");
            }
        }
        retVal.append('}');
        return retVal.toString();
    }

    private int readShort(InputStream in) throws IOException {
        return (in.read() << 8) | in.read();
    }
    
    private static class Message {
        private String debugHeader;
        private String format;
        private Object[] args;
        
        public Message(String debugHeader, String message, Object[] args) {
            this.format = message;
            this.args = args;
            this.debugHeader = debugHeader;
        }
        
        public void resolve(Map values) {
            for (int i = 0; i < args.length; i++) {
                if (args[i] instanceof Reference) {
                    Reference ref = (Reference)args[i];
                    args[i] = values.get(new Integer(ref.getRef()));
                }
            }
        }
        
        public String toString() {
            try {
                return ((format == UNKNOWN) ? getInternalRepresentation("") 
                        : (debugHeader + MessageFormat.format(format, args)));
            } catch (Exception e) {
                e.printStackTrace();
                return getInternalRepresentation("Broken message:");
            }
        }
        
        private String getInternalRepresentation(String type) {
            StringBuffer retVal = new StringBuffer(debugHeader);
            retVal.append(type);
            retVal.append("Format=");
            retVal.append(format);
            retVal.append(" Args(");
            retVal.append(args.length);
            retVal.append(")=(");
            for (int i = 0; i < args.length; i++) {
                retVal.append(" \"");
                retVal.append(args[i].toString());
                retVal.append('\"');
            }
            retVal.append(" )");
            return retVal.toString();            
        }
    }

    private static class Reference {
        int ref;
        public Reference(int ref) {
            this.ref = ref;
        }
        
        int getRef() {
            return ref;
        }
    }

    private String createException(short excpType, short reason,
        short file, short line) {
        return (getExceptionName(excpType)
                + "(reason=0x" + Integer.toHexString(reason & 0xFFFF)
                + ", in " + getFileName(file) + " at line " + (line & 0xFFFF) + ")");
    }

    private String getFileName(short id) {
        return (id == 0) ? UNKNOWN : (String)messages.get(FILE_PREFIX + (id & 0xFFFF));
    }
    
    private String getExceptionName(int excpType) {
        excpType = (excpType & 0xFFFF);
        String retVal = (String)exceptions.get(new Integer(excpType));
        if (retVal == null) {
            retVal = messages.getProperty(EXCEPTION_PREFIX + excpType);
        }
        return retVal;
    }

    private static HashMap exceptions = new HashMap();
    
    static {
        exceptions.put(new Integer(0x01), "java.lang.ArithmeticException");
        exceptions.put(new Integer(0x02), "java.lang.ArrayIndexOutOfBoundsException");
        exceptions.put(new Integer(0x03), "java.lang.ArrayStoreException");
        exceptions.put(new Integer(0x04), "java.lang.ClassCastException");
        exceptions.put(new Integer(0x06), "java.lang.IndexOutOfBoundsException");
        exceptions.put(new Integer(0x07), "java.lang.NegativeArraySizeException");
        exceptions.put(new Integer(0x08), "java.lang.NullPointerException");
        exceptions.put(new Integer(0x0A), "java.lang.SecurityException");
        exceptions.put(new Integer(0x0B), "java.io.IOException");
        exceptions.put(new Integer(0x0C), "java.rmi.RemoteException");
        exceptions.put(new Integer(0x20), "javacard.framework.APDUException");
        exceptions.put(new Integer(0x23), "javacard.framework.ISOException");
        exceptions.put(new Integer(0x24), "javacard.framework.PINException"); 
        exceptions.put(new Integer(0x25), "javacard.framework.SystemException");
        exceptions.put(new Integer(0x26), "javacard.framework.TransactionException");
        exceptions.put(new Integer(0x27), "javacard.framework.UserException");
        exceptions.put(new Integer(0x30), "javacard.security.CryptoException");
        exceptions.put(new Integer(0x40), "javacard.framework.service.ServiceException");
        exceptions.put(new Integer(0x21), "javacard.framework.CardException");
        exceptions.put(new Integer(0x22), "javacard.framework.CardRuntimeException");
        exceptions.put(new Integer(0x09), "java.lang.RuntimeException");
        exceptions.put(new Integer(0x05), "java.lang.Exception");
        exceptions.put(new Integer(0xFF), "java.lang.Throwable");
        exceptions.put(new Integer(0xFE), "UNKNOWN");
    }
}
