/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import java.util.Arrays;
import java.util.ArrayList;
import com.sun.javatest.Status;

import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.I18n;

import javacard.framework.ISO7816;
import javacard.framework.MultiSelectable;

/**
 * This class is base class for all tests which use or test Logical
 * Channels functionality. It provides routines for high-level access
 * to Logical Channels.
 */
public abstract class LCScript extends StdScript {
    
    public static final int MAX_CHANNEL_NUMBER = 19;
    public static final int TYPE16_MASK = 0x40;
    public static final int PROPRIETARY_MASK = 0x80;
    public static final int NOT_LAST_MASK = 0x10;
    public static final byte MANAGE_CHANNEL_INS = (byte)0x70;
    public static final byte SELECT_INS = (byte)0xA4;
    public static final byte SELECTION_BY_DF_NAME_P1 = (byte)0x04;
    public static final byte CHANNEL_OPEN_P1 = (byte)0x00;
    public static final byte CHANNEL_CLOSE_P1 = (byte)0x80;
    
    protected final static short SW_PASSED = (short)0x9B00;
    
    private int currentChannel;
    
    
    public LCScript() {
    }
    
    
    /**
     * executes MANAGE CLOSE CHANNEL APDU. This method does not verify
     * the response. It returns the ResponseAPDU object which encapsulates
     * all information. The Le of this APDU is 0.
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     */
    public ResponseAPDU manageChannelClose(int from, int to)
    throws CardProxyException {
        return manageChannelClose(from, to, 0);
    }

    /**
     * executes MANAGE CLOSE CHANNEL APDU. This method does not verify
     * the response. It returns the ResponseAPDU object which encapsulates
     * all information.
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     * @param le Le of the MANAGE CLOSE CHANNEL APDU.
     */
    public ResponseAPDU manageChannelClose(int from, int to, int le)
    throws CardProxyException {
        log(I18n.getString("start.managechannelclose", Integer.toHexString(from),
        Integer.toHexString(to), new Integer(le)));
        byte CLA = getInterindustryChannelCLA(from);
        checkChannel(to);
        CommandAPDU capdu = new CommandAPDU(CLA, MANAGE_CHANNEL_INS,
        CHANNEL_CLOSE_P1, (byte)to, new byte[0], le);
        return sendAPDU(capdu);
    }
    
    
    /**
     * executes MANAGE OPEN CHANNEL APDU. This method does not verify
     * the response. It returns the Channel object which encapsulates
     * all information. The Le of this APDU is 1.
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     */
    public Channel manageChannelOpen(int from, int to) throws CardProxyException {
        // According to ISO 7816-4 Section 6.16.3
        // Le field == 01    if P1-P2 == '0000'
        //          == Empty if P1-P2 != '0000'
        return manageChannelOpen(from, to, (to == 0) ? 1 : 0);
    }
    
    /**
     * executes MANAGE OPEN CHANNEL APDU. This method does not verify
     * the response. It returns the Channel object which encapsulates
     * all information.
     * @param from origin logical channel.
     * @param to number of the opened logical channel.
     * @param le Le of the MANAGE OPEN CHANNEL APDU.
     */
    public Channel manageChannelOpen(int from, int to, int le)
    throws CardProxyException {
        log(I18n.getString("start.managechannelopen", Integer.toHexString(from),
        Integer.toHexString(to), new Integer(le)));
        byte CLA = getInterindustryChannelCLA(from);
        checkChannel(to);
        CommandAPDU capdu = new CommandAPDU(CLA, MANAGE_CHANNEL_INS,
        CHANNEL_OPEN_P1, (byte)to, new byte[0], le);
        ResponseAPDU answer = sendAPDU(capdu);
        int id = -1;
        if ((answer.sw() & 0xFFFF) == (ISO7816.SW_NO_ERROR & 0xFFFF)) {
            if (to != 0) {
                id = to;
            } else if (answer.getLe() > 0) {
                id = answer.getData(0);
            } else {
                throw new ScriptFailException(I18n.getString("managechannel.open.no.data"));
            }
        }
        return new Channel(id, answer);
    }
    
    /**
     * selects applet with given name on the given channel. The method allows
     * to specify Le.
     * @param name specifies full-qualified or local name of the applet used for
     * selection.
     * @throws IllegalArgumentException if the name resolving fails.
     * @return Opened Channel object. The methods does not verify SW of response
     * on the <code>SELECT FILE</code> APDU. It assumes that selection is
     * successful.
     */
    public Channel selectFile(String name, byte channel, byte le)
        throws CardProxyException {
        return selectFile(findApplet(name), channel, le);
    }

    /**
     * selects applet with given name on the given channel.
     * The DEFAULT_LE is used.
     */
    public Channel selectFile(String name, byte channel)
        throws CardProxyException {
        return selectFile(findApplet(name), channel, DEFAULT_LE);
    }
   
    private Channel selectFile(AppletProperties applet, byte channel, byte le)
        throws CardProxyException {
        log(I18n.getString("start.select.file", applet.getClassName(), new Byte(channel)));
        byte CLA = getInterindustryChannelCLA(channel);
        CommandAPDU capdu = new CommandAPDU(CLA, SELECT_INS, SELECTION_BY_DF_NAME_P1
        , (byte)0x00, applet.getAID().getBytes(), le);
        return new Channel(channel, sendAPDU(capdu));
    }
    
    /**
     * Encodes channel number to interindustry CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b0000 00zz (Type 4) last or only command in chain, no SM
     * %b0100 zzzz (Type 16) last or only command in chain, no SM
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     *
     * @return interindustry CLA byte for given channel
     */
    public static byte getInterindustryChannelCLA(int channel) {
        checkChannel(channel);
        if (channel < 4) {
            return (byte)channel;
        } else {
            return (byte)(TYPE16_MASK | (channel - 4));
        }
    }
    
    /**
     * Encodes channel number to interindustry CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b0000 00zz (Type 4) last or only command in chain, no SM
     * %b0001 00zz (Type 4) not last command in chain, no SM
     * %b0100 zzzz (Type 16) last or only command in chain, no SM
     * %b0101 zzzz (Type 16) not last command in chain, no SM
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     * @param lastCommandInChain last command in chain indicator
     *
     * @return interindustry CLA byte for given channel and chaining indicator
     */
    protected static byte getInterindustryChannelCLA(int channel, boolean lastCommandInChain) {
        checkChannel(channel);
        if (channel < 4) {
            return (byte)(lastCommandInChain ? 0 : NOT_LAST_MASK
            | channel);
        } else {
            return (byte)(TYPE16_MASK
            | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | (channel - 4));
        }
    }
    
    /**
     * Encodes channel number to interindustry CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b0000 00zz (Type 4) last or only command in chain, no SM
     * %b0001 00zz (Type 4) not last command in chain, no SM
     * %b0000 yyzz (Type 4) last or only command in chain, with SM
     * %b0001 yyzz (Type 4) not last command in chain, with SM
     * %b0100 zzzz (Type 16) last or only command in chain, no SM
     * %b0101 zzzz (Type 16) not last command in chain, no SM
     * %b01y0 zzzz (Type 16) last or only command in chain, with SM
     * %b01y1 zzzz (Type 16) not last command in chain, with SM
     * y Secure Messaging indicator
     *   See ISO 7816-4:2005 Specification Section 6 for further information
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     * @param lastCommandInChain last command in chain indicator
     *
     * @return interindustry CLA byte for given channel and chaining indicator
     */
    protected static byte getSecureInterindustryChannelCLA(int channel, boolean lastCommandInChain, int securityMask) {
        checkChannel(channel);
        if (channel < 4) {
            if ((securityMask & 0x0C) != securityMask) {
                throw new IllegalArgumentException(I18n.getString("incorrect.secure.mask.type4",
                Integer.toHexString(securityMask)));
            }
            return (byte)((lastCommandInChain ? 0 : NOT_LAST_MASK)
            | securityMask
            | channel);
        } else {
            if ((securityMask & 0x20) != securityMask) {
                throw new IllegalArgumentException(I18n.getString("incorrect.secure.mask.type16",
                Integer.toHexString(securityMask)));
            }
            return (byte)(TYPE16_MASK
            | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | securityMask
            | (channel - 4));
        }
    }
    
    /**
     * Encodes channel number to proprietary CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b1000 00zz (Type 4) last or only command in chain, no SM
     * %b1100 zzzz (Type 16) last or only command in chain, no SM
     *
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     * @param lastCommandInChain last command in chain indicator
     *
     * @return proprietary CLA byte for given channel and chaining indicator
     */
    protected static byte getProprietaryChannelCLA(int channel) {
        checkChannel(channel);
        if (channel < 4) {
            return (byte)(PROPRIETARY_MASK
            | channel);
        } else {
            return (byte)(PROPRIETARY_MASK
            | TYPE16_MASK
            | (channel - 4));
        }
    }
    
    /**
     * Encodes channel number to proprietary CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b1000 00zz (Type 4) last or only command in chain, no SM
     * %b1001 00zz (Type 4) not last command in chain, no SM
     * %b1100 zzzz (Type 16) last or only command in chain, no SM
     * %b1101 zzzz (Type 16) not last command in chain, no SM
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     * @param lastCommandInChain last command in chain indicator
     *
     * @return proprietary CLA byte for given channel and chaining indicator
     */
    protected static byte getProprietaryChannelCLA(int channel, boolean lastCommandInChain) {
        checkChannel(channel);
        if (channel < 4) {
            return (byte)(PROPRIETARY_MASK
            | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | channel);
        } else {
            return (byte)(PROPRIETARY_MASK
            | TYPE16_MASK
            | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | (channel - 4));
        }
    }
    
    /**
     * Encodes channel number to proprietary CLA byte accordig to
     * following scheme from ISO7816-4:
     * %b1000 00zz (Type 4) last or only command in chain, no SM
     * %b1001 00zz (Type 4) not last command in chain, no SM
     * %b1000 yyzz (Type 4) last or only command in chain, with SM
     * %b1001 yyzz (Type 4) not last command in chain, with SM
     * %b1100 zzzz (Type 16) last or only command in chain, no SM
     * %b1101 zzzz (Type 16) not last command in chain, no SM
     * %b11y0 zzzz (Type 16) last or only command in chain, with SM
     * %b11y1 zzzz (Type 16) not last command in chain, with SM
     * y Secure Messaging indicator
     *   See ISO 7816-4:2005 Specification Section 6 for further information
     * z - Logical channel indicator
     *   Type 4 supports logical channels [0..3]
     *   Type 16 supports logical channels [4..19]
     *
     * @param channel Logical channel number from 0 to MAX_CHANNEL_NUMBER
     * @param lastCommandInChain last command in chain indicator
     *
     * @return interindustry CLA byte for given channel and chaining indicator
     */
    protected static byte getSecureProprietaryChannelCLA(int channel, boolean lastCommandInChain, int securityMask) {
        checkChannel(channel);
        if (channel < 4) {
            if ((securityMask & 0x0C) != securityMask) {
                throw new IllegalArgumentException(I18n.getString("incorrect.secure.mask.type4",
                Integer.toHexString(securityMask)));
            }
            return (byte)(PROPRIETARY_MASK
	    | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | securityMask
            | channel);
        } else {
            if ((securityMask & 0x20) != securityMask) {
                throw new IllegalArgumentException(I18n.getString("incorrect.secure.mask.type16",
                Integer.toHexString(securityMask)));
            }
            return (byte)(PROPRIETARY_MASK
	    | TYPE16_MASK
            | (lastCommandInChain ? 0 : NOT_LAST_MASK)
            | securityMask
            | (channel - 4));
        }
    }
    
    /**
     * Checks validity of channel number
     *
     * @param channel channel number to be checked
     *
     * @throws IllegalArgumentException if channel number is out of valid range
     * [0 - MAX_CHANNEL_NUMBER]
     */
    protected static void checkChannel(int channel) {
        if ((channel > MAX_CHANNEL_NUMBER) || (channel < 0)) {
            throw new IllegalArgumentException(I18n.getString("incorrect.channel.number",
            new Integer(channel), new Integer(MAX_CHANNEL_NUMBER)));
        }
    }
    
    /**
     * This method determines the number of channels available on card by
     * sending SELECT FILE commands to the applet identified by appletName
     * for channels from 0 to MAX_CHANNEL_NUMBER. The applet process method
     * should return ISO7816.SW_NO_ERROR status for SELECT FILE APDUs.
     *
     * @param appletName name of the applet used for channels testing
     *
     * @return byte array with numbers of available channels.
     */
    protected byte[] getAvailableChannels(String appletName) throws CardProxyException {
        ArrayList channels = new ArrayList(MAX_CHANNEL_NUMBER + 1);
        for( int channel = 0; channel <= MAX_CHANNEL_NUMBER; channel++ ){
            short status = selectFile(appletName, (byte)channel, (byte)0).getSWStatus();
            
            if( status == ISO7816.SW_LOGICAL_CHANNEL_NOT_SUPPORTED ) {
                log("Logical channel number: " + channel + " is not supported by JCRE");
            } else if( status != ISO7816.SW_NO_ERROR) {
                log("Failed to open logical channel number: " + channel);
                logIncorrectResponse("SELECT FILE", ISO7816.SW_NO_ERROR, status);
            } else {
                channels.add(new Byte((byte)channel));
            }
        }
        byte[] retValue = new byte[channels.size()];
        for (int i = 0; i < retValue.length; i++) {
            retValue[i] = ((Byte)channels.get(i)).byteValue();
        }
        log("JCRE supports following logical channels: " + Arrays.toString(retValue));
        return retValue;
    }
    
    /**
     * executes MANAGE OPEN CHANNEL APDU. This method verifies
     * the response and throws ScriptFailException if status isn't equal to
     * ISO7816.SW_NO_ERROR.
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     * @param expectedStatus status that should be returned by JCRE
     * in response to MANAGE CHANNEL OPEN.
     */
    protected void sendManageChannelOpen(int fromChannel, int toChannel)
    throws CardProxyException {
        sendManageChannelOpen(fromChannel, toChannel, ISO7816.SW_NO_ERROR);
    }
    
    /**
     * executes MANAGE OPEN CHANNEL APDU. This method verifies
     * the response and throws ScriptFailException if status isn't equal to
     * expected one.
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     * @param expectedStatus status that should be returned by JCRE
     * in response to MANAGE CHANNEL OPEN.
     */
    protected void sendManageChannelOpen(int fromChannel, int toChannel, short expectedStatus)
    throws CardProxyException {
        // According to ISO 7816-4 Section 6.16.3
        // Le field == 01    if P1-P2 == '0000'
        //          == Empty if P1-P2 != '0000'
        Channel result = manageChannelOpen(fromChannel, toChannel, (toChannel == 0) ? 1 : 0);
        short status = result.getSWStatus();
        if(status != expectedStatus) {
            String msg = I18n.getString("lcscript.openchannel.fail", Integer.toString(toChannel)
            , Integer.toHexString(0xFFFF & expectedStatus)
            , Integer.toHexString(0xFFFF & status));
            log(msg);
            throw new ScriptFailException(msg);
        }
    }
    
    /**
     * selects applet with given name on the given channel. Expected status of
     * SELECT FILE command is ISO7816.SW_NO_ERROR (0x9000). If selection should
     * return another status use sendSelectFile(name, channel, expectedStatus)
     * variant of the method.
     *
     * @param name specifies full-qualified or local name of the applet used for
     * selection.
     * @param channel channel on which applet should be selected
     *
     * @throws IllegalArgumentException if the name resolving fails.
     *
     * @return Opened Channel object. The methods does not verify SW of response
     * on the <code>SELECT FILE</code> APDU. It assumes that selection is
     * successful.
     */
    protected void sendSelectFile(String name, byte channel)
    throws CardProxyException {
        sendSelectFile(name, channel, ISO7816.SW_NO_ERROR, DEFAULT_LE);
    }

    /**
     * selects applet with given name on the given channel.
     * @param name specifies full-qualified or local name of the applet used for
     * selection.
     * @param channel channel on which applet should be selected
     * @param expectedStatus status that should be returned by applet process method
     * in response to selection.
     * @param le Le used in SELECT FILE APDU being sent on card.
     * @throws IllegalArgumentException if the name resolving fails.
     * @return Opened Channel object. The methods does not verify SW of response
     * on the <code>SELECT FILE</code> APDU. It assumes that selection is
     * successful.
     */
    protected void sendSelectFile(String name, byte channel, short expectedStatus, byte le)
    throws CardProxyException {
        Channel result = selectFile(findApplet(name), channel, le);
        short status = result.getSWStatus();
        if( status != expectedStatus) {
            String msg = I18n.getString("lcscript.selectchannel.fail", Integer.toString(channel)
            , Integer.toHexString(0xFFFF & expectedStatus)
            , Integer.toHexString(0xFFFF & status));
            log(msg);
            throw new ScriptFailException(msg);
        }
    }
    /**
     * selects applet with given name on the given channel.
     * The DEFAULT_LE is used as le.
     */
    protected void sendSelectFile(String name, byte channel, short expectedStatus)
    throws CardProxyException {
        sendSelectFile(name, channel, expectedStatus, DEFAULT_LE);
    }    
    /**
     * executes MANAGE CLOSE CHANNEL APDU. This method verifies
     * the response and throws ScriptFailException if status isn't equal to
     * ISO7816.SW_NO_ERROR.
     *
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     * @param expectedStatus status that should be returned by JCRE
     * in response to MANAGE CHANNEL CLOSE.
     *
     */
    protected void sendManageChannelClose(int fromChannel, int toChannel)
    throws CardProxyException {
        sendManageChannelClose(fromChannel, toChannel, ISO7816.SW_NO_ERROR);
    }
    
    /**
     * executes MANAGE CLOSE CHANNEL APDU. This method verifies
     * the response and throws ScriptFailException if status isn't equal to
     * expected one.
     *
     * @param from origin logical channel.
     * @param to number of the logical channel to be opened.
     * @param expectedStatus status that should be returned by JCRE
     * in response to MANAGE CHANNEL CLOSE.
     *
     */
    protected void sendManageChannelClose(int fromChannel, int toChannel, short expectedStatus)
    throws CardProxyException {
        ResponseAPDU result = manageChannelClose(fromChannel, toChannel, 0);
        short status = (short) result.sw();
        if (status != expectedStatus) {
            String msg = I18n.getString("lcscript.closechannel.fail", Integer.toString(toChannel)
            , Integer.toHexString(0xFFFF & expectedStatus)
            , Integer.toHexString(0xFFFF & status));
            log(msg);
            throw new ScriptFailException(msg);
        }
    }
    
    protected void logIncorrectResponse(String command, int expectedStatus, int receivedStatus) {
        log(I18n.getString("lcscript.incorrect.response", command
        , Integer.toHexString(0xFFFF & expectedStatus)
        , Integer.toHexString(0xFFFF & receivedStatus)));
    }
}
