/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.HashMap;

import com.sun.javatest.Status;

import com.sun.javacard.cjck.invoke.CardProxyArguments;
import com.sun.javacard.cjck.invoke.CardProxyTest;
import com.sun.javacard.cjck.invoke.ConfigFileReader;
import com.sun.javacard.cjck.invoke.ConfiguredScript;
import com.sun.javacard.cjck.invoke.CustomizableScript;
import com.sun.javacard.cjck.invoke.MultiTestScript;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.I18n;

/**
 * This script contains methods for working with ECSignatureApplet. This scripts
 * assumes that selected applet supports 'doTest' with data format. (P1 is used
 * as testcase number and data can be used by testcase). This applet contains one
 * public and one private EC Key and contains special processingKey, which is
 * used for parameters setting. Also testcases from 1 to 11 should implement the
 * following functionality:
 * <ol>
 *   <li>CREATE_SIG - creates signature object
 *   <li>CREATE_KA  - creates keyagreement object
 *   <li>CREATE_KA2 - creates two instance keyagreement object
 *   <li>CREATE_PRIVATE_KEY - creates private key and sets this created key as processingKey.
 *   <li>CREATE_PUBLIC_KEY  - creates public key and sets this created key as processingKey.
 *   <li>SET_P - sets APDU data as P parameter in processingKey.
 *   <li>SET_A - sets APDU data as A parameter in processingKey.
 *   <li>SET_B - sets APDU data as B parameter in processingKey.
 *   <li>SET_G - sets APDU data as G parameter in processingKey.
 *   <li>SET_R - sets APDU data as R parameter in processingKey.
 *   <li>SET_K - sets APDU data as K parameter in processingKey.
 *   <li>SET_S - sets APDU data as S parameter in private key.
 *   <li>SET_W - sets APDU data as W parameter in public key.
 *   <li>SET_PRIVATE_KEY - sets private key as processingKey.
 *   <li>SET_PUBLIC_KEY - sets public key as processingKey.
 * </ol>
 * All set methods do nothing if processingKey is null.
 * <p>
 * This class contains two EC key data for each length supported by
 * JavaCard (112, 128,160, 192).
 * 
 */
public abstract class ECScript extends MultiStdScript {
    
    public static final byte CREATE_SIG = 1;
    public static final byte CREATE_KA = 2;
    public static final byte CREATE_KA2 = 3;
    public static final byte CREATE_PRIVATE_KEY = 4;
    public static final byte CREATE_PUBLIC_KEY = 5;
    public static final byte SET_P = 6;
    public static final byte SET_A = 7;
    public static final byte SET_B = 8;
    public static final byte SET_G = 9;
    public static final byte SET_R = 10;
    public static final byte SET_K = 11;
    public static final byte SET_S = 12;
    public static final byte SET_W = 13;
    public static final byte SET_PRIVATE_KEY = 14;
    public static final byte SET_PUBLIC_KEY = 15;

    public ECScript(String[] cases) {
        super(cases);
    }

    /**
     * encapsulates data for pair of public and private elliptic curves keys
     */
    public static class KeyData {

        public KeyData(byte[] P, byte[] A, byte[] B, byte[] G, byte[] R, byte[] K, byte[] S, byte[] W) {
           this.P = P;
           this.A = A;
           this.B = B;
           this.G = G;
           this.R = R;
           this.K = K;
           this.S = S;
           this.W = W;
        }            
        public byte[] P;
        public byte[] A;
        public byte[] B;
        public byte[] G;
        public byte[] R;
        public byte[] K;
        public byte[] S;
        public byte[] W;
    }

    /**
     * returns KeyData for given length and given set number.
     * @param set number of set.
     * @param length length of the KeyData
     * @throws IllegalArgumentException if KeyData for given set and length does
     * not exist.
     */
    protected KeyData getKeyData(byte set, short length) {
        KeyData current = (KeyData)keydata.get(set+"_"+length);
        if (current == null) {
            throw new IllegalArgumentException(I18n.getString("ecscript.no.dataset", 
                                                              new Byte(set), new Short(length)));
        }
        return current;
    }


    /**
     * encodes key length value into byte array.
     */
    private byte[] encodeKeyCreateData(short keyLength) {
        return new byte[] {
            (byte)(keyLength >> 8),
            (byte)keyLength
        };
    }

    private boolean createKey(byte step, short keyLength)
        throws CardProxyException {
        CommandAPDU apdu = new CommandAPDU((byte)0x80, (byte)0x20, step,
                                           (byte)0,
                                           encodeKeyCreateData(keyLength),
                                           (byte)0x7f); 
        ResponseAPDU answer = sendAPDU(apdu);
        int sw = answer.sw();
        if (((sw & 0xFFFF) != 0x9b00) && ((sw & 0xFFFF) != 0x9b01)) {
            log(I18n.getString("ecscript.incorrect.response", new Integer(this.apduCount),
                               Integer.toHexString(0xFFFF & sw)));
            throw new ScriptFailException(I18n.getString("ecscript.exception", new Integer(this.apduCount)));
        }
        return ((sw & 0xFFFF) == 0x9b00);
    }


    /**
     * sends request on private key creation with given key length.
     * The created key should be set as processingKey.
     * @return true if key is successfully created and false if this algorithm is not supported.
     * @throws ScriptFailException if errors occur.
     */
    public boolean createPrivateKey(short keyLength)
        throws CardProxyException {
        return createKey(CREATE_PRIVATE_KEY, keyLength);
    }
    
    /**
     * sends request on public key creation with given key length.
     * This request sets created key as 
     * @return true if key is successfully created or false if this algorithm is not supported.
     * @throws ScriptFailException if errors occur.
     */
    public boolean createPublicKey(short keyLength)
        throws CardProxyException {
        return createKey(CREATE_PUBLIC_KEY, keyLength);
    }


   /**
     * encodes externalAccess into byte array.
     */
    private byte[] encodeObjCreateData(byte algorithm, boolean externalAccess) {
        return new byte[] {
            algorithm, 
            (byte)(externalAccess ? 1 : 0)
        };
    }

    private boolean createObj(byte step, byte algorithm, boolean external_access)
        throws CardProxyException {
        CommandAPDU apdu = new CommandAPDU((byte)0x80, (byte)0x20, step,
                                           (byte)0,
                                           encodeObjCreateData(algorithm, external_access),
                                           (byte)0x7f); 
        ResponseAPDU answer = sendAPDU(apdu);
        int sw = answer.sw();
        if (((sw & 0xFFFF) != 0x9b00) && ((sw & 0xFFFF) != 0x9b01)) {
            log(I18n.getString("ecscript.incorrect.response", new Integer(this.apduCount), Integer.toHexString(0xFFFF & sw)));
            throw new ScriptFailException(I18n.getString("ecscript.exception", new Integer(this.apduCount)));
        }
        return ((sw & 0xFFFF) == 0x9b00);
    }


   /**
     * sends request on signature object creation with give algorithm and exter_access.
     * @return true if signature is successfully created and false if this algorithm is not supported.
     * @throws ScriptFailException if errors occur.
     */
    public boolean createSignature(byte algorithm, boolean external_access)
        throws CardProxyException {
        return createObj(CREATE_SIG, algorithm, external_access);
    }
    

    /**
     * sends request on keyagreement object creation with given algorithm and externalAccess 
     * @return true if keyagreement object is successfully created or false if this algorithm is not supported.
     * @throws ScriptFailException if errors occur.
     */
    public boolean createKeyAgreement(byte algorithm, boolean external_access)
        throws CardProxyException {
        return createObj(CREATE_KA, algorithm, external_access);
    }


    /**
     * create two instance of KeyAgreement
     */
    public boolean createKeyAgreement2(byte algorithm, boolean external_access)
        throws CardProxyException {
        return createObj(CREATE_KA2, algorithm, external_access);
    }
    /**
     * set domain parameters in processingKey. This methods do nothing If processingKey is null.
     */
    public void setECDomainParameters(byte set, short length) throws CardProxyException {
  
        KeyData current = getKeyData(set, length);

        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_P, 
                                 (byte)0, current.P, (byte)0x7f), 
                 (short)0x9b00);
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_A,
                                 (byte)0, current.A, (byte)0x7f),
                 (short)0x9b00);
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_B, 
                                 (byte)0, current.B, (byte)0x7f), 
                 (short)0x9b00);
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_G,
                                 (byte)0, current.G, (byte)0x7f),
                 (short)0x9b00);
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_R, 
                                 (byte)0, current.R, (byte)0x7f), 
                 (short)0x9b00);
    }


    /**
     * set cofactor
     */
    public void setCofactor(byte set, short length)
        throws CardProxyException {
        KeyData current = getKeyData(set, length);
        
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_K, 
                                 (byte)0, current.K, (byte)0x7f), 
                 (short)0x9b00);            
    }  

    /**
     * set specific private key data in public key.
     */
    public void setPrivateKeyData(byte set, short length)
        throws CardProxyException {
        KeyData current = getKeyData(set, length);
        
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_S, 
                                 (byte)0, current.S, (byte)0x7f), 
                 (short)0x9b00);            
    }

    /**
     * set specific public key data in public key.
     */
    public void setPublicKeyData(byte set, short length)
        throws CardProxyException {
        KeyData current = getKeyData(set, length);
        
        sendAPDU(new CommandAPDU((byte)0x80, (byte)0x20, SET_W, 
                                 (byte)0, current.W, (byte)0x7f), 
                 (short)0x9b00);            
    }

    /**
     * sets private key as processingKey.
     */
    public void setPrivateKey() throws CardProxyException {
        sendAPDU(new CommandAPDU(new byte[] {
            (byte)0x80, (byte)0x20, SET_PRIVATE_KEY, (byte)0}),
                 (short)0x9b00);
    }
    
    /**
     * sets public key as processingKey.
     */
    public void setPublicKey() throws CardProxyException {
        sendAPDU(new CommandAPDU(new byte[] {
            (byte)0x80, (byte)0x20, SET_PUBLIC_KEY, (byte)0}),
                 (short)0x9b00);
    }
    
    private static HashMap keydata; 


    // KeyData initialization
    static {
        keydata = new HashMap();
        keydata.put("1_112",
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x66, 
                            (byte) 0x80, (byte) 0x76, (byte) 0xbe, (byte) 0xad, 
                            (byte) 0x20, (byte) 0x8b
                        }, 
                        new byte[] { //A
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x66, 
                            (byte) 0x80, (byte) 0x76, (byte) 0xbe, (byte) 0xad, 
                            (byte) 0x20, (byte) 0x88
                        },
                        new byte[] { //B
                            (byte) 0x65, (byte) 0x9e, (byte) 0xf8, (byte) 0xba, 
                            (byte) 0x04, (byte) 0x39, (byte) 0x16, (byte) 0xee, 
                            (byte) 0xde, (byte) 0x89, (byte) 0x11, (byte) 0x70, 
                            (byte) 0x2b, (byte) 0x22
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x09, (byte) 0x48, (byte) 0x72, (byte) 0x39, 
                            (byte) 0x99, (byte) 0x5a, (byte) 0x5e, (byte) 0xe7, 
                            (byte) 0x6b, (byte) 0x55, (byte) 0xf9, (byte) 0xc2, 
                            (byte) 0xf0, (byte) 0x98,
                            (byte) 0xa8, (byte) 0x9c, (byte) 0xe5, (byte) 0xaf, 
                            (byte) 0x87, (byte) 0x24, (byte) 0xc0, (byte) 0xa2, 
                            (byte) 0x3e, (byte) 0x0e, (byte) 0x0f, (byte) 0xf7, 
                            (byte) 0x75, (byte) 0x00
                        }, 
                        new byte[] { // R
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x76, 
                            (byte) 0x28, (byte) 0xdf, (byte) 0xac, (byte) 0x65, 
                            (byte) 0x61, (byte) 0xc5   
                        },
			new byte[] {//K
                            (byte)0x00, (byte)0x01
			},
                        new byte[] { // S (private key data)
                            (byte) 0x59, (byte) 0x5d, (byte) 0xa0, (byte) 0x5e, 
                            (byte) 0x61, (byte) 0x8d, (byte) 0xa5, (byte) 0xa6, 
                            (byte) 0x64, (byte) 0xef, (byte) 0x6a, (byte) 0x93, 
                            (byte) 0x12, (byte) 0x72
                        },
                        new byte[] { // W (public key data)
                            (byte) 0x04,
                            (byte) 0xbd, (byte) 0x63, (byte) 0x3f, (byte) 0x84, 
                            (byte) 0x86, (byte) 0x74, (byte) 0x28, (byte) 0x9e, 
                            (byte) 0xac, (byte) 0xbd, (byte) 0xb6, (byte) 0xa3, 
                            (byte) 0xcf, (byte) 0x28,
                            (byte) 0x74, (byte) 0xf5, (byte) 0x67, (byte) 0xe0, 
                            (byte) 0xb2, (byte) 0xce, (byte) 0x3e, (byte) 0xc0, 
                            (byte) 0xb9, (byte) 0x37, (byte) 0x72, (byte) 0x92, 
                            (byte) 0xa9, (byte) 0x54
                        }));
        keydata.put("2_112", 
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x66, 
                            (byte) 0x80, (byte) 0x76, (byte) 0xbe, (byte) 0xad, 
                            (byte) 0x20, (byte) 0x8b
                        }, 
                        new byte[] { //A
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x66, 
                            (byte) 0x80, (byte) 0x76, (byte) 0xbe, (byte) 0xad, 
                            (byte) 0x20, (byte) 0x88
                        },
                        new byte[] { //B
                            (byte) 0x65, (byte) 0x9e, (byte) 0xf8, (byte) 0xba, 
                            (byte) 0x04, (byte) 0x39, (byte) 0x16, (byte) 0xee, 
                            (byte) 0xde, (byte) 0x89, (byte) 0x11, (byte) 0x70, 
                            (byte) 0x2b, (byte) 0x22
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x09, (byte) 0x48, (byte) 0x72, (byte) 0x39, 
                            (byte) 0x99, (byte) 0x5a, (byte) 0x5e, (byte) 0xe7, 
                            (byte) 0x6b, (byte) 0x55, (byte) 0xf9, (byte) 0xc2, 
                            (byte) 0xf0, (byte) 0x98,
                            (byte) 0xa8, (byte) 0x9c, (byte) 0xe5, (byte) 0xaf, 
                            (byte) 0x87, (byte) 0x24, (byte) 0xc0, (byte) 0xa2, 
                            (byte) 0x3e, (byte) 0x0e, (byte) 0x0f, (byte) 0xf7, 
                            (byte) 0x75, (byte) 0x00
                        }, 
                        new byte[] { // R
                            (byte) 0xdb, (byte) 0x7c, (byte) 0x2a, (byte) 0xbf, 
                            (byte) 0x62, (byte) 0xe3, (byte) 0x5e, (byte) 0x76, 
                            (byte) 0x28, (byte) 0xdf, (byte) 0xac, (byte) 0x65, 
                            (byte) 0x61, (byte) 0xc5   
                        },
			new byte[] {//K
                            (byte)0x00, (byte)0x01
			},
                        new byte[] { // S (private key data)
                            (byte)0xcb, (byte)0x6b, (byte)0x1f, (byte)0x4e, 
                            (byte)0x44, (byte)0xf0, (byte)0x69, (byte)0x7d, 
                            (byte)0x1a, (byte)0xdb, (byte)0x5b, (byte)0x26,
                            (byte)0x0e, (byte)0x9d   
                        },
                        new byte[] { // W (public key data)
                           (byte)0x04,
                           (byte)0x2b, (byte)0x50, (byte)0x8b, (byte)0x48, 
                           (byte)0x41, (byte)0x1e, (byte)0x8a, (byte)0xa7,
                           (byte)0x5a, (byte)0xb6, (byte)0x32, (byte)0xdb,
                           (byte)0xf9, (byte)0x46,
                           (byte)0xd2, (byte)0xe0, (byte)0x82, (byte)0x30,
                           (byte)0xd6, (byte)0xb0, (byte)0xa8, (byte)0x3d,
                           (byte)0x29, (byte)0xa7, (byte)0xad, (byte)0xb5,
                           (byte)0x9f, (byte)0x36  
                        }));
        keydata.put("1_128",
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfd, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff
                        },
                        new byte[] { //A
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfd, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfc
                        },
                        new byte[] { //B
                            (byte) 0xe8, (byte) 0x75, (byte) 0x79, (byte) 0xc1, 
                            (byte) 0x10, (byte) 0x79, (byte) 0xf4, (byte) 0x3d, 
                            (byte) 0xd8, (byte) 0x24, (byte) 0x99, (byte) 0x3c, 
                            (byte) 0x2c, (byte) 0xee, (byte) 0x5e, (byte) 0xd3
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x16, (byte) 0x1f, (byte) 0xf7, (byte) 0x52, 
                            (byte) 0x8b, (byte) 0x89, (byte) 0x9b, (byte) 0x2d, 
                            (byte) 0x0c, (byte) 0x28, (byte) 0x60, (byte) 0x7c, 
                            (byte) 0xa5, (byte) 0x2c, (byte) 0x5b, (byte) 0x86,
                            (byte) 0xcf, (byte) 0x5a, (byte) 0xc8, (byte) 0x39, 
                            (byte) 0x5b, (byte) 0xaf, (byte) 0xeb, (byte) 0x13, 
                            (byte) 0xc0, (byte) 0x2d, (byte) 0xa2, (byte) 0x92, 
                            (byte) 0xdd, (byte) 0xed, (byte) 0x7a, (byte) 0x83
                        },
                        new byte[] { // R
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x75, (byte) 0xa3, (byte) 0x0d, (byte) 0x1b, 
                            (byte) 0x90, (byte) 0x38, (byte) 0xa1, (byte) 0x15
                        },
			new byte[] { //K
                            (byte)0x00, (byte)0x01
			},
                        new byte[] {  // S (private key data)
                            (byte) 0x01, (byte) 0x0f, (byte) 0x9e, (byte) 0x7f,
                            (byte) 0xd7, (byte) 0x59, (byte) 0x65, (byte) 0xbb,
                            (byte) 0xfc, (byte) 0x53, (byte) 0x92, (byte) 0x8a,                
                            (byte) 0x3c, (byte) 0xf6, (byte) 0x77, (byte) 0x05
                        },
                        new byte[] { // W (public key data)
                            (byte) 0x04,
                            (byte) 0xfa, (byte) 0xc3, (byte) 0xa5, (byte) 0xbf, 
                            (byte) 0xe3, (byte) 0xa6, (byte) 0x79, (byte) 0xf8, 
                            (byte) 0xf4, (byte) 0x9a, (byte) 0x66, (byte) 0x1a, 
                            (byte) 0xee, (byte) 0x73, (byte) 0x8d, (byte) 0x42,
                            (byte) 0x5a, (byte) 0x68, (byte) 0xac, (byte) 0x8d, 
                            (byte) 0x93, (byte) 0x6a, (byte) 0xb9, (byte) 0x5d, 
                            (byte) 0x10, (byte) 0xcb, (byte) 0x26, (byte) 0x91, 
                            (byte) 0x3d, (byte) 0x83, (byte) 0xd1, (byte) 0x35
                      }));
        keydata.put("2_128",
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfd, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff
                        },
                        new byte[] { //A
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfd, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfc
                        },
                        new byte[] { //B
                            (byte) 0xe8, (byte) 0x75, (byte) 0x79, (byte) 0xc1, 
                            (byte) 0x10, (byte) 0x79, (byte) 0xf4, (byte) 0x3d, 
                            (byte) 0xd8, (byte) 0x24, (byte) 0x99, (byte) 0x3c, 
                            (byte) 0x2c, (byte) 0xee, (byte) 0x5e, (byte) 0xd3
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x16, (byte) 0x1f, (byte) 0xf7, (byte) 0x52, 
                            (byte) 0x8b, (byte) 0x89, (byte) 0x9b, (byte) 0x2d, 
                            (byte) 0x0c, (byte) 0x28, (byte) 0x60, (byte) 0x7c, 
                            (byte) 0xa5, (byte) 0x2c, (byte) 0x5b, (byte) 0x86,
                            (byte) 0xcf, (byte) 0x5a, (byte) 0xc8, (byte) 0x39, 
                            (byte) 0x5b, (byte) 0xaf, (byte) 0xeb, (byte) 0x13, 
                            (byte) 0xc0, (byte) 0x2d, (byte) 0xa2, (byte) 0x92, 
                            (byte) 0xdd, (byte) 0xed, (byte) 0x7a, (byte) 0x83
                        },
                        new byte[] { // R
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x75, (byte) 0xa3, (byte) 0x0d, (byte) 0x1b, 
                            (byte) 0x90, (byte) 0x38, (byte) 0xa1, (byte) 0x15
                        },
                        new byte[] { //K
                            (byte)0x00, (byte)0x01
			},
                        new byte[] { // S (private key data)
                            (byte)0xcb, (byte)0x6b, (byte)0x1f, (byte)0x4e,
                            (byte)0x44, (byte)0xf0, (byte)0x69, (byte)0x7d,
                            (byte)0x1a, (byte)0xdb, (byte)0x5b, (byte)0x26,
                            (byte)0x0e, (byte)0x9d, (byte)0x1c, (byte)0x16     
                        },
                        new byte[] { // W (public key data)
                            (byte)0x04, 
                            (byte)0xab, (byte)0xd9, (byte)0xbb, (byte)0xf2, 
                            (byte)0x10, (byte)0xfc, (byte)0xd5, (byte)0x13, 
                            (byte)0x13, (byte)0x50, (byte)0x38, (byte)0xc4, 
                            (byte)0xb4, (byte)0x2c, (byte)0x8e, (byte)0x1d, 
                            (byte)0xfc, (byte)0xe0, (byte)0x1b, (byte)0xde, 
                            (byte)0xa2, (byte)0x7f, (byte)0x58, (byte)0xad, 
                            (byte)0xc7, (byte)0xe6, (byte)0xde, (byte)0x8b, 
                            (byte)0x40, (byte)0x17, (byte)0xae, (byte)0x9e 
                        }));
        keydata.put("1_160",
                    new KeyData(
                        new byte[] {  //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xac, (byte) 0x73
                        },
                        new byte[] { //A
                            (byte) 0x00
                        },
                        new byte[] { //B
                            (byte) 0x07
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x3b, (byte) 0x4c, (byte) 0x38, (byte) 0x2c, 
                            (byte) 0xe3, (byte) 0x7a, (byte) 0xa1, (byte) 0x92, 
                            (byte) 0xa4, (byte) 0x01, (byte) 0x9e, (byte) 0x76, 
                            (byte) 0x30, (byte) 0x36, (byte) 0xf4, (byte) 0xf5, 
                            (byte) 0xdd, (byte) 0x4d, (byte) 0x7e, (byte) 0xbb,
                            (byte) 0x93, (byte) 0x8c, (byte) 0xf9, (byte) 0x35, 
                            (byte) 0x31, (byte) 0x8f, (byte) 0xdc, (byte) 0xed, 
                            (byte) 0x6b, (byte) 0xc2, (byte) 0x82, (byte) 0x86, 
                            (byte) 0x53, (byte) 0x17, (byte) 0x33, (byte) 0xc3, 
                            (byte) 0xf0, (byte) 0x3c, (byte) 0x4f, (byte) 0xee
                        },
                        new byte[] { // R
                            (byte) 0x01, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x01, (byte) 0xb8, 
                            (byte) 0xfa, (byte) 0x16, (byte) 0xdf, (byte) 0xab, 
                            (byte) 0x9a, (byte) 0xca, (byte) 0x16, (byte) 0xb6, 
                            (byte) 0xb3
                        },
                        new byte[] { //K
                            (byte)0x00, (byte)0x01
                        },
                        new byte[] { // S (private key data)
                            (byte) 0x5a, (byte) 0xbf, (byte) 0xcd, (byte) 0x75,
                            (byte) 0xeb, (byte) 0x2f, (byte) 0xb4, (byte) 0xb8,
                            (byte) 0x7a, (byte) 0x6b, (byte) 0x81, (byte) 0x72,
                            (byte) 0x59, (byte) 0x72, (byte) 0x58, (byte) 0x98,
                            (byte) 0x6e, (byte) 0x37, (byte) 0x6d, (byte) 0xd4
                        },
                        new byte[] { // W (public key data)
                            (byte) 0x04,
                            (byte) 0x25, (byte) 0x1b, (byte) 0x22, (byte) 0x8d, 
                            (byte) 0x4b, (byte) 0x11, (byte) 0xaf, (byte) 0xf3, 
                            (byte) 0x24, (byte) 0xf8, (byte) 0x6f, (byte) 0x02, 
                            (byte) 0xf2, (byte) 0x05, (byte) 0x43, (byte) 0xb2, 
                            (byte) 0x78, (byte) 0xfd, (byte) 0x8a, (byte) 0xa0,
                            (byte) 0xac, (byte) 0x96, (byte) 0xf9, (byte) 0x27, 
                            (byte) 0x3b, (byte) 0xa3, (byte) 0x5a, (byte) 0x1e, 
                            (byte) 0x5d, (byte) 0xc2, (byte) 0xdc, (byte) 0xce, 
                            (byte) 0xf3, (byte) 0xd0, (byte) 0x5a, (byte) 0x8b, 
                            (byte) 0x9f, (byte) 0x2d, (byte) 0x71, (byte) 0xa4
                        }));
        keydata.put("2_160",
                    new KeyData(
                        new byte[] {  //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xac, (byte) 0x73
                        },
                        new byte[] { //A
                            (byte) 0x00
                        },
                        new byte[] { //B
                            (byte) 0x07
                        },
                        new byte[] { //G
                            (byte) 0x04,
                            (byte) 0x3b, (byte) 0x4c, (byte) 0x38, (byte) 0x2c, 
                            (byte) 0xe3, (byte) 0x7a, (byte) 0xa1, (byte) 0x92, 
                            (byte) 0xa4, (byte) 0x01, (byte) 0x9e, (byte) 0x76, 
                            (byte) 0x30, (byte) 0x36, (byte) 0xf4, (byte) 0xf5, 
                            (byte) 0xdd, (byte) 0x4d, (byte) 0x7e, (byte) 0xbb,
                            (byte) 0x93, (byte) 0x8c, (byte) 0xf9, (byte) 0x35, 
                            (byte) 0x31, (byte) 0x8f, (byte) 0xdc, (byte) 0xed, 
                            (byte) 0x6b, (byte) 0xc2, (byte) 0x82, (byte) 0x86, 
                            (byte) 0x53, (byte) 0x17, (byte) 0x33, (byte) 0xc3, 
                            (byte) 0xf0, (byte) 0x3c, (byte) 0x4f, (byte) 0xee
                        },
                        new byte[] { // R
                            (byte) 0x01, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00, 
                            (byte) 0x00, (byte) 0x00, (byte) 0x01, (byte) 0xb8, 
                            (byte) 0xfa, (byte) 0x16, (byte) 0xdf, (byte) 0xab, 
                            (byte) 0x9a, (byte) 0xca, (byte) 0x16, (byte) 0xb6, 
                            (byte) 0xb3
                        },
                        new byte[] { //K
                            (byte)0x00, (byte)0x01
                        },
                        new byte[] { // S (private key data)
                            (byte)0xcb, (byte)0x6b, (byte)0x1f, (byte)0x4e, 
                            (byte)0x44, (byte)0xf0, (byte)0x69, (byte)0x7d, 
                            (byte)0x1a, (byte)0xdb, (byte)0x5b, (byte)0x26, 
                            (byte)0x0e, (byte)0x9d, (byte)0x1c, (byte)0x16, 
                            (byte)0x51, (byte)0xd8, (byte)0x5d, (byte)0x5a 
                        },
                        new byte[] { // W (public key data)
                           (byte)0x04,
                           (byte)0xaf, (byte)0x90, (byte)0xf6, (byte)0xb6,
                           (byte)0x50, (byte)0x7b, (byte)0xd5, (byte)0x5c,
                           (byte)0xbe, (byte)0x14, (byte)0x82, (byte)0x9a,
                           (byte)0x2a, (byte)0xd5, (byte)0xf8, (byte)0xf6,
                           (byte)0x52, (byte)0x2b, (byte)0x57, (byte)0x06,
                           (byte)0x97, (byte)0x00, (byte)0x55, (byte)0xd7,
                           (byte)0x72, (byte)0x8a, (byte)0xfd, (byte)0xf5,
                           (byte)0xc3, (byte)0xdb, (byte)0x9e, (byte)0x99,
                           (byte)0x1d, (byte)0x14, (byte)0xe3, (byte)0x04,
                           (byte)0xe9, (byte)0x8a, (byte)0x9f, (byte)0xb2
                        }));
        keydata.put("1_192",
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff
                        },
                        new byte[] { //A
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfc
                        },
                        new byte[] {//B 
                            (byte) 0x64, (byte) 0x21, (byte) 0x05, (byte) 0x19, 
                            (byte) 0xe5, (byte) 0x9c, (byte) 0x80, (byte) 0xe7, 
                            (byte) 0x0f, (byte) 0xa7, (byte) 0xe9, (byte) 0xab, 
                            (byte) 0x72, (byte) 0x24, (byte) 0x30, (byte) 0x49, 
                            (byte) 0xfe, (byte) 0xb8, (byte) 0xde, (byte) 0xec, 
                            (byte) 0xc1, (byte) 0x46, (byte) 0xb9, (byte) 0xb1
                        },
                        new byte[] {//G
                            (byte) 0x04,
                            (byte) 0x18, (byte) 0x8d, (byte) 0xa8, (byte) 0x0e, 
                            (byte) 0xb0, (byte) 0x30, (byte) 0x90, (byte) 0xf6, 
                            (byte) 0x7c, (byte) 0xbf, (byte) 0x20, (byte) 0xeb, 
                            (byte) 0x43, (byte) 0xa1, (byte) 0x88, (byte) 0x00, 
                            (byte) 0xf4, (byte) 0xff, (byte) 0x0a, (byte) 0xfd, 
                            (byte) 0x82, (byte) 0xff, (byte) 0x10, (byte) 0x12,
                            (byte) 0x07, (byte) 0x19, (byte) 0x2b, (byte) 0x95, 
                            (byte) 0xff, (byte) 0xc8, (byte) 0xda, (byte) 0x78, 
                            (byte) 0x63, (byte) 0x10, (byte) 0x11, (byte) 0xed, 
                            (byte) 0x6b, (byte) 0x24, (byte) 0xcd, (byte) 0xd5, 
                            (byte) 0x73, (byte) 0xf9, (byte) 0x77, (byte) 0xa1, 
                            (byte) 0x1e, (byte) 0x79, (byte) 0x48, (byte) 0x11
                        },
                        new byte[] {//R
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0x99, (byte) 0xde, (byte) 0xf8, (byte) 0x36, 
                            (byte) 0x14, (byte) 0x6b, (byte) 0xc9, (byte) 0xb1, 
                            (byte) 0xb4, (byte) 0xd2, (byte) 0x28, (byte) 0x31
                        },
			new byte[] { //K
                            (byte)0x00, (byte)0x01
                        },
                        new byte[] {  // S (private key data)
                           (byte) 0x1a, (byte) 0x8d, (byte) 0x59, (byte) 0x8f, 
                           (byte) 0xc1, (byte) 0x5b, (byte) 0xf0, (byte) 0xfd, 
                           (byte) 0x89, (byte) 0x03, (byte) 0x0b, (byte) 0x5c, 
                           (byte) 0xb1, (byte) 0x11, (byte) 0x1a, (byte) 0xeb, 
                           (byte) 0x92, (byte) 0xae, (byte) 0x8b, (byte) 0xaf, 
                           (byte) 0x5e, (byte) 0xa4, (byte) 0x75, (byte) 0xfb  
                        },
                        new byte[] { // W (public key data)
                            (byte) 0x04,
                            (byte)0x62, (byte)0xb1, (byte)0x2d, (byte)0x60, 
                            (byte)0x69, (byte)0x0c, (byte)0xdc, (byte)0xf3, 
                            (byte)0x30, (byte)0xba, (byte)0xba, (byte)0xb6, 
                            (byte)0xe6, (byte)0x97, (byte)0x63, (byte)0xb4, 
                            (byte)0x71, (byte)0xf9, (byte)0x94, (byte)0xdd, 
                            (byte)0x70, (byte)0x2d, (byte)0x16, (byte)0xa5, 
                            (byte)0x63, (byte)0xbf, (byte)0x5e, (byte)0xc0, 
                            (byte)0x80, (byte)0x69, (byte)0x70, (byte)0x5f, 
                            (byte)0xff, (byte)0xf6, (byte)0x5e, (byte)0x5c, 
                            (byte)0xa5, (byte)0xc0, (byte)0xd6, (byte)0x97, 
                            (byte)0x16, (byte)0xdf, (byte)0xcb, (byte)0x34, 
                            (byte)0x74, (byte)0x37, (byte)0x39, (byte)0x02
                        }));
        keydata.put("2_192",
                    new KeyData(
                        new byte[] { //P
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff
                        },
                        new byte[] { //A
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfe, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xfc
                        },
                        new byte[] {//B 
                            (byte) 0x64, (byte) 0x21, (byte) 0x05, (byte) 0x19, 
                            (byte) 0xe5, (byte) 0x9c, (byte) 0x80, (byte) 0xe7, 
                            (byte) 0x0f, (byte) 0xa7, (byte) 0xe9, (byte) 0xab, 
                            (byte) 0x72, (byte) 0x24, (byte) 0x30, (byte) 0x49, 
                            (byte) 0xfe, (byte) 0xb8, (byte) 0xde, (byte) 0xec, 
                            (byte) 0xc1, (byte) 0x46, (byte) 0xb9, (byte) 0xb1
                        },
                        new byte[] {//G
                            (byte) 0x04,
                            (byte) 0x18, (byte) 0x8d, (byte) 0xa8, (byte) 0x0e, 
                            (byte) 0xb0, (byte) 0x30, (byte) 0x90, (byte) 0xf6, 
                            (byte) 0x7c, (byte) 0xbf, (byte) 0x20, (byte) 0xeb, 
                            (byte) 0x43, (byte) 0xa1, (byte) 0x88, (byte) 0x00, 
                            (byte) 0xf4, (byte) 0xff, (byte) 0x0a, (byte) 0xfd, 
                            (byte) 0x82, (byte) 0xff, (byte) 0x10, (byte) 0x12,
                            (byte) 0x07, (byte) 0x19, (byte) 0x2b, (byte) 0x95, 
                            (byte) 0xff, (byte) 0xc8, (byte) 0xda, (byte) 0x78, 
                            (byte) 0x63, (byte) 0x10, (byte) 0x11, (byte) 0xed, 
                            (byte) 0x6b, (byte) 0x24, (byte) 0xcd, (byte) 0xd5, 
                            (byte) 0x73, (byte) 0xf9, (byte) 0x77, (byte) 0xa1, 
                            (byte) 0x1e, (byte) 0x79, (byte) 0x48, (byte) 0x11
                        },
                        new byte[] {//R
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0xff, (byte) 0xff, (byte) 0xff, (byte) 0xff, 
                            (byte) 0x99, (byte) 0xde, (byte) 0xf8, (byte) 0x36, 
                            (byte) 0x14, (byte) 0x6b, (byte) 0xc9, (byte) 0xb1, 
                            (byte) 0xb4, (byte) 0xd2, (byte) 0x28, (byte) 0x31
                        },
			new byte[] { //K
                            (byte)0x00, (byte)0x01
                        },
                        new byte[] { // S (private key data)
                           (byte) 0x1a, (byte) 0x72, (byte) 0x34, (byte) 0x0f, 
                           (byte) 0x11, (byte) 0x04, (byte) 0xa2, (byte) 0x34, 
                           (byte) 0x31, (byte) 0x03, (byte) 0x1c, (byte) 0x06, 
                           (byte) 0xc1, (byte) 0x21, (byte) 0x06, (byte) 0x1c, 
                           (byte) 0x23, (byte) 0x35, (byte) 0x6b, (byte) 0xcd, 
                           (byte) 0x50, (byte) 0x14, (byte) 0x25, (byte) 0x0b  
                        },
                        new byte[] { // W (public key data)
                            (byte) 0x04,
                            (byte) 0x1b, (byte) 0xfe, (byte) 0x89, (byte) 0x05, 
                            (byte) 0x5d, (byte) 0x98, (byte) 0x2e, (byte) 0x7d, 
                            (byte) 0xbf, (byte) 0x07, (byte) 0x23, (byte) 0xfd, 
                            (byte) 0x43, (byte) 0x94, (byte) 0x82, (byte) 0x08, 
                            (byte) 0xc2, (byte) 0xb7, (byte) 0xb6, (byte) 0x24, 
                            (byte) 0x39, (byte) 0xe4, (byte) 0xff, (byte) 0xa6,
                            (byte) 0x34, (byte) 0x23, (byte) 0xe4, (byte) 0x59, 
                            (byte) 0x13, (byte) 0x51, (byte) 0x79, (byte) 0x14, 
                            (byte) 0xbd, (byte) 0xcc, (byte) 0x51, (byte) 0x81, 
                            (byte) 0x8e, (byte) 0x44, (byte) 0xf9, (byte) 0x1d, 
                            (byte) 0xbe, (byte) 0x56, (byte) 0x85, (byte) 0x63, 
                            (byte) 0xd8, (byte) 0xd5, (byte) 0x47, (byte) 0x27
                        }));
    }
}
