/*
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.scripts;

import com.sun.javacard.cjck.scripts.StdScript;
import com.sun.javacard.cjck.invoke.CustomizableScript;
import com.sun.javacard.cjck.invoke.ConfiguredScript;
import com.sun.javacard.cjck.invoke.CardProxyArguments;
import com.sun.javacard.cjck.scripts.ScriptFailException;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;

import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CommunicationService;
import com.sun.javacard.cjck.userinterface.CJCKCardService;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;

import com.sun.javatest.Status;

/**
 *  <tt>DualInterfaceScript</tt> is the base script for working with dual
 *  interface cards.
 *  <p>
 *  Test developer should extend this class and implements the abstract method
 *  <code>runTestCase(int index, String name)</code> (just like for
 *  MultiStdScript), and  (optionally) implement the default constructor which
 *  will call to the <code>DualInterfaceScript(String[] cases)</code>
 *  constructor with list of testcase IDs.
 *  <p>
 *  The <tt>RunMode</tt> will create an instance of this script and invoke its
 *  <code>CustomizableScript.setCardProxyArguments(...)</code>. Here the
 *  <tt>DualInterfaceScript</tt> gets the <tt>ContactlessCardService</tt>
 *  implementation class name and instantiates it. Then its
 *  <code>JavaCardScript.run(...)</code> method is invoked and the execution of
 *  the script begins.
 *  <p>
 *  During execution, the <code>runTestCase(int index, String name)</code>
 *  method is invoked for each not excluded testcase.
 *  <p>
 *  Note : When the <code>runTestCase(int index, String name)</code> is invoked
 *  only the contacted i/o interface is powered up, and no applet is selected
 *  on the basic logical channel. The implementor have to call 
 *  the <code>select(...)</code> and <code>contactlessCardService.powerUp()</code>
 *  methods if needed.
 *
 * @author Dmitri Trounine
 */
public abstract class DualInterfaceScript extends StdScript {
    /**
     * CLA byte: jcre test class
     */
    public final static byte CLA_JCRE_TEST = (byte)0x80;

    /**
     * INS byte: DoTests command
     */
    public final static byte INS_DO_TEST = (byte)0x20;
    
    public static final short SW_OK = (short) 0x9b00;
    
    public static final short SW_SELECT_PASSED  = (short)0x9000;

    protected CommunicationService contactedService;
    protected CommunicationService contactlessService;
    
    public DualInterfaceScript() {
        super();
    }
    
    protected void init(CJCKCardService cardService, PrintWriter out, PrintWriter ref) {
        super.init(cardService, out, ref);
	contactedService = cardService;
        try {
            contactlessService = cardService.getCommunicationService(CardService.CONTACTLESS_INTERFACE);
        } catch (CardProxyException e) {
            throw new ScriptFailException(I18n.getString("no.contactless.interface"));
        }
    }
    
    /**
     * sends APDU to the card over the contactless i/o interface and verifies
     * response. If the response does not equal to the expected response then
     * ScriptFailException will be thrown.
     *
     * @param   apdu    specifies APDU which will send to the card over the
     *                  contactless i/o interface
     * @param   expectedResponse    specifies expected response for verification
     * @throws  ScriptFailException if the response does not equals to
     *          expectedResponse or
     * @throws  CardProxyException  if CardProxyException is thrown by
     *          ContactlessCardService implementation. 
     */
    public void sendAPDUContactless(CommandAPDU apdu, short expectedResponse)
        throws CardProxyException {
        this.apduCount++;
        ResponseAPDU answer = sendAPDUContactless(apdu);
        int sw = answer.sw();
        if ((sw & 0xFFFF) != (0xFFFF & expectedResponse)) {
            logContactless(I18n.getString("incorrect.response", new Integer(this.apduCount), 
                               Integer.toHexString(0xFFFF & expectedResponse), 
                               Integer.toHexString(0xFFFF & sw)));
            throw new ScriptFailException(I18n.getString("incorrect.response.1", 
                                          new Integer(this.apduCount)));
        }
    }
    
    public ResponseAPDU sendAPDUContactless(CommandAPDU apdu)
            throws CardProxyException {
        this.apduCount++;
        ResponseAPDU answer = contactlessService.sendAPDU(apdu);
        int sw = answer.sw();
        if ((sw & 0xFF00) != 0x6100) {
            return answer;
        } else {
            // SW1=0x61 => SW2 encodes the number of bytes still available
            ArrayList list = new ArrayList();
            int length = 0; //0xFF & sw; 
            while (((sw & 0xFF00) == 0x6100)
                   || ((sw & 0xFF00) == 0x6200)) {
                list.add(answer);
                length += answer.getLe();
                answer = contactlessService.sendAPDU(
                        getGetResponseAPDUContactless(apdu.getCLA(), sw));
                sw = answer.sw();
            }
            list.add(answer);
            length += answer.getLe();
            byte[] data = new byte[length + 2];
            int offset = 0;
            for (Iterator e = list.iterator(); e.hasNext();) {
                answer = (ResponseAPDU)e.next();
                for (int i = 0; i < answer.getLe(); i++) {
                    data[offset++] = (byte)answer.getData(i); 
                }
            }
            data[offset++] = (byte)(sw >> 8);
            data[offset++] = (byte)sw;
            return new ResponseAPDU(data);
        }
    }
    
    /**
     * selects applet with its AID over the contacted I/O interface and
     * verifies status.
     * 
     * @param   applet    AppletProperties instance of the applet beeing selected.
     * @param   expected    expected status of the response.
     * @throws  ScriptFailException     if the selection APDU returns SW which
     *          is not equal to <code>expected</code>.
     * @throws  IllegalArgumentException    if the name resolving fails.
     */
    public void select(AppletProperties applet, short expected) throws CardProxyException {
        log(I18n.getString("start.select.applet", applet.getClassName(), Integer.toHexString(expected)));
        sendAPDU(new CommandAPDU((byte)0x00, (byte)0xA4, (byte)0x04, (byte)0x00,
                                 applet.getAID().getBytes(), DEFAULT_LE),
                 expected);
    }
    
    /**
     * selects applet with its name over the contactless i/o interface and
     * verifies status.
     * 
     * @param   name    specifies full-qualified or local name of the applet
     *                  used for selection.
     * @param   expected    expected status of the response.
     * @throws  ScriptFailException     if the selection APDU returns SW which
     *          is not equal to <code>expected</code>.
     * @throws  IllegalArgumentException    if the name resolving fails.
     */
    public void selectContactless(String name, short expected)
        throws CardProxyException {
        selectContactless(findApplet(name), expected);
    }
    
    /**
     * Selects applet with its name over the contactless interface.
     * 
     * @param   name    specifies full-qualified or local name of the applet
     *                  used for selection.
     * @throws  ScriptFailException     if the selection APDU returns SW which
     *          is not equal to <code>0x9000</code>.
     * @throws  IllegalArgumentException    if the name resolving fails.
     */
    public void selectContactless(String name) throws CardProxyException {
        logContactless(I18n.getString("try.select.applet", name));
        selectContactless(findApplet(name), (short)0x9b00);
    }
    
    /**
     * selects applet with its AID over the contacted I/O interface and
     * verifies status.
     * 
     * @param   applet    AppletProperties instance of the applet beeing selected.
     * @param   expected    expected status of the response.
     * @throws  ScriptFailException     if the selection APDU returns SW which
     *          is not equal to <code>expected</code>.
     * @throws  IllegalArgumentException    if the name resolving fails.
     */
    public void selectContactless(AppletProperties applet, short expected)
        throws CardProxyException {
        logContactless(I18n.getString("start.select.applet", applet.getClassName(), Integer.toHexString(expected)));
        sendAPDUContactless(new CommandAPDU((byte)0x00, (byte)0xA4, (byte)0x04,
                (byte)0x00, applet.getAID().getBytes(), DEFAULT_LE),
                 expected);
    }
    
    
    protected static byte GET_RESPONSE_CLA_BYTE_CONTACTLESS = (byte) 0x00;
    
    public void setCardProxyArguments(CardProxyArguments arguments) {
        this.args = arguments;
        this.GET_RESPONSE_CLA_BYTE_CONTACTLESS =
                arguments.getGET_RESPONSE_CLA_BYTE();
    }
    
    private CommandAPDU getGetResponseAPDUContactless(byte original_cla,
            int sw) {
        return new CommandAPDU(new byte[] {
                (byte) ((GET_RESPONSE_CLA_BYTE_CONTACTLESS & 0xF0) 
                        | (original_cla & 0x0F)),
                (byte) 0xc0,
                (byte) 0x00, (byte) 0x00,
                (byte) sw});
    }
    
    protected void logContactless(Object o) {
        out.print("CONTACTLESS: ");
        super.log(o);
    }
    
    /**
     *	Send a DoTest command APDU for specified testcase number over the Contacted I/O interface.
     */
    public void sendDoTest(int testCase) throws CardProxyException {
	sendDoTest(testCase, (byte) 0);
    }

    /**
     *	Send a DoTest command APDU for specified testcase number on the specified logical channel
     *  over the Contacted I/O interface.
     */
    public void sendDoTest(int testCase, byte channel) throws CardProxyException {
	sendDoTest(testCase, channel, SW_OK);
    }
    
    /**
     *  Send a DoTest command APDU for specified testcase number on the specified logical channel
     *  over the Contacted I/O interface.
     *
     *	@param	expected    Expected response SW.
     *	@exception  ScriptFailexception	if returned SW differs from expected SW.
     */
    public void sendDoTest(int testCase, byte channel, short expected) throws CardProxyException {
	sendAPDU(getDoTestAPDU(testCase, channel), expected);
    }

    /**
     *	Send a DoTest command APDU for specified testcase number over the Contactless I/O interface.
     */
    public void sendDoTestContactless(int testCase) throws CardProxyException {
	sendDoTestContactless(testCase, (byte) 0);
    }

    /**
     *  Send a DoTest command APDU for specified testcase number on the specified logical channel
     *  over the Contactless I/O interface.
     *
     *	@param	expected    Expected response SW.
     *	@exception  ScriptFailexception	if returned SW differs from expected SW.
     */
    public void sendDoTestContactless(int testCase, byte channel, short expected) throws CardProxyException {
	sendAPDUContactless(getDoTestAPDU(testCase, channel), expected);
    }
    /**
     *	Send a DoTest command APDU for specified testcase number on the specified logical channel
     *  over the Contactless I/O interface.
     */
    public void sendDoTestContactless(int testCase, byte channel) throws CardProxyException {
	sendDoTestContactless(testCase, channel, SW_OK);
    }

    /** 
     * Get the DoTest command APDU for specified testcase number
     */
    public CommandAPDU getDoTestAPDU(int testCase) {
	return getDoTestAPDU(testCase, (byte) 0);
    }
    
    /** 
     * Get the DoTest command APDU for specified testcase number and logical channel.
     */
    public CommandAPDU getDoTestAPDU(int testCase, byte channel) {
	byte cla = (byte) (getChannelCLA(channel) | CLA_JCRE_TEST);
	return new CommandAPDU(cla, INS_DO_TEST, (byte) 0, (byte) 0,
		    new byte[] { (byte) testCase }, (byte) 0x7f);
    }

    /**
     * Issues a SELECT FILE command to the specified logical channel over the contacted I/O interface
     */
    public void select(AppletProperties applet, byte channel, short expectedSW) throws CardProxyException {
	log(I18n.getString("start.select.applet", applet.getClassName(), Integer.toHexString(expectedSW)));
        sendAPDU(getSelectAPDU(applet, channel), expectedSW);
    }
    
    /**
     * Issues a SELECT FILE command to the specified logical channel over the contacted I/O interface
     */
    public void selectContactless(AppletProperties applet, byte channel, short expectedSW)
	    throws CardProxyException {
	log(I18n.getString("start.select.applet", applet.getClassName(), Integer.toHexString(expectedSW)));
        sendAPDUContactless(getSelectAPDU(applet, channel), expectedSW);
    }

    /**
     * Get the CLA byte for SELECT FILE command for requested channel.
     */
    public byte getChannelCLA(byte channel) {
	if (channel < (byte) 0) {
	    throw new IllegalArgumentException(I18n.getString("negative.channel.exception"));
	}
	if (channel > (byte) 19) {
	    throw new IllegalArgumentException(I18n.getString("notsupported.channel.number"));
	}
	if (channel < (byte) 4) {
	    return channel;
	}
	return (byte) ((byte) 0x40 & (byte) (channel - 4));
    }
    
    /**
     *	Get the SELECT FILE command for requested applet and logical channel.
     */
    public CommandAPDU getSelectAPDU(AppletProperties applet, byte channel) {
	return new CommandAPDU(getChannelCLA(channel), (byte) 0xA4, (byte) 0x04, (byte) 0x00,
		applet.getAID().getBytes(), DEFAULT_LE);
    }
    
    /**
     * Issue the MANAGE CHANNEL OPEN to the requested logical channel over the contacted
     * I/O interface.
     *
     * @param from  origin logical channel.
     * @param to    number of the logical channel to be opened.
     */
    public void manageChannelClose(byte from, byte to) throws CardProxyException {
        log(I18n.getString("start.managechannelclose", Integer.toHexString(from),
                           Integer.toHexString(to), new Byte((byte) 0)));
        sendAPDU(getManageChannelCloseAPDU(from, to), SW_SELECT_PASSED);
    }

    /**
     * Issue the MANAGE CHANNEL OPEN to the requested logical channel over the contactless
     * I/O interface.
     *
     * @param from  origin logical channel.
     * @param to    number of the logical channel to be opened.
     */
    public void manageChannelCloseContactless(byte from, byte to) throws CardProxyException {
        logContactless(I18n.getString("start.managechannelclose", Integer.toHexString(from),
                           Integer.toHexString(to), new Byte((byte) 0)));
        sendAPDUContactless(getManageChannelCloseAPDU(from, to), SW_SELECT_PASSED);
    }

    public CommandAPDU getManageChannelCloseAPDU(byte from, byte to) {
	byte cla = getChannelCLA(from);
	return new CommandAPDU(new byte[] { cla, (byte) 0x70, (byte) 0x80, to });
    }
}
