/*
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CommunicationService;
import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.ResponseAPDU;

import java.util.Hashtable;
import java.util.Map;
import java.io.PrintWriter;
/**
 *  This class encapsulate CardService and CommunicationService instances and
 *  is used as a proxy between old CJCKCardService and new interfaces.
 *  Existing scripts should use it in same manner as CJCKCardService.
 *
 * @author Dmitri Trounine
 */
public class WrapperCardService implements CJCKCardService {
    
    private CardService cardService;
    /**
     * Base i/o interface used by this methods.
     */
    private CommunicationService comService;
    /**
     * Contacted communication service which will be used for powerUp and powerDown
     * in case of dual interface support
     */
    private CommunicationService contactedDualService;
    /**
     * Cache for communication interfaces.
     */
    private Map services;
    
    /**
     * If dual interfaces supported and communication service is contactless, 
     * contacted service should be affected
     */
    private boolean isContactedAffected = false;
    
    /**
     * Creates a new instance of WrapperCardService
     *
     * @param   cardService     An instance of the licensee's CardService implementation
     *                          obtained based on the interview info.
     * @param   iface   Base communication I/O interface for this card. Should be
     *                  CardService.CONTACTED_INTERFACE or CardService.CONTACTLESS_INTERFACE.
     *                  All i/o methods of this WrapperCardService calss will adress to
     *                  that i/o interface.
     */
    public WrapperCardService(CardService cardService, int iface) throws CardProxyException {
        this.cardService = cardService;
        services = new Hashtable();
            comService = cardService.getCommunicationService(iface);
        addService(iface, comService);
    }
    
    /**
     * Creates a new instance of WrapperCardService
     *
     * @param   cardService     An instance of the licensee's CardService implementation
     *                          obtained based on the interview info.
     * @param   iface   Base communication I/O interface for this card. Should be
     *                  CardService.CONTACTED_INTERFACE or CardService.CONTACTLESS_INTERFACE.
     *                  All i/o methods of this WrapperCardService calss will adress to
     *                  that i/o interface.
     * @param   isDualIOSupported Whether implementation supports both interfaces. 
     *          In case of such support powerUp() and powerDown() methods of contactless 
     *          communication interface will also invoke powerUp() and powerDown() methods
     *          of contacted communication interface.
     */
    public WrapperCardService(CardService cardService, int iface, boolean isDualIOSupported) throws CardProxyException {
        if (isDualIOSupported && iface == CardService.CONTACTLESS_INTERFACE) {
            isContactedAffected = true;
            contactedDualService = cardService.getCommunicationService(CardService.CONTACTED_INTERFACE);
        }
        this.cardService = cardService;
        services = new Hashtable();
        comService = cardService.getCommunicationService(iface);
        addService(iface, comService);
    }
    
    /**
     * Prepare to start a test.  Initialize the Proxy.
     * Called by the cJCK to notify the Card Service that a test is starting, and
     * specify the path of the working directory.
     * <p>Note:<ul>
     * <li>The Card Service should prepare the working directory for the test.
     * This includes deleting any intermediate files that may have been left over
     * from a prior test execution.
     * The cJCK assumes that no test applets exist in either the ROM image or EEPROM.
     * </ul>
     * @param workingDir
     *        The <tt>working directory</tt> to be used for this test.
     * @param numberOfExecutions
     *        Indicates the number of times that the proxy will be powered on.
     * @exception CardProxyException
     *        Thrown if unable to initialize or set the working directory.
     * @see #stopTest()
     */
    public void startTest(String workingDir, int numberOfExecutions)
    throws CardProxyException {
        cardService.startTest(workingDir, numberOfExecutions);
    }
    
    
    /**
     * Stop the test.  Terminate the Proxy.
     * Called by the cJCK to notify the Card Service that a test is stopping.
     * @see CJCKCardService#startTest
     * @exception CardProxyException
     *     Thrown if unable to terminate the Proxy.
     */
    public void stopTest()
    throws CardProxyException {
        cardService.stopTest();
    }
    
    /////////////////////////////////////////////////////////////////////////////
    // Utility functions
    /////////////////////////////////////////////////////////////////////////////
    
    /**
     * Converts the specified applet and creates a CAP file.
     * <P>This method enables the cJCK to invoke a Converter.
     * <br>The goal is to verify the functionality of the Converter.
     * <P>The cJCK will attempt to convert both "correct" and "incorrect" applets.<ul>
     * <li>An "incorrect" applet is one which violates the Java Card Language subset.
     * <li>A "correct" applet is one which conforms to the Java Card Language subset
     * and which runs correctly on the Reference Implementation.
     * The resulting CAP file will be installed and the test applet exercised to verify that
     * conversion was done correctly.  The byte code sequences in the CAP file are not examined.
     * </ul>
     * <br>How the conversion is accomplished is at the discretion of the implementer.
     * <P>Implementation Notes:<ul>
     * <li>When implementing this method, you should allow your Converter to
     * write messages to Standard Out and Standard Error.  You may want to
     * invoke your Converter with a "verbose" option to provide as much diagnostic
     * information as possible.
     * All output is captured in each test's .jtr file and can be viewed with a file editor
     * or from within JavaTest.
     * </ul>
     * @param packageName the name of the package being converted.
     * @param packageAid the AID of the package being converted.
     * @param majorVersion the major version of the package.
     * @param minorVersion the minor version of the package.
     * @param classRootDir the root directory of the class hierarchy.
     * The path /java/lang/javacard is appended to this root,
     * yielding the directory in which the Converter will look for classes.
     * @param exportRootDir the root directory of the path in which the
     * Converter will search for export files.
     * @param isExportMap if it is true, then the package should be
     * converted using the <code>export</code> file of the preexisting
     * cap file. The export file should be searched in the
     * <code>exportRootDir</code>.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @param apa[] an array of {@link com.sun.javacard.cjck.userinterface.AppletProperties}
     * which specifies the applets to be converted.
     * Each appletProperties entry must include name of class that defines the install method.
     * @return the fully-qualified file name of the cap file that was created.
     * @exception CardProxyException
     * if the Converter is unable to create the CAP file.
     * The exception message should be appropriately descriptive.
     * @see com.sun.javacard.cjck.userinterface.AppletProperties
     */
    public String convertPackage(String packageName,
            AppletID packageAid,
            int majorVersion,
            int minorVersion,
            String classRootDir,
            String exportRootDir,
            boolean isExportMap,
            String outputRootDir,
            AppletProperties[] apa)
            throws CardProxyException {
        return cardService.convertPackage(packageName, packageAid, majorVersion, minorVersion,
                classRootDir, exportRootDir, isExportMap, outputRootDir, apa);
    }
    
    
    /////////////////////////////////////////////////////////////////////////////
    // Loading functions
    /////////////////////////////////////////////////////////////////////////////
    
    
    /**
     * Loads the specified applets from their class files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support class file format.
     * <br>How the loading is done and where the applets are loaded is at the discretion
     * of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param packageNames an array with the package names. The array contains names of the all packages,
     * which are required for the test execution.
     * @param appletProperties an {@link AppletProperties} array which specifies the applets to be loaded.
     * @param classRootDir the root directory of the class hierarchy.
     * The path /java/lang/javacard is appended to this root,
     * yielding the directory in which the Converter will look for classes.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @return true if the packages with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then loadClassFileApplets should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadClassFileApplets(String[] packageNames,
            AppletProperties [] appletProperties,
            String classRootDir,
            String outputRootDir)
            throws CardProxyException {
        return cardService.loadClassFileApplets(packageNames,  appletProperties,
                classRootDir, outputRootDir);
    }
    
    /**
     * Loads the specified applets from their cap files into the Proxy.
     * This method should be implemented for pre-issuance Products, which
     * support CAP file format.
     * <br>How the loading is done and where the applets are loaded
     * is at the descriptive of the implementer.
     * <P>Note:<ul>
     * <li><em>Only the applet's package name, class name, AID, and params
     * are guaranteed to be valid.</em>
     * <li>Values of other applet properties, such as MAX_STACK, will be -1 if not specified.
     * <li>The implementer should use default values whenever a value of -1 is encountered.
     * </ul>
     * @param capfiles array with the cap file names. The array contains names
     * of the all cap files, which are required for the test execution.
     * @param appletProperties an {@link AppletProperties} array
     *     which specifies the applets to be loaded.
     * @param outputRootDir the root directory for output.
     * The Export File, if specified, must be put into the directory at
     * the outputRootDir and the path implied by the package name.
     * Any intermediate or other files produced by the Converter
     * must also go into this directory.
     * @return true if the package with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then loadCapFileApplets should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     if the Proxy is unable to load an applet in the list.
     *     The exception message should be appropriately descriptive.
     */
    public boolean loadCapFileApplets(String[] capfiles,
            AppletProperties [] appletProperties,
            String outputRootDir)
            throws CardProxyException {
        return cardService.loadCapFileApplets(capfiles, appletProperties, outputRootDir);
    }
    
    
    /**
     * initializes the CardService instance. Test Run Framework invokes this
     * method before invocation of any other methods of the interface.
     * @param args specifies initialization options.
     * These options are defined in the jte file by option -s.
     * @param out specifies PrintWriter for output.
     * @param ref a stream to which to write any diagnostic messages.
     */
    public void init(String[] args, PrintWriter out, PrintWriter ref) {
        cardService.init(args, out, ref);
    }
    
    
    /**
     * Retrieve the Proxy properties Hashtable.
     * The cJCK will call this method to retrieve the properties of the Proxy.
     * <br>The implementor is required to initialize the Hashtable with the keys listed below
     * and their associated values.
     * <br>Note that some values are String values, some are Integers, and some are booleans.
     * @return Hashtable containing the properties associated with the Proxy Properties constants.
     * <br>Valid keys are the static constants:
     * {@link #PROXY_VERSION_PROPERTY PROXY_VERSION_PROPERTY},
     * {@link #API_VERSION_PROPERTY API_VERSION_PROPERTY},
     * {@link #MAX_STACK_PROPERTY MAX_STACK_PROPERTY},
     * {@link #MAX_TRANSIENT_RAM_PROPERTY MAX_TRANSIENT_RAM_PROPERTY},
     * {@link #MAX_NVRAM_PROPERTY MAX_NVRAM_PROPERTY},
     * {@link #INT_SUPPORT_PROPERTY INT_SUPPORT_PROPERTY}
     */
    public Hashtable getProperties() {
        return cardService.getProperties();
    }
    
    /**
     *  Get a CommunicationService instance for specified i/o interface.
     *
     *  @param  iface   The I/O interface, that the requested CommunicationService will work over.
     *
     *  @exception  CardProxyException  if requested i/o interface is not avaliable.
     */
    public CommunicationService getCommunicationService(int iface) throws CardProxyException {
        return getService(iface);
    }
    
    /////////////////////////////////////////////////////////////////////////////
    // Control functions
    /////////////////////////////////////////////////////////////////////////////
    
    /**
     * Power-up the Proxy. If the Proxy has already been powered-up, this
     * method does nothing.
     * @see CJCKCardService#loadClassFileApplets
     * @see CJCKCardService#installCAPFile
     * @exception CardProxyException
     *     Thrown if powerUp failed.
     */
    public void powerUp()
    throws CardProxyException {
        if (isContactedAffected) {
            contactedDualService.powerUp();
        }
        comService.powerUp();        
    }
    
    
    /**
     * Power-down the Proxy. If the Proxy has already been powered-down, this
     * method do nothing.
     * @exception CardProxyException
     *     Thrown if powerDown failed.
     */
    public void powerDown()
    throws CardProxyException {
        comService.powerDown();
        if (isContactedAffected) {
            contactedDualService.powerDown();
        }
    }
    
    /**
     * Reset the Proxy.  Equivalent to resetting a smart card using the RESET line.
     * @exception CardProxyException
     *     Thrown if reset failed.
     */
    public void reset()
    throws CardProxyException {
        comService.powerDown();
        if (isContactedAffected) {
            contactedDualService.reset();
        }
        comService.powerUp();
    }
    
    
    /////////////////////////////////////////////////////////////////////////////
    // I/O functions
    /////////////////////////////////////////////////////////////////////////////
    
    /**
     * Installs a CAP file into the Proxy. This method should be implemented for
     * post-issuance products.
     * <P>Note:<ul>
     * Assumes that powerUp() has already been called.
     * </ul>
     * @param appletPropertiesArray an {@link AppletProperties} array which
     *     specifies the applets in the CAP file. If the cap file contains
     *     library package, then the array has zero length.
     * @param aCAPFileName is a fully-qualified file name including the path name.
     * @param outputRootDir the root directory for output.
     * Any intermediate or other files produced by the Converter must be created in
     * the directory at the outputRootDir and the path implied by the package name.
     * @return true if the package with applets are installed
     * successfully, false otherwise. If the applet's install method
     * throws an exception, then installCAPFile should return false
     * instead of throwing CardProxyException.
     * @exception CardProxyException
     *     thrown if the Proxy is unable to load the CAP file.
     *     The exception message should be appropriately detailed and descriptive.
     */
    public boolean installCAPFile(AppletProperties[] appletPropertiesArray,
            String aCAPFileName,
            String outputDir)
            throws CardProxyException {
        return cardService.installCAPFile(appletPropertiesArray, aCAPFileName, outputDir);
    }
    
    
    /**
     * Sends a <tt>CommandAPDU</tt> to the Proxy and gets the <tt>ResponseAPDU</tt>.
     * <br>Note:<ul>
     * The Card Service must copy the <tt>CommandAPDU</tt> bytes into the
     * <tt>ResponseAPDU</tt>.
     * </ul>
     * @param capdu
     *     The <tt>CommandAPDU</tt> to send.
     * @exception CardProxyException
     *     if sendAPDU failed.
     */
    public ResponseAPDU sendAPDU(CommandAPDU capdu)
    throws CardProxyException {
        return comService.sendAPDU(capdu);
    }
    
    /**
     * deletes list of the applets.
     * @param applets specifies list of the applets for deletion.
     * @return true if the all applets are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete applet instances.
     */
    public boolean deleteAppletInstances(AppletProperties[] applets)
    throws CardProxyException {
        return cardService.deleteAppletInstances(applets);
    }
    
    /**
     * deletes given applet/library package.
     * @param packageID specifies AID of the given package.
     * @param packageName specifies full qualified name of the given package.
     * @return true if the package are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete package.
     */
    public boolean deletePackage(AppletID packageID, String packageName)
    throws CardProxyException {
        return cardService.deletePackage(packageID, packageName);
    }
    
    /**
     * deletes given applet/library package and contained instances.
     * @param packageID specifies AID of the given package.
     * @param packageName specifies full qualified name of the given package.
     * @return true if the package are deleted successfully, false otherwise.
     * @exception CardProxyException
     *     thrown if unable to delete package and contained instances.
     */
    public boolean deletePackageAndInstances(AppletID packageID,
            String packageName)
            throws CardProxyException {
        return cardService.deletePackageAndInstances(packageID, packageName);
    }
    
    private void addService(int iface, CommunicationService comService) {
        Integer key = new Integer(iface);
        if (!services.containsKey(key)) {
            services.put(key, comService);
        }
    }
    
    private CommunicationService getService(int iface) throws CardProxyException {
        Integer key = new Integer(iface);
        CommunicationService comService = (CommunicationService) services.get(key);
        if (comService == null) {
            comService = cardService.getCommunicationService(iface);
            addService(iface, comService);
        }
        return comService;
    }

    public void setCardTerminal(StatefulCardTerminal terminal) {
        this.cardService.setCardTerminal(terminal);
    }
}
