/**
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Enumeration;

import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.JCScript;
import com.sun.javacard.cjck.userinterface.BinaryToolService;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CardServiceFactory;
import com.sun.javacard.cjck.userinterface.CardProxyException;
import com.sun.javacard.cjck.userinterface.CardSystemException;
import com.sun.javacard.cjck.userinterface.DeploymentException;
import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FrameworkException;
import com.sun.javacard.cjck.userinterface.RerunnableException;
import com.sun.javatest.Status;
import javacard.framework.CardException;

/**
 * BinaryTool's handler for particular test instance
 *
 * @author Mikhail Smirnov
 */
public class CardTest {

    public enum State {NOT_STARTED, EXECUTING, WAITING_FOR_RERUN,
            RERUNNING, COMPLETED};

    public static final int CARD_REPLACEMENT_ATTEMPTS = 3;


    protected DelayedPrintWriter logWriter;
    protected DelayedPrintWriter refWriter;

    private String log_id;
    private ConfigFileReader configFile;
    private CJCKCardService cardService;
    private BinaryToolService baseService;
    private State state;
    private JCScript script;

    private static final String VERSION = "2.2.1 Binary Verification Tool";
    private static final String BUILD_DATE = "10/08/24";
    private static int instance;            // the instance number
    private CardProxyArguments args;


    /**
     * The default constructor.
     */
    public CardTest() {
        state = State.NOT_STARTED;
    }

    public CardTest(boolean rerun, JCScript script) {
        state = rerun ? State.WAITING_FOR_RERUN : State.NOT_STARTED;
        this.script = script;
    }

    /**
     * Sets Test state
     */
    public void setState(State state) {
        this.state = state;
    }

    /*
     * Gets Test state as an integer constant
     */
    public State getState() {
        return state;
    }

    /**
     * Can be used to run CardTest from the command line.
     * 
     * @param args  the command-line arguments
     */
    public static void main(String[] args) {
        Status status;
        PrintWriter log = new PrintWriter(System.out, true);
        DelayedPrintWriter logWriter = new DelayedPrintWriter(log);
        DelayedPrintWriter refWriter = new DelayedPrintWriter(new PrintWriter(System.err));
        logWriter.println(I18n.getString("cardproxy.version.build", VERSION, BUILD_DATE));
        CardTest test = new CardTest();
        status = test.run(args, logWriter, refWriter);
        if (status.isPassed()) {
            logWriter.resetData();
            refWriter.resetData();
        } else {
            logWriter.flushData();
            refWriter.flushData();
        }
        log.println(I18n.getString("cardproxy.run.status", status));
        status.exit();  // return the status to JavaTest
    }
    
    
    /**
     * The method invoked by JavaTest if CardProxyTest is used as a JavaTest command.
     *
     * @param args  the command-line arguments supplied by JavaTest
     * @return Status a JavaTest status object
     *
     */
    public Status run(String[] arguments, PrintWriter log, PrintWriter ref) {
        // save the parameters
        this.logWriter = new DelayedPrintWriter(log);
        this.refWriter = new DelayedPrintWriter(ref);

        log_id = I18n.getString("cardproxy.log.id", new Integer(instance));
        log(I18n.getString("run.starting.msg", log_id, VERSION, BUILD_DATE));
        
        RunMode mode = new InstallMode();
        log(I18n.getString("java.classpath", System.getProperty("java.class.path")));
        Status status = Status.passed("OK");
        try {
            status = instantiateResources(arguments);
            if (!status.isPassed()) {
                return status;
            }
            ref(I18n.getString("contacted.test.started"));
            mode.init(logWriter, refWriter, cardService);
            if (state == State.NOT_STARTED) {
                setState(State.EXECUTING);
            } else if (state == State.WAITING_FOR_RERUN) {
                setState(State.RERUNNING);
            }
            try {
//TODO-Logging                System.out.println("Before run: CardService " + cardService + " state: " + baseService.getState()
//TODO-Logging                        + ", Test " + args.getTestDir() + " state: " + state); //FIXME: debug output
                status = mode.run(args, configFile);
            } catch (FatalException ex) {
                signalFatalException(ex);
            }  catch (CardSystemException ex) {
                signalRerunnableFailure(ex);
            } catch (OutOfMemoryError ex) {
                signalRerunnableFailure(ex);
            }
            ref(I18n.getString("contacted.test.status", status));
            ref(I18n.getString("contacted.test.finished"));
        } catch (RerunnableException ex) {
            status = Status.failed(ex.getMessage()); //TODO: we should be able to set NOT_RUN status
        } catch (CardProxyException e) {
            status = Status.failed(I18n.getString("cardservice.exception", e));
        } catch (Throwable t) {
            t.printStackTrace(this.logWriter);
            logWriter.flushData();
            refWriter.flushData();
            return Status.error(I18n.getString("unexpected.exception", t));
        } finally {
            finalizeTest();
        }
        if (!status.isPassed() && state != State.WAITING_FOR_RERUN) {
            logWriter.flushData();
            refWriter.flushData();
        } else {
            logWriter.resetData();
            refWriter.resetData();
        }
        if (state != State.WAITING_FOR_RERUN) {
            setState(State.COMPLETED);
        }
//TODO-Logging        System.out.println("After run: CardService state: " + baseService.getState()
//TODO-Logging                        + ", Test result: " + status 
//TODO-Logging                        + ", Test state: " + state); //FIXME: debug output
        baseService.releaseTerminal();
        return status;
    }

    private void signalRerunnableFailure(Throwable ex) throws Throwable {
        System.out.println("Exception: " + ex + " was caught while executing test");//FIXME: debug output
        if (!baseService.validateCardService()) {
            baseService.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
            throw ex;
        } else {
            if (state == State.EXECUTING) {
                setState(State.WAITING_FOR_RERUN);
                baseService.setState(StatefulCardTerminal.State.RECOVERED);
                throw new RerunnableException(ex.getMessage());
            } else if (state == State.RERUNNING) {
                baseService.setState(StatefulCardTerminal.State.VERIFIED);
                throw ex;
            }
        }
    }

    private void signalFatalException(Throwable ex) throws RerunnableException, Throwable {
        if (((FatalException) ex).getScope() == FatalException.Scope.CAD) {
            baseService.setState(StatefulCardTerminal.State.INVALID);
            if (state == State.EXECUTING) {
                setState(State.WAITING_FOR_RERUN);
                throw new RerunnableException(ex.getMessage());
            }
        } else if (((FatalException) ex).getScope() == FatalException.Scope.Card) {
            baseService.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
            if (state == State.EXECUTING) {
                setState(State.WAITING_FOR_RERUN);
                throw new RerunnableException(ex.getMessage());
            }
        }
        throw ex;
    }
    
    /**
     * Instantiates and initializes CardServise and ConfigFileReader.
     * @param args
     * @return
     */
    protected Status instantiateResources(String args[]) throws CardSystemException {
        try {
            this.args = new CardProxyArguments(args);
            // instantiate the ConfigFileReader
            configFile = new ConfigFileReader(logWriter, refWriter);
            for (Enumeration e = this.args.getLibConfigFiles(); e.hasMoreElements();
            configFile.read((String)e.nextElement()));
            configFile.read(this.args.getConfigFile());
            // instantiate a CJCKCardService
            instantiateCardServices();
            return Status.passed("");
        } catch (IllegalArgumentException ex) {
            return Status.failed(I18n.getString("can.not.parse.arg", ex));
        } catch (IOException io) {
            return Status.error(I18n.getString("can.not.read.config", io));
        }
    }
    
    /**
     * Depending on what card services supported by implementation this method
     * instantiate card services implementing CJCKCardService interface for
     * contacted or contactless or both I/O. Each CJCKCardService instance is created
     * based on CardService implementation class name and supported I/Os from interview
     */
    protected void instantiateCardServices() throws CardSystemException {
        log(I18n.getString("contacted.init"));
        cardService = new WrapperCardService(getBaseCardService(), CardService.CONTACTED_INTERFACE);
    }
    
    /**
     * This method creates and initializes CardService implementation
     */
    protected CardService getBaseCardService() throws CardSystemException {
        // instantiate a CJCKCardService
        String name = args.getCardServiceClass();
        String path = args.getClassPath();
        log(I18n.getString("instantiate.cjckcardservice", log_id, name, path));
        //FIXME: mock implementation
        CardService realService = CardServiceFactory.getCardService(name);
        baseService = CardServicesPool.getCardService(realService, args.getCardServiceArgs(), logWriter,
                refWriter);
        if (state == State.WAITING_FOR_RERUN && baseService.getState() == StatefulCardTerminal.State.RECOVERED) {
            baseService.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
        }
        if (baseService.getState() == StatefulCardTerminal.State.REPLACEMENT_REQUIRED) {
            for (int i = 0; i < CARD_REPLACEMENT_ATTEMPTS; i++) {
                if (baseService.getState() != StatefulCardTerminal.State.VERIFIED) {
                    script.cancelTimeout();
                    CardServicesPool.requireReplacement(baseService.getCardTerminal(), baseService);
                }
            }
        }
        if (baseService.getState() != StatefulCardTerminal.State.VERIFIED &&
                 baseService.getState() != StatefulCardTerminal.State.RECOVERED) {
            throw new CardProxyException(I18n.getString("card.verification.failed", 
                    CARD_REPLACEMENT_ATTEMPTS));
        }
        script.resetTimeout();
        
        log(I18n.getString("init.cjckcardservice", log_id, name));
        // initialize the Proxy
        baseService.startTest(args.getOutputDir(), args.getNuberOfReset());
        log(I18n.getString("cardservice.workdir.scriptcount", log_id,
                args.getOutputDir(), new Integer(args.getScriptFileCount())));
        return baseService;
    }
    
    /**
     * Power down the proxy.
     * New in 2.1.1
     * @return status
     */
    Status finalizeTest() {
        if (baseService == null) {
            return Status.failed("null.base.service");
        }
        try {
            baseService.deleteInstalledPackages();
            // powerDown the Proxy
            log(I18n.getString("call.powerdown", log_id));
            if (cardService != null) {
                cardService.powerDown();
                cardService.stopTest();
            }
            baseService.stopTest();
            return Status.passed("");
        } catch (CardProxyException e) {
            return Status.failed(I18n.getString("powerdown.got.exception", e));
        }
    }
    
    /**
     * Write a message to the log.
     * @param msg the message to be written
     */
    void log(String msg) {
        logWriter.println(msg);
    }
    
    void ref(String msg) {
        log(msg);
        refWriter.println(msg);
    }
};
