/*
 * %W% %E%
 * @Copyright
 */

package com.sun.javacard.cjck.invoke;

import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FatalException.Scope;
import com.sun.javatest.Parameters;
import com.sun.javatest.TestResult;
import com.sun.tck.bvtool.terminal.CardPresenceNotifier.CardEvent;
import com.sun.tck.bvtool.terminal.StatefulCardTerminal;
import com.sun.tck.bvtool.terminal.CardTerminalFactory;
import com.sun.tck.bvtool.terminal.PCSCTerminalFactory;
import com.sun.tck.bvtool.terminal.TestTerminalFactory;
import com.sun.javacard.cjck.I18n;
import com.sun.javacard.cjck.userinterface.BinaryToolService;
import javax.smartcardio.CardException;
import javax.smartcardio.CardTerminal;

import com.sun.javacard.cjck.userinterface.CardService;
import com.sun.javacard.cjck.userinterface.CJCKCardService;
import com.sun.javacard.cjck.userinterface.CardSystemException;
import com.sun.javacard.globalimpl.GPCardService;
import com.sun.javatest.Harness.Observer;
import com.sun.tck.bvtool.terminal.CardPresenceObserver;
import com.sun.tck.jc.interview.BVTEnvInterview;
import com.sun.tck.jc.javatest.cmdui.CmdUserInteraction;
import com.sun.tck.jc.javatest.gui.GuiUserInteraction;
import com.sun.tck.jc.javatest.ui.UserInteraction;
import java.io.File;
import java.io.FileInputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Vector;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * This class is used to provide CardService instances based
 * on card terminals availability. If there's no card terminals
 * available at a moment, then client should wait till some other
 * service releases a terminal.
 *
 * @author Mikhail Smirnov
 */
public class CardServicesPool {

    public enum State {NON_INITIALIZED, INITIALIZED, READY, NO_TERMINALS, INITIALIZE_ERROR};

    public static final int TIMEOUT = 300;
    public static final int MAX_CONNECTION_ATTEMPTS = 3;
    private static final String DEFAULT_TERMINALS_FACTORY = "com.sun.tck.bvtool.terminal.PCSCTerminalFactory";

    private static int capacity = 0;
    private static State state = State.NON_INITIALIZED;
    private static String reason = null;

    private static final LinkedBlockingQueue<StatefulCardTerminal> terminalsQueue =
            new LinkedBlockingQueue<StatefulCardTerminal>();
    private static Vector<String> terminalLabels = new Vector<String>();
    private static List<StatefulCardTerminal> availableTerminals = new ArrayList<StatefulCardTerminal>();
    private static UserInteraction ui;
    private static boolean isObserverRegistered;
    private static boolean testRunStarted;
    
    static {
        initTerminals();
    }

    /*
     * This method instantiates terminals queue and fills it with available
     * card terminals
     */
    private static boolean initTerminals() {
        try {
            String factoryName = getTerminalFactoryName();
            if (factoryName == null) {
                factoryName = DEFAULT_TERMINALS_FACTORY;
            }
            CardTerminalFactory factory = null;
            if (!DEFAULT_TERMINALS_FACTORY.equals(factoryName)) {
                ui = CmdUserInteraction.getUserInteraction();
                factory = new TestTerminalFactory();
            } else {
                ui = GuiUserInteraction.getUserInteraction();
                factory = new PCSCTerminalFactory();
            }
            List<CardTerminal> terminalsList = factory.terminals().list();
            if (terminalsList == null || terminalsList.isEmpty()) {
                state = State.NO_TERMINALS;
            } else {
                capacity = terminalsList.size();
                int i = 1;
                for (CardTerminal cad: terminalsList) {
                    terminalLabels.add(String.valueOf(i++));
                    availableTerminals.add(new StatefulCardTerminal(cad));
                }
                state = State.INITIALIZED;
            }
        } catch (CardException ex) {
            state = State.INITIALIZE_ERROR;
            reason = ex.toString();
        }
        return true;
    }

    /**
     * Returns the number of available terminals
     */
    public static int getCapacity() {
        return capacity;
    }

    /**
     * @return Observer object that will be notified about test run starting and
     * finishing events
     */
    public static Observer getTestRunObserver() {
        isObserverRegistered = true;
        return testRunObserver;
    }

    public static boolean isObserverRegistered() {
        return isObserverRegistered;
    }

    /**
     * Releases all identified terminals
     */
    private static void releaseTerminals() {
        for (StatefulCardTerminal terminal: getAvailableTerminals()) {
            if (terminal.getLabel() != null) {
                releaseTerminal(terminal);
            }
        }
        capacity = terminalsQueue.size();
//TODO-Logging        System.out.println("After releasing terminals: terminalsQueue.size=" + capacity);//FIXME remove debug output
    }

    /**
     * This method is invoked after test run is finished.
     * It starts card presence observer for all teminals available in the queue.
     */
    private static void startIdleTerminalsMonitoring() {
        CardPresenceObserver observer = new CardPresenceObserver() {
            @Override
            public void notified(CardEvent event) {
                StatefulCardTerminal srcTerminal = getSourceTerminal();
                if (event.equals(CardEvent.CARD_INSERTED)) {
                    srcTerminal.setState(StatefulCardTerminal.State.CARD_IDENTIFICATION_REQUIRED);
                } else if (event.equals(CardEvent.CARD_REMOVED)){
                    srcTerminal.setState(StatefulCardTerminal.State.CREATED);
                }
            }
        };
        for (StatefulCardTerminal terminal: terminalsQueue) {
            terminal.addCardPresenceObserver(observer);
        }
    }
    
    /**
     * This method is invoked after test run is started and checks
     * whether there terminals pending replacement or card identification
     */
    private static void checkIdleTerminalsMonitoring() {
        for (StatefulCardTerminal terminal: terminalsQueue) {
            terminal.removeAllCardPresenceObservers();
            if (terminal.getState().equals(StatefulCardTerminal.State.REPLACEMENT_REQUIRED)) {
                try {
                    requireReplacement(terminal, null);
                } catch (CardSystemException ex) {
                    terminal.setState(StatefulCardTerminal.State.INVALID);
                    terminalsQueue.remove(terminal);
                }
            } else if (terminal.getState().equals(StatefulCardTerminal.State.CARD_IDENTIFICATION_REQUIRED)) {
                //First we remove terminal from queue
                terminalsQueue.remove(terminal);
                //and then add it again provided that it passes all necessary checks
                addTerminalToQueue(terminal);
                terminal.setState(StatefulCardTerminal.State.CREATED);
            }
        }
    }

    /**
     * This method provides initialized CardService instance bounded to a terminal from
     * the pool. If there's no terminals available at a moment, this method will wait
     * for <link>TIMEOUT</link> seconds.
     * 
     * @throws GPException
     */
    //FIXME: there shouldn't be realService parameter in final implementation
    public static BinaryToolService getCardService(CardService realService, String[] args, PrintWriter log, PrintWriter ref) throws CardSystemException {
        BinaryToolService cardService = null;
        int count = 0;
        while(count++ < capacity) {
            try {
                StatefulCardTerminal terminal = null;
                synchronized(terminalsQueue) {
                    terminal = terminalsQueue.poll(TIMEOUT, TimeUnit.SECONDS);
                    if (terminal == null) {
                        throw new CardSystemException(I18n.getString("pool.waiting.timeout"));
                    }
//TODO-Logging                    System.out.println("Terminal: " + terminal + " is requested"); //FIXME: debug output
                    cardService = new GPCardService();
                    cardService.setRealCardService((CJCKCardService)realService);
                    cardService.setCardTerminal(terminal);
                    cardService.init(args, log, ref);
                    checkCardServiceState(cardService);

                }
                if (cardService.getState() == StatefulCardTerminal.State.VERIFIED
                        || cardService.getState() == StatefulCardTerminal.State.RECOVERED) {
//TODO-Logging                    System.out.println("Terminal: " + terminal + " is taken"); //FIXME: debug output
                    break;
                } else {
                    cardService = null;
//TODO-Logging                    System.out.println("Terminal: " + terminal + " is unavailable"); //FIXME: debug output
                }
            } catch (InterruptedException ex) {
                throw new CardSystemException(I18n.getString("pool.waiting.interrupted", ex));
            }
        }
        if (cardService == null) {
            throw new CardSystemException("no.more.terminals.available");
        }
        return cardService;
    }

    /**
     * Releases terminal and puts it back to queue if it is still usable
     * @param terminal
     */
    public static void releaseTerminal(StatefulCardTerminal terminal) {
        if (terminal.getState() != StatefulCardTerminal.State.INVALID &&
                !terminalsQueue.contains(terminal)) {
            terminalsQueue.offer(terminal);
        } else {
            capacity--;
        }
    }

    /**
     * This method checks if a static initialization of CardServicesPool was successful
     * @throws GPException
     */
    public static void checkState() throws CardSystemException {
        if (state == State.READY) {
            return;
        } else if (state == State.INITIALIZED) {
            if (!identifyCardTerminals()) {
                throw new CardSystemException(I18n.getString("pool.not.identified"));
            } else {
                state = State.READY;
            }
        } else if (state == State.NO_TERMINALS) {
            throw new CardSystemException(I18n.getString("pool.no.terminals"));
        } else if (state == State.INITIALIZE_ERROR) {
            throw new CardSystemException(I18n.getString("pool.initialization.error", reason));
        } else {
            throw new CardSystemException(I18n.getString("pool.not.initialized"));
        }
    }

    /**
     *
     * FIXME: refactoring needed: non-mandatory cardService parameter should be removed
     *
     * This method is called when a terminal that is associated with this
     * cardService instance contains a card that should be replaced. If user
     * doesn't want to replace a card we continue using other terminals
     *
     * @param cardService
     * @throws GPException
     */
    public static void requireReplacement(StatefulCardTerminal terminal, BinaryToolService cardService) throws CardSystemException {
//TODO-Logging        System.out.println("Card replacement required for terminal: " + terminal);//FIXME: debug output
        //StatefulCardTerminal terminal = cardService.getCardTerminal();
        if (testRunStarted) {
            if (!ui.askForCardReplacement(terminal)) {
                if (cardService != null) {
                    replaceTerminal(terminal, cardService);
                }
            } else {
                if (checkConnection(terminal)) {
                    setCardID(terminal);
                } else {
                    if (cardService != null) {
                        replaceTerminal(terminal, cardService);
                    }
                }
            }
            terminal.setState(StatefulCardTerminal.State.CREATED);
            if (cardService != null) {
                checkCardServiceState(cardService);
            }
        } else {
            terminal.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
        }
    }



    private static void replaceTerminal(StatefulCardTerminal terminal, BinaryToolService cardService) throws CardSystemException {
        terminal.setState(StatefulCardTerminal.State.INVALID);
        capacity--;
        if (capacity == 0) {
            throw new CardSystemException("no.more.terminals.available");
        }
        try {
            StatefulCardTerminal anotherTerminal = terminalsQueue.poll(TIMEOUT, TimeUnit.SECONDS);
            if (anotherTerminal == null) {
                throw new FatalException(Scope.CAD_SYSTEM, "no.more.terminals.available");
            }
            cardService.setCardTerminal(anotherTerminal);
        } catch (InterruptedException ex) {
            throw new CardSystemException(I18n.getString("pool.waiting.interrupted", ex));
        }
    }

    private static void checkCardServiceState(BinaryToolService cardService) throws CardSystemException {
        StatefulCardTerminal terminal = cardService.getCardTerminal();
        if (terminal.getState() == StatefulCardTerminal.State.CREATED) {
            if (cardService.initializeCardService() && cardService.validateCardService()){
                terminal.setState(StatefulCardTerminal.State.VERIFIED);
            } else {
                terminal.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
                requireReplacement(cardService.getCardTerminal(), cardService);
            }
        } else if (terminal.getState() == StatefulCardTerminal.State.VERIFIED) {
            try {
                if (!terminal.checkCardPresent()) {
                    requireReplacement(terminal, cardService);
                }
            } catch (CardSystemException ex) {
                terminal.setState(StatefulCardTerminal.State.REPLACEMENT_REQUIRED);
                requireReplacement(terminal, cardService);
            }
        }
    }

    /**
     * This method checks connection between terminal and card, asks for card's ID
     * and after these checks adds terminal to the queue
     * @param terminal
     */
    private static void addTerminalToQueue(StatefulCardTerminal terminal) {
        try {
            if (checkConnection(terminal)) {
                setCardID(terminal);
                terminalsQueue.offer(terminal);
            } else {
                capacity--;
            }
        } catch (CardSystemException ex) {
            capacity--;
        }
    }

    /**
     * Asks user to set ID of the card in the terminal
     * @param terminal
     * @throws CardSystemException
     */
    private static void setCardID(StatefulCardTerminal terminal) throws CardSystemException {
        String cardID = ui.showCardIDSelectDialog(terminal.getLabel());
        if (!checkCardID(cardID)) {
            throw new CardSystemException(I18n.getString("card.id.incorrect", cardID));
        }
        terminal.setInsertedCardID(cardID);
    }

    public static List<StatefulCardTerminal> getAvailableTerminals() {
        return availableTerminals;
    }

    /**
     * This method creates threads for each terminal to wait for card insertion.
     * Then it asks user to insert cards one after another into terminals with
     * indicated labels. When a terminal detects card insertion event it is associated
     * with this label.
     *
     * @throws GPException
     */
    private static boolean identifyCardTerminals() throws CardSystemException {
        if (capacity == 1) {
            StatefulCardTerminal terminal = availableTerminals.get(0);

            String terminalLabel = terminalLabels.elementAt(0);
            try {
                if (!terminal.isCardPresent()) {
                    ui.askForCardInsertion(terminalLabel, false);
                }
            } catch (CardException ex) {
                throw new CardSystemException(I18n.getString("pool.not.initialized"));
            }
            terminal.setLabel(terminalLabel);
            addTerminalToQueue(terminal);
        } else {
            for (StatefulCardTerminal terminal : availableTerminals) {
                try {
                    //Before identification terminals should be empty
                    if (terminal.isCardPresent()) {
                        ui.askForCardsRemoval();
                    }
                } catch (CardException ex) {
                    throw new CardSystemException(I18n.getString("pool.initialization.error", ex.getMessage()));
                }
            }
            for (int i = 0; i < capacity; i++) {
                String terminalLabel = terminalLabels.remove(0);
                StatefulCardTerminal identifiedTerminal = ui.askForCardInsertion(terminalLabel, false);
                if (identifiedTerminal == null) {
                    break;
                } else {
                    identifiedTerminal.setLabel(terminalLabel);
                    addTerminalToQueue(identifiedTerminal);
                }
            }
        }
        capacity = terminalsQueue.size();
        if (capacity < 1) {
            state = State.INITIALIZE_ERROR;
            return false;
        }
        for (StatefulCardTerminal terminal : terminalsQueue) {
            if (terminal.getLabel() == null || terminal.getInsertedCard() == null) {
                state = State.INITIALIZE_ERROR;
                return false;
            }
        }
        return true;
    }

    /**
     * Checks connection between card and terminal with CardTerminal.connect() method
     * This method doesn't check other things like possibility to install or select some
     * applets.
     *
     * @param terminal
     * @return true is connection is OK, false otherwise.
     * @throws CardSystemException
     */
    private static boolean checkConnection(StatefulCardTerminal terminal) throws CardSystemException {
        int connectionAttempts = 0;
        boolean connectOK = false;
        while (!(connectOK = terminal.checkConnection()) && ++connectionAttempts < MAX_CONNECTION_ATTEMPTS) {
            if (ui.askForRetryAfterFailedConnection(terminal.getLabel())) {
                if (!ui.askForCardReplacement(terminal)) {
                    throw new CardSystemException(I18n.getString("card.reinsertion.failed"));
                }
            } else {
                return false;
            }
        }
        return connectOK;
    }

    private static boolean checkCardID(String cardID) {
        return cardID != null; //TODO: probably some other checks should be added
    }

    private static String getTerminalFactoryName() {
        String root = getRootDir();
        if (root == null) {
            return DEFAULT_TERMINALS_FACTORY;
        }
        try {
            File file = new File(root + "/config.properties");
            if (!file.exists()) {
                return DEFAULT_TERMINALS_FACTORY;
            }
            Properties config = new Properties();
            config.load(new FileInputStream(file));
            return config.getProperty(BVTEnvInterview.TERMINAL_FACTORY_CLASS);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    private static String getRootDir() {

        String classPath = System.getProperty("java.class.path");
        String[] elements = classPath.split(File.pathSeparator);
        String jarLocation = null;
        for (String element : elements) {
            if (element.indexOf(".jar") == -1) {
                continue;
            } else {
                element = element.replace('/', File.separatorChar);
                if (element.indexOf(File.separator) != -1) {
                    jarLocation = element.substring(0, element.lastIndexOf(File.separator));
                } else {
                    jarLocation = ".";
                }
            }
        }
        return jarLocation;
    }

    private static Observer testRunObserver = new Observer() {

            @Override
            public void startingTestRun(Parameters params) {
                testRunStarted = true;
                checkIdleTerminalsMonitoring();
            }

            @Override
            public void startingTest(TestResult tr) {
                //do nothing
            }

            @Override
            public void finishedTest(TestResult tr) {
                //do nothing
            }

            @Override
            public void stoppingTestRun() {
                testRunStarted = false;
            }

            @Override
            public void finishedTesting() {
                //do nothing
            }

            @Override
            public void finishedTestRun(boolean allOK) {
                // we set testRunStarted to false in both stoppingTestRun and
                // finishingTestRun because stoppingTestRun doesn't notify
                // in case of normal tests execution completion
                testRunStarted = false;
                releaseTerminals();
                startIdleTerminalsMonitoring();
            }

            @Override
            public void error(String string) {
                //do nothing
            }
        };
}
