package uicc.scws;

/**
 * Interface providing the means to build a HTTP response. The object
 * implementing the <code>HttpResponse</code> interface is a <b>Temporary Entry Point Object</b>.
 */
public interface HttpResponse
{
    /**
     * Finalizes the header data in the outgoing buffer. It automatically adds missing
     * header data with default values by the SCWS.
     *
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                <li><code>BUFFER_OVERFLOW</code> if the adding of missing header data would cause
     *                access outside the Response buffer.
     *                </ul>
     */
    public void finalizeHeader() throws ScwsException;

    /**
     * Appends a header variable to outgoing buffer (e.g. "content-type: text/html").
     * The colon and space (":") as well as the "CRLF" will be appended automatically.
     * <p>
     * The header variables content-length or transfer-encoding are inserted by the 
     * framework depending on the transfer mode
     * <p>
     * @param name buffer containing the header name (for example "content-type")
     * @param nameOffset offset within <code>name</code> buffer
     * @param nameLength length of the value within <code>name</code> buffer
     * @param value buffer containing the header value (for example "text/html")
     * @param valueOffset offset within <code>value</code> buffer
     * @param valueLength length of the value within <code>value</code> buffer
     * @exception NullPointerException  if <code>name</code> or <code>value</code> is equal to null
     * @exception ArrayIndexOutOfBoundsException if the operation would cause access outside array bounds
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>BUFFER_OVERFLOW</code> If appending the header variable
     *                would cause an overflow of the response buffer.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void appendHeaderVariable(
        byte[] name,
        short nameOffset,
        short nameLength,
        byte[] value,
        short valueOffset,
        short valueLength) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Appends a header variable to the outgoing buffer (e.g. "content-type:text/html").
     * The colon and space (":") as well as the "CRLF" will be appended automatically.
     * <p>
     * The header variables content-length or transfer-encoding are inserted by the 
     * framework depending on the transfer mode
     * <p>
     * @param headerKeywordNameId Identifier of the required header name.
     * @param value buffer containing the header value (for example "text/html").
     * @param valueOffset offset within <code>value</code> buffer.
     * @param valueLength length of the value within <code>value</code> buffer.
     * @exception NullPointerException if <code>value</code> is equal to null
     * @exception ArrayIndexOutOfBoundsException if the operation would cause access outside array bounds
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *		      <li><code>UNKNOWN_KEYWORD_ID</code> if the <code> headerKeywordNameId</code>
     *		       is not correct according to <code>SCWSConstants</code>
     *                <li><code>BUFFER_OVERFLOW</code> If appending the header variable
     *                would cause an overflow of the response buffer.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void appendHeaderVariable(
        short headerKeywordNameId,
        byte[] value,
        short valueOffset,
        short valueLength) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Writes a header variable to outgoing buffer (e.g. "content-type: text/html") "CRLF"
     * will be appended automatically.
     * <p>
     * The header variables content-length or transfer-encoding are inserted by the 
     * framework depending on the transfer mode
     * <p>
     * @param data buffer holding the header variable
     * @param offset start offset of the header variable in the data buffer
     * @param length the length of the header variable in the buffer
     * @exception NullPointerException if <code>data</code> is equal to null
     * @exception ArrayIndexOutOfBoundsException if the operation would cause access outside array bounds
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>BUFFER_OVERFLOW</code> If appending the header variable
     *                would cause an overflow of the response buffer.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void appendHeaderVariable(byte[] data, short offset, short length)
        throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Writes the content of <code>data</code> into the response object.
     * <p>
     * Invoking this method implicitly finalizes the header of the
     * <code>HttpResponse</code>.
     * In case of chunked transfer encoding hex length of chunked data is automatically
     * added by the framework.  
     *
     * @param data buffer holding the content of the response
     * @param offset start offset of the content in the data buffer
     * @param length the length of the content in the buffer
     * @exception NullPointerException if <code>data</code> is equal to null.
     * @exception ArrayIndexOutOfBoundsException if the operation would cause access outside array bounds
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>BUFFER_OVERFLOW</code> If appending the content 
     *                         in fixed buffer size mode would cause an overflow of
     *                         the response buffer.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void appendContent(byte[] data, short offset, short length)
        throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Write an HTTP status code into the outgoing buffer (e.g. 200)
     *
     * @param code status code of the HTTP 1.1 protocol
     * @exception ScwsException with reason code
     *                 <ul>
     *                 <li><code>HEADER_ALREADY_FINALIZED</code>if the HTTP header was already finalized
     *		       <li><code>STATUS_LINE_ALREADY_SET</code> if the <code>writeStatusCode</code> or the
     *		       <code>appendHeaderVariable</code> has already been invoked
     *                 <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the flush or
     *                           sendError method was called previously
     *                 <li><code>HTTP_CODE_UNKNOWN</code> if the status code is unknown with respect to the HTTP 
     *			1.1 protocol
     *                 </ul>
     */
    public void writeStatusCode(short code) throws ScwsException;

    /**
     * Sends all data contained in the HTTP response. 
     * If this method is invoked in fixed buffer mode after a previous invocation of the same method or 
     * the sendError() message was invoked previously the SCWS will do nothing.
     * <p>
     * Invoking this method for the first time implicitly finalizes the header of the
     * <code>HttpResponse</code>.
     * 
     *  @throws ScwsException with reason codes: <ul><li><code>BUFFER_OVERFLOW</code> if the adding of missing header data would cause
     *          access outside the Response buffer. </ul>
     *
     */
    public void flush() ;

    /**
     * Sends an error code to the SCWS. The HTTP response is sent immediately by the SCWS
     * no other data is send by the SCWS. Missing header variables are added by the SCWS
     * with their default values.
     * <p>
     * Invoking this method implicitly finalizes the header of the
     * <code>HttpResponse</code>.
     *
     * @param errorCode The error code to be sent in the HTTP Response.
     * //@exception ScwsException with reason code
     * //		  <li><code>HTTP_CODE_UNKNOWN</code> if the error code is unknown with respect to the HTTP 
     * //		  1.1 protocol
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HTTP_CODE_UNKNOWN</code> if the error code is unknown with respect to the HTTP 
     *		      1.1 protocol
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void sendError(short errorCode);

    /**
     * Returns the remaining size of the Response Buffer available for this applet.
     * <p>
     * The size returned by this method is guaranteed until the next header data or
     * content data is added to the Response Buffer.
     * 
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously and the selected transfer mode is Fixed transfer mode
     *                </ul>
     *
     * @return The remaining size of the Buffer, or -1 if chunked mode is switched on
     */
    public int getRemainingResponseBufferSize();
    
    /**
     * Informs the SCWS to use the chunked mode for sending the response, 
     * chunked mode off is the default mode
     * <p>
     * This allows the SCWS to handle large ammounts of response data
     * 
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously
     *                </ul>
     */
    public void enableChunkMode();
    
    /**
     * Sets the content type of the HTTP response
     *
     * @param type the content-type keyword ID of the response
     *
     * @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code> UNKNOWN_KEYWORD_ID </code> if type is unknown.
     *                <li><code>HEADER_ALREADY_FINALIZED</code> if the HTTP header was
     *                already finalized.
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                previously.
     *                <ul><li><code>BUFFER_OVERFLOW</code> if the adding of header data would cause
     *                access outside the Response buffer. 
     *                </ul>
     */
    public void setContentType(short type) throws ScwsException;
    
    /**
     * Clears the Response Buffer content and reset the state of the <code>HttpResponse</code> Object to its initial
     *     state. All the header variables and content that was appended to the <code>HttpResponse</code> until the
     *   invocation of the <code>reset</code> method is lost.
     *  @exception ScwsException with reason codes:
     *                <ul>
     *                <li><code>HTTP_RESPONSE_ALREADY_SENT</code> if the
     *                <code>flush</code> or <code>sendError</code> method was called
     *                 previously or the method <code>enableChunkMode</code> was already invoked.
     *                </ul>
     */
    public void reset() throws ScwsException;

}
