/*
 *
 */
package uicc.scws;

/**
 * Interface providing the means to analyse the incoming HTTP request. The object
 * implementing the <code>HttpRequest</code> interface is a <b>Temporary Entry Point Object</b>.
 */
public interface HttpRequest
{
    /**
     * Find and copy the value of a header.
     *
     * @param headerKeywordName buffer that contains the name of the requested header
     * @param nameOffs offset within <code>headerKeywordName</code>
     * @param nameLength length of data within <code>headerKeywordName</code>
     * @param buffer buffer where the value shall be copied
     * @param bufferOffs offset where the value shall be copied
     * @param maxLength maximum length of the value that shall be copied
     * @return <code>bufferOffs</code>+length of the copied value or <code>KEYWORD_NOT_FOUND</code> in case the keyword is not found
     *
     * @throws NullPointerException if <code>headerKeywordName</code> or <code>buffer</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the copy operation would cause access out of bounds
     */
    public short findAndCopyKeywordValue(
        byte[] headerKeywordName,
        short nameOffs,
        short nameLength,
        byte[] buffer,
        short bufferOffs,
        short maxLength) throws NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Find and copy the value of a header or the value of a keyword/value part 
     * of the URI indicated by a <code>keywordId</code>. The supported keywords are
     * defined in <code>ScwsConstants</code> Interface.
     *
     * @param keywordId value of the request header name or part of the URI
     * @param buffer buffer where the value shall be copied
     * @param bufferOffs offset where the value shall be copied
     * @param maxLength maximum length of the value that shall be copied
     * @return <code>bufferOffs</code>+length of the copied value or <code>KEYWORD_NOT_FOUND</code> in case the keyword is not found
     * @throws ScwsException with reason <code>UNKNOWN_KEYWORD_ID</code> if the <code>keywordId</code> is not recognized
     *
     * @throws NullPointerException if <code>buffer</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the copy operation would cause access out of bounds
     *
     * @see uicc.scws.ScwsConstants
     */
    public short findAndCopyKeywordValue(
        short keywordId,
        byte[] buffer,
        short bufferOffs,
        short maxLength) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Returns the length of the HTTP request body if this is known from the header field
     * <code>Content-Length</code>. If chunked transfer encoding is used this method shall return -1. Chunked
     * transfer encoding is indicated in the header field <code>Transfer-Encoding</code> according to
     * HTTP V1.1 [RFC2616].
     *
     * @return content length of the request or -1 if chunked transfer encoding is used
     */
    public int getContentLength();

    /**
     * Returns the version of the HTTP protocol of the current request.
     *
     * @return HTTP version of the request or <code>KEYWORD_NOT_FOUND</code> in case the content type
     *      header field is not present, or <code>CONTENT_TYPE_UNKNOWN</code> in case the content type doesn't 
     *      match one of the possible content type keywords defined in <code>ScwsConstants</code>.
     *
     */
    public short getRequestHttpVersion();

    /**
     * Returns the content-type keyword of the HTTP request
     *
     * @return the content type of the request.
     *
     * @see uicc.scws.ScwsConstants
     */
    public short getContentType();

    /**
     * Reads the content type given in the HTTP request (e.g. content type: text/html will
     * write text/html into the buffer).
     *
     * @param data buffer to hold the content type from the request.
     * @param offs start in the buffer to hold the content type.
     * @param len maximum length of the content type to be copied
     * @return <code>offs</code>+length of the copied value.
     * @throws NullPointerException if <code>data</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the read operation would
     * cause access out of bounds
     */
    public short readContentType(byte[] data, short offs, short len) throws NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * This method copies up to <code>len</code> bytes of the
     * incoming HTTP request body to <code>data</code>.
     *
     * <p>If chunked transfer encoding is not used:<br>
     * The complete HTTP request body is contained in the HTTP request buffer
     * at the time of invocation of  the doXXX() method of the interface ScwsExtension.
     * This method copies <code>len</code> bytes to <code>data</data>
     * starting at <code>requestOffset</code> in the HTTP request buffer.
     * If <code>requestOffset</code> + <code>len</code> exceeds the length
     * of the HTTP request then the <code>ScwsException</code> with reason
     * code <code>UNREACHABLE_OFFSET</code> will be thrown.
     *
     * <p>If chunked transfer encoding is used or the request does not fit into the <code>HttpRequest buffer</code>:<br>
     *  The HTTP request has (in general) not been completely received by the
     * SCWS at the time of invocation of  the doXXX() method of the interface ScwsExtension.
     * This method copies <code>len</code> bytes to <code>data</data> starting at
     * <code>requestOffset</code> in the HTTP request buffer which contains
     * new HTTP request data which were not yet made available to the Applet in
     * any previous invocation of the doXXX() method of the interface ScwsExtension.
     * If there are less than <code>len</code> bytes
     * in the HTTP request buffer then this method will return <code>offset</code>
     * + length of the copied value.

     *
     * @param requestOffset Offset within the HTTP request buffer to start reading the request
     *        content. If the entire HTTP request fits into the buffer this is also the absolute
     *        offset into the HTTP request, otherwise it is the offset into the current subset of the HTTP request.
     * @param data byte array where request content shall be copied.
     * @param offset offset within <code>buffer</code>.
     * @param len maximum length of the content to be copied.
     * @return <code>offset</code>+length of the copied value.
     * @throws ScwsException with reason <code>UNREACHABLE_OFFSET</code> if the requested offset could not be reached.
     * @throws NullPointerException if <code>data</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the read operation would
     * cause access out of bounds
     */
    public short readContent(
        int requestOffset,
        byte[] data,
        short offset,
        short len) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * This method provides information if the HTTP request was completely received.
     * @return true if this fragment is the last fragment of this HTTP request,
     *         false if the HTTP request is not yet complete.
     */
     boolean isComplete();

      /**
       * This method provides information if the HTTP request is starting.
       * @return true if this fragment is the first fragment of this HTTP request,  false otherwise.
       */
      boolean isFirst();

      /**
       * This method returns a unique identifier (per card session) for the HTTP request
       * object. This allows the application to distinguish
       * between different requests when it is processing them concurrently.
       *
       * @return unique identifier for this HTTP request object
       */
     short getRequestID();

     /**
      * This method indicates to the SCWS that the applet does not want to be invoked
      * with any data fragments related to this HTTP request any more.
      */
     void cancelProcessing();

}
