/*
 *
 */
package uicc.scws;

/**
 * Interface providing the means to analyse the incoming HTTP request. The object
 * implementing the <code>HttpRequest</code> interface is a <b>Temporary Entry Point Object</b>.
 */
public interface HttpRequest
{
    /**
     * Find and copy the value of a header.
     *
     * @param headerKeywordName buffer that contains the name of the requested header
     * @param nameOffs offset within <code>headerKeywordName</code>
     * @param nameLength length of data within <code>headerKeywordName</code>
     * @param buffer buffer where the value shall be copied
     * @param bufferOffs offset where the value shall be copied
     * @param maxLength maximum length of the value that shall be copied
     * @return <code>bufferOffs</code>+length of the copied value or <code>KEYWORD_NOT_FOUND</code> in case the keyword is not found
     *
     * @throws NullPointerException if <code>headerKeywordName</code> or <code>buffer</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the copy operation would cause access out of bounds
     */
    public short findAndCopyKeywordValue(
        byte[] headerKeywordName,
        short nameOffs,
        short nameLength,
        byte[] buffer,
        short bufferOffs,
        short maxLength) throws NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Find and copy the value of a header or the value of a keyword/value part 
     * of the URI indicated by a <code>keywordId</code>. The supported keywords are
     * defined in <code>ScwsConstants</code> Interface.
     *
     * @param keywordId value of the request header name or part of the URI
     * @param buffer buffer where the value shall be copied
     * @param bufferOffs offset where the value shall be copied
     * @param maxLength maximum length of the value that shall be copied
     * @return <code>bufferOffs</code>+length of the copied value or <code>KEYWORD_NOT_FOUND</code> in case the keyword is not found
     * @throws ScwsException with reason <code>UNKNOWN_KEYWORD_ID</code> if the <code>keywordId</code> is not recognized
     *
     * @throws NullPointerException if <code>buffer</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the copy operation would cause access out of bounds
     *
     * @see uicc.scws.ScwsConstants
     */
    public short findAndCopyKeywordValue(
        short keywordId,
        byte[] buffer,
        short bufferOffs,
        short maxLength) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Returns the length of the HTTP request content
     *
     * @return content length of the request.
     */
    public int getContentLength();

    /**
     * Returns the version of the HTTP protocol of the current request.
     *
     * @return HTTP version of the request or <code>KEYWORD_NOT_FOUND</code> in case the content type
     *      header field is not present, or <code>CONTENT_TYPE_UNKNOWN</code> in case the content type doesn't 
     *      match one of the possible content type keywords defined in <code>ScwsConstants</code>.
     *
     */
    public short getRequestHttpVersion();

    /**
     * Returns the content-type keyword of the HTTP request
     *
     * @return the content type of the request.
     *
     * @see uicc.scws.ScwsConstants
     */
    public short getContentType();

    /**
     * Reads the content type given in the HTTP request (e.g. content type: text/html will
     * write text/html into the buffer).
     *
     * @param data buffer to hold the content type from the request.
     * @param offs start in the buffer to hold the content type.
     * @param len maximum length of the content type to be copied
     * @return <code>offs</code>+length of the copied value.
     * @throws NullPointerException if <code>data</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the read operation would
     * cause access out of bounds
     */
    public short readContentType(byte[] data, short offs, short len) throws NullPointerException, ArrayIndexOutOfBoundsException;

    /**
     * Read a part of the request content.
     *
     * @param requestOffset Offset within the HTTP request to start reading the request
     *            content.
     * @param data byte array where request content shall be copied.
     * @param offset offset within <code>buffer</code>.
     * @param len maximum length of the content to be copied.
     * @return <code>offset</code>+length of the copied value.
     * @throws ScwsException with reason <code>UNREACHABLE_OFFSET</code> if the requested offset could not be reached.
     * @throws NullPointerException if <code>data</code> is equal to null
     * @throws ArrayIndexOutOfBoundsException if the read operation would
     * cause access out of bounds
     */
    public short readContent(
        int requestOffset,
        byte[] data,
        short offset,
        short len) throws ScwsException,NullPointerException, ArrayIndexOutOfBoundsException;
}
