package uicc.connection;

/**
 * This event notifies a <code>UICCTransportLink</code>, <code>BIPLink</code> or <code>SCP81Connection</code> about data reception.
 *
 * The object implementing <code>DataReceived</code>
 * interface is a <b>Temporary Entry Point Object</b>.
 *
 */
public interface DataReceived extends Event
{

    /**
     * Returns the amount of data received, or -1 if data length is unknown (chunked transfer encoding is used),
     * or 0 if all received data have been read or discarded by the <code>Observer</code>.
     */
    public int getReceivedDataLength();

    /**
     * Copy the received data in a buffer. Data are handled as a stream, and are received upon applet request.
     * It is not possible to move backward in the data by using the <code>srcOffset>/code> parameter.
     * This is a non-blocking methid.
     *
     * @param srcOffset the offset of the first byte to copy in the received data
     * @param dstBuffer a reference to the destination buffer
     * @param dstOffset the position in the destination buffer
     * @param dstLength the data length to be copied
     * @return <code>dstOffset+length of the copied value</code>
     *
     * @exception  NullPointerException if <code>dstBuffer</code> is <code>null</code>
     *
     * @exception  ArrayIndexOutOfBoundsException  if <code>dstOffset</code>
     * or <code>dstLength</code> or both would cause access outside array bounds,
     * or if <code>dstLength</code> is negative.
     *
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>OUT_OF_DATA_BOUNDARIES</code> if <code>srcOffset</code>
     *                    or <code>dstOffset</code> or <code>dstLength</code> would cause
     *                    access outside received data.
     *              <li><code>INVALID_PARAMETER</code> if the requested <code>srcOffset</code> could not be reached.
     *          </ul>
    * @exception NullPointerException if <code>dstBuffer</code> is equal to <code>null</code>
     *
     */
    public short copyReceivedData(int srcOffset, byte[] dstBuffer, short dstOffset, short dstLength);

    /**
     * This method provides information if the transfer of the HTTP POST response was completely received.
     * @return true if this fragment is the last or only fragment of the response data, false if the transfer
     *         is not yet complete (e.g. more data is expected in transfer coding chunked).
     */
     public boolean responseIsComplete();

    /**
     * Discard data that have not been already read. Once invoked, no data could be read anymore.
     * <br>
     * This method is blocking until all data have been received by the underlying connection.
     *
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>OPERATION_NOT_ALLOWED</code>  if all data have already been received or discarded.
     *         </ul>
     */
    public void discardRemainingData();
}

