package uicc.connection;

/**
 * The <code>UICCTransportLink</code> interface represents a logical connection to a remote entity based on a reliable protocol.
 *
 * Objects implementing <code>UICCTransportLink</code> interface and returned by the <code>ConnectionServer.getConnection</code> method
 * are <b>Permanent Entry Point Objects</b>.
 *
 */
public interface UICCTransportLink extends Observable
{
    /**
     * Connects the application via the reliable link to a remote entity.
     * The configuration of the reliable link is made according to the delivered
     * parameters.
     * The underlying connection specified by the <code>Connection</code>
     * object is implicitly opened.
     *
     * <p>Note: this method returns when the link is established.
     *
     * @param iMaxRequestedInSDUSize the maximum SDU size requested by the application for data reception.
     * @param iMaxRequestedOutSDUSize the maximum SDU size requested by the application for data emission.
     * @param sDestPort link destination port to connect with
     * @param baLinkData Reference of the byte array containing the data to be used during the link establishment.
     * @param sOffset Position, within the <code>baLinkData</code> byte array, where the identification data begins
     * @param sLength Length of the data
     *
     * @return
     *         <ul>
     *              <li><code>CONNECT_OPERATION_FAILED</code> if the connect operation has failed.
     *              <li><code>CONNECT_OPERATION_PERFORMED_SUCCESSFULLY</code>
     *                                   if the connect operation has been performed successfully.
     *              <li><code>CONNECT_OPERATION_PERFORMED_WITH_MODIFICATIONS</code>
     *                                   if the connect operation has been performed with modifications,
     *                                   i.e. SDU sizes requested have been modified.
     *         </ul>
     *
     * @exception ArrayIndexOutOfBoundsException if <code>sOffset</code> and <code>sLength</code> are causing access
     *         outside of <code>baLinkData</code> array bounds
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *                  issued.
     *              <li><code>INVALID_STATE</code> if the reliable link is already connected.
     *              <li><code>USER_CANCELLATION</code> if the user cancelled the opening
     *                                  of the underlying connection.
     *              <li><code>CONNECTION_REFUSED</code> if the remote entity
     *                                  refused the connection setup attempt.
     *              <li><code>TIME_OUT</code> if a timeout occurs.
     *		    <li><code>UNSUPPORTED_OPERATION</code> if the Terminal Equipment does not support the required mechanisms to perform the action.
     *    			<li><code>RESOURCE_NOT_AVAILABLE</code> if there are not sufficient
     *                resources available (on the UICC or on the terminal).
     *         </ul>
     * @exception NullPointerException if <code>baLinkData</code> or <code>oObserver</code> equal to <code>null</code>
     */
    public short connect(int iMaxRequestedInSDUSize, int iMaxRequestedOutSDUSize,
                    short sDestPort,
                    byte[] baLinkData, short sOffset, short sLength)

    		throws ArrayIndexOutOfBoundsException,
    		       ConnectionException,
    		       NullPointerException;

    /**
     * Disconnects the application from the remote entity.
     * The underlying connection is not closed implicitly.
     *
     * @return <code>true</code> if the proactive command has been issued successfully or if the <code>UICCTransportLink</code> is already
     *         disconnected; <code>false</code> otherwise
     *
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *                <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *                  issued.
     *                <li><code>USER_CANCELLATION</code> if the user cancelled the disconnecting of the reliable link
     *         </ul>
     */
    public boolean disconnect()
    		throws ConnectionException;

   /**
     * Disconnects the application from the remote entity and requests the closing of the underlying connection.
     *
     * In case of multiplexing <code>Connection</code>, this is performed only if there is no other <code>UICCTransportLink</code> connected using the same Connection.
     *
     * @return <code>true</code> if the proactive command has been issued successfully and the <code>UICCTransportLink</code> gets
     *         disconnected; <code>false</code> otherwise
     *
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *                <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *                  issued.
     *                <li><code>USER_CANCELLATION</code> if the user cancelled the disconnecting of the reliable link
     *         </ul>
     */
    public boolean closeConnection()
    		throws ConnectionException;

    /**
     * Sends a SDU on a reliable link to the remote entity.
     *
     * <p>Note: The method returns when all ACK PDUs related to all data
     *                PDUs sent have been received.
     *
     * @param baSDUData the SDU to be sent
     * @param sOffset the starting offset of the SDU in the <code>baSDUData</code> byte array
     * @param sLength the length of the data
     *
     * @return <code>true</code> if the proactive command has been issued successfully;
     *             <code>false</code> otherwise
     *
     * @exception ArrayIndexOutOfBoundsException if <code>sOffset</code> and <code>sLength</code> are causing access
     *         outside of <code>baSDUData</code> array bounds
     *
     * @exception NullPointerException if <code>baSDUData</code> is equal to <code>null</code>
     * @exception ConnectionException with the following reason code:
     *         <ul>
     *              <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
     *                  issued.
     *              <li><code>INVALID_STATE</code> if the reliable link is not connected.
     *              <li><code>USER_CANCELLATION</code> if the user cancelled the data sending
     *              <li><code>OPERATION_NOT_ALLOWED</code> if the data sending is not allowed.
     *              <li><code>TIME_OUT</code> if a timeout occurs.
     *         </ul>
     */
    public boolean sendSDU(byte[] baSDUData, short sOffset, short sLength);

    /**
     * Retrieves the size used for incoming SDUs on this reliable link.
     *
     * @return the size used for the incoming SDUs
     */
    public int getInSDUSize();

    /**
     * Retrieves the size used for outgoing SDUs on this reliable link.
     *
     * @return the size used for the outgoing SDUs
     */
    public int getOutSDUSize();

    /**
	 * Checks if the remote entity is still connected to the application.
	 * In case of CAT_TP it sends a NUL-ACK PDU to check if the connection is alive.
	 *
	 * <p>Note: The method returns when an acknowledgement has been received
	 *          and it returns false if the connection is no longer active
	 *          (e.g. the retransmission counter is exceeded in the CAT_TP protocol layer).
	 *
	 * @return <code>true</code> if the remote entity is still connected to the application;
	 *         <code>false</code> if the remote entity is not connected to the application
	 *
	 * @exception ConnectionException with the following reason code:
	 *         <ul>
	 *              <li><code>TOOLKIT_RESOURCE_NOT_AVAILABLE</code> if the proactive command could not be
	 *                  issued.
	 *              <li><code>INVALID_STATE</code> if the reliable link is not connected.
	 *              <li><code>USER_CANCELLATION</code> if the user cancelled the data sending.
	 *         </ul>
	 */
	public boolean isAlive();
}
