package uicc.hci.framework;

/**
 * This interface is the base interface for any HCI service.
 * <P> 
 * An HCI service instance shall be a permanent JCRE Entry Point Object
 * <P>
 * It provides the means for an Applet instance to get access to the resources
 * of a specific HCI service. An Applet instance can retrieve an implementation of
 * a specific service via the <code>HCIDevice</code>.
 * <P>
 * For a service object that is implementing one of the subinterfaces defined in this API
 * the Applet can only register the corresponding Listener defined in the same package
 * (e.g. a <code>CardEmulationService</code> shall only accept the regsitration of a
 * <code>CardEmulationListener</code>.
 * <P>
 * An Applet instance shall only be able to register one type of listener per service
 * at any time. Listener registration is persistent.
 * <P>
 * To receive events via the registered Listeners an Applet instance shall activate the events.
 * All events shall be deactivated by default. Event activation is persistent.
 **/
public interface HCIService {

    /**
     * This method registers a Listener object to the Service instance
     *
     * @param listener listener to be registered with the given service
     * @exception HCIException with reason code
     *            <ul>
     *            <li><code>HCIException.HCI_WRONG_LISTENER_TYPE</code> in case the listener Object registerd to the service
     *            is not implementing the corresponding interface of the HCIService instance.
     * <ul>
     * <li>a <code>ReaderListener</code> can only be registered to a <code>ReaderService</code>,
     * <li>a <code>CardEmulationListener</code> can only be registered to a <code>CardEmulationService</code>,
     * <li>a <code>ConnectivityListener</code> can only be registered to a <code>ConnectivityService</code>
     * </ul>
     *           <li><code>HCIException.HCI_LISTENER_ALREADY_REGISTERED</code> in case a listener Object of the same type has already been registered.
     * </ul>
     */
    public void register(HCIListener listener) throws HCIException;

    /** This method unregisters the given listener from the service.
     * <P>
     *  Calling <code>deregister</code> with an argument that does not identify a
     *  currently registered <code>HCIListener</code> has no effect.
     * 
     * @param listener listener to be unregistered from the service
     */
    public void deregister(HCIListener listener);

    /** 
     * This method shall not be used. It is retained for binary interoperability
     * only and will throw and exception when invoked.
     *
     * @param event without meaning (as any invocation of the method will throw and exception)
     * @throws HCIException with reason code <code>HCI_ACCESS_NOT_GRANTED</code>.
     */
    public void requestCallbackNotification(byte event) throws HCIException;

    /**
     * This method is used to activate an event.
     * <p>
     * The Applet can at any time activate an event, it will receive this event only
     * when the corresponding <code>Listener</code> sub-interface is registered.
     * </p>
     *
     * <ul>
     * <li><P>Possible values for the parameter <code>event</code> for the different Service objects are:
     * <ul>
     * <li>For all *Service interfaces:
     * <ul>
     * <li>HCIListenr.EVENT_HCI_TRANSMISSION_FAILED
     * <li>HCIListener.EVENT_HCI_RECEPTIONS_FAILED
     * </ul>
     * <li>for CardEmulationService:
     * <ul>
     * <li><code>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>CardEmulationListener.EVENT_ON_SEND_DATA</code>,
     * <li><code>CardEmulationListener.EVENT_FIELD_OFF</code>.
     * </ul>
     * <li>for ReaderService:
     * <ul>
     * <li><code>ReaderListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_WRITE_EXCHANGE_DATA_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_TARGET_DISCOVERED</code>.
     * </ul>
     * <li>for ConnectivityService:
     * <ul>
     * <li><code>ConnectivityListener.EVENT_STANDBY</code>.
     * </ul>
     * </ul>
     * </ul>
     * @param event event to be activated
     * @throws HCIException with reason code 
     *     <ul>
     *      <li><code>HCI_WRONG_EVENT_TYPE</code> in case a wrong event was activated for this Service instance
     *      <li><code>HCI_CONDITIONS_NOT_SATISFIEDE</code> if one or more conditions to activate the event are not satisfied
     *      <li><code>HCI_CURRENTLY_DISABLED</code> if the interface to the contactless frontend (CLF) is currently disabled and the 
     *          event is <code>ReaderListener.EVENT_TARGET_DISCOVERED</code>
     *     </ul>
     */
    public void activateEvent(byte event) throws HCIException;

    /**
     * This method is used to deactivate an event from the list of activated events.
     * 
     * <ul>
     * <li><P>Possible values for the parameter <code>event</code> for the different Service objects are:
     * <ul>
     * <li>For all *Service interfaces:
     * <ul>
     * <li>HCIListenr.EVENT_HCI_TRANSMISSION_FAILED
     * <li>HCIListener.EVENT_HCI_RECEPTIONS_FAILED
     * </ul>
     * <li>for CardEmulationService:
     * <ul>
     * <li><code>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>CardEmulationListener.EVENT_ON_SEND_DATA</code>,
     * <li><code>CardEmulationListener.EVENT_FIELD_OFF</code>.
     * </ul>
     * <li>for ReaderService:
     * <ul>
     * <li><code>ReaderListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_WRITE_EXCHANGE_DATA_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_TARGET_DISCOVERED</code>.
     * </ul>
     * <li>for ConnectivityService:
     * <ul>
     * <li><code>ConnectivityListener.EVENT_STANDBY</code>.
     * </ul>
     * </ul>
     * </ul>
     * @param event event to be deactivated
      * @throws HCIException with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE</code> in case a wrong event was deactivated for this Service instance	 
     *          or in case the event wasn't activated before.
     */
    public void deactivateEvent(byte event) throws HCIException;

    /**
     * return the activation state of an event.
     * 
     * <ul>
     * <li><P>Possible values for the parameter <code>event</code> for the different Service objects are:
     * <ul>
     * <li>For all *Service interfaces:
     * <ul>
     * <li>HCIListenr.EVENT_HCI_TRANSMISSION_FAILED
     * <li>HCIListener.EVENT_HCI_RECEPTIONS_FAILED
     * </ul>
     * <li>for CardEmulationService:
     * <ul>
     * <li><code>CardEmulationListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>CardEmulationListener.EVENT_SEND_DATA</code>,
     * <li><code>CardEmulationListener.EVENT_FIELD_OFF</code>.
     * </ul>
     * <li>for ReaderService:
     * <ul>
     * <li><code>ReaderListener.EVENT_GET_PARAMETER_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_WRITE_EXCHANGE_DATA_RESPONSE</code>,
     * <li><code>ReaderListener.EVENT_TARGET_DISCOVERED</code>.
     * </ul>
     * <li>for ConnectivityService:
     * <ul>
     * <li><code>ConnectivityListener.EVENT_STANDBY</code>.
     * </ul>
     * </ul>
     * </ul>
     *
     * @param event event to request the activation state
     * @return true if the event is activated for this listener, else false
      * @throws HCIException with reason code <code>HCIException.HCI_WRONG_EVENT_TYPE</code>  in case the event was not one of the possible values.
     */
    public boolean getEventNotificationStatus(byte event) throws HCIException;
}

