/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.me.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class Utils {
    public static final String ENCODING = "UTF-8";
    private static final int BUFF_SIZE = 128*1024;
    static private MessageDigest md5;
    static private byte[] md5Buff = new byte[BUFF_SIZE];
    static private byte[] buff = new byte[BUFF_SIZE];
    private static Object md5Monitor = new Object();

    public static boolean isHexString(String val) {
        char[] data = val.toCharArray();
        boolean retVal = true;
        for (int i = 0; i < data.length; i++) {
            char c = data[i];
            retVal = retVal && (((c >= '0') && (c <= '9'))
                                || ((c >= 'a') && (c <= 'f'))
                                || ((c >= 'A') && (c <= 'F')));
        }
        return retVal;
    }

    public static class InstallationException extends RuntimeException {
        public InstallationException(String msg) {
            super(msg);
        }

        public InstallationException(String msg, Throwable t) {
            super(msg, t);
        }

    }
    private static void initMd5() {
        if (md5 != null) {
            return;
        }
        try {
            md5 = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("MD5 should be supported.", e);
        }
    }

    public static String getMD5CheckSum(String value) throws IOException {
        return getMD5CheckSum(new ByteArrayInputStream(value.getBytes(ENCODING)));
    }

    public static String getMD5CheckSum(InputStream in) throws IOException {
        int total = 0;
        synchronized (md5Monitor) {
            initMd5();
            try {
                int read = 0;
                while ((read = in.read(md5Buff, 0, Math.min(in.available(), md5Buff.length))) > 0) {
                    total += read;
                    md5.update(md5Buff, 0, read);
                }
                return getHexString(md5.digest());// + ":" + Integer.toHexString(total);
            } finally {
                md5.reset();
                in.close();
            }
        }
    }
    
    public static String getHexString(byte[] bs) {
        if (bs == null) {
            return "null";
        } else if (bs.length == 0) {
            return "empty";
        } else {
            StringBuffer retVal = new StringBuffer();
            for (int i = 0; i < bs.length; i++) {
                retVal.append(Integer.toHexString(0xFF & bs[i]));
            }
            return retVal.toString();
        }
    }
    
    public static void addArrayToCollection(Collection list, Object[] array) {
        if ((array != null) && (list != null)) {
            for (int i = 0; i < array.length; list.add(array[i++]));
        }
    }

    public static String copyDataWithMD5Checksum(InputStream in, OutputStream out) throws IOException {
        int read = 0;
        BufferedInputStream buffIn = new  BufferedInputStream(in);
        BufferedOutputStream buffOut = new BufferedOutputStream(out);
        try {
            while ((read = buffIn.read()) >= 0) {
                md5.update((byte)read);
                buffOut.write((byte)read);
            }
        } finally {
            buffOut.flush();
            in.close();
        }

        return getHexString(md5.digest());
    }

    public static void copyData(Reader in, Writer out) throws IOException {
        int read = 0;
        BufferedReader buffIn = new  BufferedReader(in);
        BufferedWriter buffOut = new BufferedWriter(out);
        try {
            while ((read = buffIn.read()) >= 0) {
                buffOut.write((char)read);
            }
        } finally {
            buffOut.flush();
            in.close();
        }

    }

    public static void copyData(InputStream in, OutputStream out, int length) throws IOException {
        int c;
        for (int i = 0; ((i < length) && ((c = in.read())>=0)); i++) {
            out.write(c);
        }
    }

    public static void copyData(InputStream in, OutputStream out) throws IOException {
        int read = 0;
        BufferedInputStream buffIn = new  BufferedInputStream(in);
        BufferedOutputStream buffOut = new BufferedOutputStream(out);
        try {
            while ((read = buffIn.read()) >= 0) {
                buffOut.write((byte)read);
            }
        } finally {
            buffOut.flush();
            in.close();
        }
    }

    public static String getRelativePath(String root, File file) {
        String path;
        try {
            path = file.getCanonicalPath();
        } catch (IOException e) {
            e.printStackTrace();
            throw new IllegalArgumentException("Can not obtain canonical path: " + e);
        }
        if (path.startsWith(root)) {
            return path.substring(root.length());
        } else {
            throw new IllegalArgumentException("The list that can be linked is not inside. Probably symboluc links are exist.");
        }
    }
  
    public static File createFile(String root, String relativePath) {
        File file = new File(root + File.separator + relativePath.replace('/', File.separatorChar));
        mkdirs(file.getParentFile());
        return file;
    }

    public static OutputStream open(String root, String relativePath) throws IOException {
        File file = createFile(root, relativePath);
        return new FileOutputStream(file);
    }
    
    public static void mkdirs(File dir) throws InstallationException {        
        dir.mkdirs();
        if (!dir.isDirectory()) {
            throw new InstallationException("Can not create '" + dir.getAbsolutePath() + "'");
        }
    }

    public static void moveFile(String relPath, File from, File to) throws InstallationException {
        String localRelPath = relPath.replace('/', File.separatorChar);
        File fromFile = new File (from, localRelPath);
        File toFile = new File (to, localRelPath);
        if (toFile.exists() && (!toFile.delete())) {
            throw new InstallationException("Can not delete file in back up directory:" + toFile);
        }
        File parent = toFile.getParentFile();
        if ((!parent.isDirectory()) && (!parent.mkdirs())) {
            throw new InstallationException("can not create a parent directory:" + parent);
        }
        if (!fromFile.renameTo(toFile)) {
            throw new InstallationException("Can not move the file from " + fromFile + " to " + toFile);
        }
    }

    public static InputStream openFileForRead(File file) {
        try {
            if (file.exists()) {
                return new FileInputStream(file);
            } else {
                throw new InstallationException("Can not open file to read:" + file);
            }
        } catch (Exception e) {
            throw new InstallationException("Can not open file to read:" + file, e);
        }
    }

    public static Writer openFileToWrite(File file) {
        try {
            mkdirs(file.getParentFile());
            FileOutputStream out = new FileOutputStream(file);
            return new BufferedWriter(new OutputStreamWriter(out, ENCODING));
        } catch (Exception e) {
            throw new InstallationException("Can not open file to read:" + file, e);
        }
    }

    public static Map<String, String> loadProperties(InputStream in) {
        Closables closables = new Closables();
        try {
            Properties props = new Properties();
            closables.add(in);
            props.load(new InputStreamReader(in, "UTF-8"));
            HashMap<String,String> retVal = new HashMap<String,String>();
            for (Object key : props.keySet()) {
                retVal.put((String)key, (String)props.get(key));
            }
            return retVal;
        } catch (InstallationException e) {
            throw e;
        } catch (Exception e) {
            throw new InstallationException("Can not open bundle property file", e);
        } finally {
            closables.close();
        }
    }

    public static String readTextStream(InputStream in) {
        Closables cls = new Closables();
        cls.add(in);
        try {
            StringBuffer retVal = new StringBuffer();
            BufferedReader reader = new BufferedReader(new InputStreamReader(in, ENCODING));
            int c;
            while ((c = reader.read()) >= 0) {
                retVal.append((char)c);
            }
            return retVal.toString();
        } catch (Exception e) {
            throw new InstallationException("Can not read stream:" + in, e);
        } finally {
            cls.close();
        }
    }

    public static byte[] readBytesFromStream(InputStream in) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        copyData(in, out);
        return out.toByteArray();
    }


    public static Set<String> listFilesFromZipFile(File zipFile) throws IOException {
        LinkedHashSet<String> retVal = new LinkedHashSet<String>();
        ZipFile zip = new ZipFile(zipFile);
        for (Enumeration<? extends ZipEntry> e = zip.entries(); e.hasMoreElements();) {
            ZipEntry entry = e.nextElement();
            if (!entry.isDirectory()) {
                retVal.add(entry.getName());
            }
        }
        zip.close();
        return retVal;
    }

    public static StringBuffer join(String delim, Iterable list) {
        StringBuffer retVal = new StringBuffer();
        String add = "";
        for (Object arg : list) {
            retVal.append(add).append(arg);
            add = delim;
        }
        return retVal;
    }

    public static void storeMap(String name, Map<String,String> map, OutputStream stream) throws IOException {
        Writer out = new BufferedWriter(new OutputStreamWriter(stream, ENCODING));
        for (String key : map.keySet()) {
            out.write(key);
            out.write("=");
            out.write(map.get(key));
            out.write("\n");
        }
        out.close();
    }

    public static void storeProperties(String name, Map<String,String> map, OutputStream out) throws IOException {
        Properties props = new Properties();
        for (String  key : map.keySet()) {
            props.setProperty(key, map.get(key));
        }
        props.store(out, name);
        out.close();
    }

    public static <T> T[] combine(T[] first, T[] second) {
        if (first == null) {
            return second;
        } else if (second == null) {
            return first;
        }
        int length = first.length + second.length;
        T[] retVal = (T[])Array.newInstance(first.getClass().getComponentType(), length);
        System.arraycopy(first, 0, retVal, 0, first.length);
        System.arraycopy(second, 0, retVal, first.length, second.length);
        return retVal;
    }

    public static <K,V> Map<K,V> filter(Map<K,V> in, K[] keys, Map<K,V> out) {
        for (K path : keys) {
            V value = in.get(path);
            if (value != null) {
                out.put(path, value);
            }
        }
        return out;
    }


    public List<Pair<Integer, String>> getConstants(Class current) {
        ArrayList<Pair<Integer, String>> retVal = new ArrayList<Pair<Integer, String>>();
        try {
            for (Field f : current.getFields()) {
                int mods = f.getModifiers();
                Class type = f.getType();
                if (Modifier.isFinal(mods) && Modifier.isStatic(mods)
                        && isIntType(type)) {
                    int val = toInteger(type, f.getInt(null));
                    retVal.add(new Pair<Integer, String>(val, f.getName()));
                }
            }
        } catch  (Exception e) {
            // Ignore
        }
        return retVal;
    }

    public static String findName(int value, Class... classes) {
        try {
            for (Class current : classes) {
                for (Field f : current.getFields()) {
                    int mods = f.getModifiers();
                    Class type = f.getType();
                    if (Modifier.isFinal(mods) && Modifier.isStatic(mods)
                            && isIntType(type)) {
                        int val = f.getInt(null);
                        if (toInteger(type, val) == toInteger(type, value)) {
                            return f.getName() + "(0x" + Integer.toHexString(toInteger(type, value)) +")";
                        }
                    }
                }
            }
        } catch  (Exception e) {
            // Ignore
        }
        return Integer.toHexString(value);
    }
    public static String canonize(byte[] path) {
        if (path == null) {
            return null;
        }
        return canonize(path, 0, path.length);
    }

    public static String canonize(byte[] path, int offset, int length) {
        if (path == null) {
            return null;
        }
        StringBuffer retVal = new StringBuffer();
        for (int i = 0; i < length; i++) {
            String e =  Integer.toHexString(path[i + offset] & 0xFF);
            if (e.length() == 1) {
                retVal.append('0');
            }
            retVal.append(e);
        }
        return retVal.toString();
    }

//    public static String findTagName(byte tag) {
//        return findName(tag, ETSITags.class);
//    }
    private static boolean isIntType(Class cl) {
        return Byte.TYPE.equals(cl) || Short.TYPE.equals(cl) || Integer.TYPE.equals(cl);
    }
    private static int toInteger(Class cl, int val) {
        if (Byte.TYPE.equals(cl)) {
            return val & 0xFF;
        } else if (Short.TYPE.equals(cl)) {
            return val & 0xFFFF;
        } else {
            return val;
        }
    }
    public static byte[] parse(String line) {
        ByteArrayOutputStream buff = new ByteArrayOutputStream();
        char[] data = line.toCharArray();
        boolean hasPrevious = false;
        char previous = (byte)0;
        for (int i = 0; i < data.length; i++) {
            char c = data[i];
            if (isHexChar(c) && hasPrevious) {
                hasPrevious = false;
                buff.write(parseByte(previous, c));
            } else if (isHexChar(c) && !hasPrevious) {
                hasPrevious = true;
                previous = c;
            } else if (hasPrevious) {
                hasPrevious = false;
                buff.write(parseByte('0', previous));
            }
        }
        if (hasPrevious) {
            buff.write(parseByte('0', previous));
        }
        return buff.toByteArray();
    }

    private static byte parseByte(char high, char low) {
        return (byte) (read(high) * 16 + read(low));
    }

    private static int read(char c) {
        switch (c) {
            case 'a': case 'A': return 10;
            case 'b': case 'B': return 11;
            case 'c': case 'C': return 12;
            case 'd': case 'D': return 13;
            case 'e': case 'E': return 14;
            case 'f': case 'F': return 15;
            default: return c - '0';
        }
    }

    private static boolean isHexChar(char c) {
        return (((c >= '0') && (c <= '9'))
                || ((c >= 'a') && (c <= 'f'))
                || ((c >= 'A') && (c <= 'F')));
    }
}
