/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.me.utils;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import static com.sun.tck.me.utils.Converters.TO_ARRAY;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class CommandLineAdapter {
    public static interface CommandBundle {
        public void setCommand(String commad);
        public Object[] getObjects();
        public void init();
    }

    // TODO add two mode for the multiply objects
    //   - override - only the property for the first object is set (default)
    //   - all - properties for the all objects a set
    // TODO add subcommand option
    //   the subcommand is defined by the first parameter and
    //   after setting the commad the object returns the lis of the object
    //   being set.

    private String optionPrefix = "-";
    private Object[] commands;
    private HashMap<String, MethodDescriptor> methods = new HashMap();
    private MethodDescriptor setArguments;
    private boolean hasCommand;

    private static final Map<String, Converter> converters = Converters.createConverters();
    private String[] prefixes = new String[0];

    public CommandLineAdapter(Object... commands) {
        this(false, commands);
    }

    public CommandLineAdapter(boolean hasCommand, Object... commands) {
        this.hasCommand = hasCommand;
        this.commands = commands;
        for (Object command : commands) {
            for (Method meth : command.getClass().getMethods()) {
                Class[] params = meth.getParameterTypes();
                if (params.length == 1) {
                    registerMethod(command, meth);
                }
            }
        }
    }

    public void setOptionPrefix(String prefix) {
        optionPrefix = prefix;
    }

    public void setPrefixes(String... list) {
        this.prefixes = list;
    }

    private void registerMethod(Object command, Method meth) {
        String key = meth.getName().toLowerCase();
        Class[] params = meth.getParameterTypes();
        if (key.equals("setarguments") && params[0].equals(String[].class)) {
            this.setArguments = new MethodDescriptor(command, meth, TO_ARRAY);
        } else if ((key.startsWith("set") && params.length == 1)
                && (converters.get(params[0].getName()) != null)) {
            if (methods.containsKey(key)) {
                throw new IllegalArgumentException("More than one method for the same attribute:" + key);
            }
            Converter converter = converters.get(params[0].getName());
            methods.put(key, new MethodDescriptor(command, meth, converter));
        }
    }

    private void printHelp() {
        System.out.print("java " + this.commands[this.commands.length - 1].getClass().getName() + " [options]");
        if (this.setArguments == null) {
            System.out.println();
        } else {
            System.out.println(" ... ");
        }
        System.out.println("  Where options are:");
        for (String key : this.methods.keySet()) {
            System.out.println("    -" + methods.get(key).getMethod().getName().substring(3));
        }
    }

    private boolean isSingle(String arg) {
        for (String prefix : this.prefixes) {
            if (arg.startsWith(prefix)) {
                return true;
            }
        }
        return false;
    }

    public synchronized void init(String... args) {
        if (args == null) {
            return;
        }
        try {
            boolean isParamsEnded = false;
            int start = (hasCommand && args.length > 0 && (args[0].charAt(0) != '-')) ? 1 : 0;
            for (int i = start; i < args.length; i++) {
                if (isParamsEnded || isSingle(args[i])) {
                    addDefaultArgument(args[i]);
                } else if (args[i].equals("-h") || args[i].equals("-help")
                           ||args[i].equals("-?") || args[i].equals("--help")) {
                    printHelp();
                    System.exit(0);
                } else if (args[i].equals(optionPrefix + optionPrefix)) {
                    isParamsEnded = true;
                } else if (isParamsEnded || (!args[i].startsWith(optionPrefix))) {
                     addDefaultArgument(args[i]);
                } else {
                    setValue(args[i], args[i + 1]);
                    i++;
                }
            }
            invokeSetters();
            
        } catch (Exception e) {
            throw new IllegalArgumentException("Can not set value", e);
        } finally {
            clearMethodDescriptors();
        }
    }

    private void addDefaultArgument(String value) {
        if (this.setArguments == null) {
            throw new IllegalArgumentException("The default argument list is not supported");
        }
        this.setArguments.addValue(value);
    }

    private void setValue(String name, String value) {
        String methName = ("set" + name.substring(1)).toLowerCase();
        MethodDescriptor meth = methods.get(methName);
        if (meth == null) {
            throw new IllegalArgumentException("Unknown parameter:" + name);
        }
        meth.addValue(value);
    }

    private void invokeSetters() throws Exception {
        for (String methName : methods.keySet()) {
            MethodDescriptor meth = methods.get(methName);
            meth.invoke();
        }
        if (setArguments != null) {
            setArguments.invoke();
        }
    }

    private void clearMethodDescriptors() {
        for (String methName : methods.keySet()) {
            MethodDescriptor meth = methods.get(methName);
            meth.clear();
        }
        if (setArguments != null) {
            setArguments.clear();
        }
    }
}

class MethodDescriptor {
    private boolean isValid = true;
    private Object object;
    private Method method;
    private Converter converter;
    private ArrayList<String> values;

    public MethodDescriptor(Object object, Method method, Converter converter) {
        this.object = object;
        this.method = method;
        this.converter = converter;
    }

    public boolean isValid() {
        return isValid;
    }

    public void invalidate() {
        isValid = false;
    }

    public void addValue(String value) {
        if (values == null) {
            values = new ArrayList<String>();
        }
        values.add(value);
    }

    public Object invoke() throws Exception {
        if (converter == null) {
            return method.invoke(object);
        }
        if (values != null) {
            return method.invoke(object, converter.convert(values));
        }
        return null;
    }
    
    public Method getMethod() {
        return method;
    }
    
    public void clear() {
        values = null;
    }
}
