/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.me.utils;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class CommandLine {
    private User user = new User(User.INTERACTIVE);
    private Map<String,Converter> converters = Converters.createConverters();
    private Map<String,MethodDescriptor> methods  = new HashMap<String, MethodDescriptor>();
    private boolean isBatch;
    private String[] args;


    public CommandLine(Object... list) {
        this.converters = Converters.createConverters();
        setBatch(true);
        converters.put(String[].class.getName(), new SingleArgumentConverter() {

            @Override
            public Object convertElement(String value) throws Exception {
                return (value == null) ? null : value.split("[ \n\t]+");
            }
        });

        this.methods  = new HashMap<String, MethodDescriptor>();
        for (Object current : list) {
            for (Method meth : current.getClass().getMethods()) {
                Class[] params = meth.getParameterTypes();
                Converter converter = null;
                if (params.length > 1) {
                    continue;
                } else if ((params.length == 1)
                           && (converter = converters.get(params[0].getName())) == null) {
                        continue;
                }
                String name = meth.getName();
                MethodDescriptor descr  = methods.get(name);
                if (descr != null) {
                    descr.invalidate();
                    continue;
                } else {
                    descr = new MethodDescriptor(current, meth, converter);
                    methods.put(name, descr);
                }
            }
        }
    }

    public boolean isBatch() {
        return isBatch;
    }

    public void setBatch(boolean isBatch) {
        this.isBatch = isBatch;
        if (isBatch) {
            setUser(User.NOT_INTERACTIVE);
        }
    }

    public void setArguments(String[] args) {
        this.args = args;
    }

    public void setUser(User.UserConfirmation user) {
        this.user = new User(user);
    }

    public boolean processLine(String line) {
        line = line.trim();
        String cmd;
        String args;
        int pos = line.indexOf(' ');
        if (pos < 0) {
            cmd = line;
            args = null;
        } else {
            cmd = line.substring(0, pos);
            args = line.substring(pos + 1);
        }
        try {
            Object val = execute(cmd, args);
            if (val != null) {
                user.log(">> " + val);
            }
            return true;
        } catch (Throwable t) {
            if (t instanceof InvocationTargetException) {
                t = ((InvocationTargetException)t).getTargetException();
            }
            user.log(t);
            return user.isYes("Continue?");
        }
    }

    public Object execute(String name, String args) throws Exception {
        name = name.replace('-', '_');
        MethodDescriptor desr = methods.get(name);
        if ((desr == null) || !desr.isValid()) {
            throw new IllegalArgumentException("Unknown command:" + name);
        }
        desr.clear();
        desr.addValue(args);
        return desr.invoke();
    }
    private static String joint(String[] args) {
        StringBuffer retVal = new StringBuffer();
        String delim = "";
        for (String arg : args) {
            retVal.append(delim).append(arg);
            delim = " ";
        }
        return retVal.toString();
    }
    public void run() throws Exception {
        if (isBatch) {
            processLine(joint(args));
            return;
        }
        BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
        String line;
        System.out.print("# ");
        while ((line = in.readLine()) != null) {
            if (line.equals("")) {
                System.out.print("# ");
                continue;
            } else if (line.indexOf("<<") >= 0) {
                return;
            } else if (!processLine(line)) {
                return;
            }
            System.out.print("# ");
        }
    }
}
