/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.me.tools.fs;

import com.sun.tck.me.utils.Closables;
import com.sun.tck.me.utils.Utils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class ZipFileSystem implements FileSystem {
    private File source;
    private File destination;
    private LinkedHashMap<String, ByteArrayOutputStream> written = new LinkedHashMap<String, ByteArrayOutputStream>();
    private LinkedHashMap<String, ZipEntry> read = new LinkedHashMap<String, ZipEntry>();
    private Closables manager = new Closables();
    private ZipFile zip;
    private boolean verbose = false;
    private String pwd = "/";

    public ZipFileSystem(File source, File desctination) throws IOException {
        this.source = source;
        this.destination = desctination;
        if (source != null) {
            zip = new ZipFile(source);
            for (Enumeration<? extends ZipEntry> e = zip.entries(); e.hasMoreElements();) {
                ZipEntry entry = e.nextElement();
                read.put(entry.getName(), entry);
            }
        }
    }

    private static boolean isIn(String path, String[] prefixes) {
        if ((prefixes == null) || (prefixes.length == 0)) {
            return true;
        }
        for (String current : prefixes) {
            if (path.startsWith(current)) {
                return true;
            }
        }
        return false;
    }

    public static void unzip(File zipFile, FileSystem fs, String... prefixes) throws IOException {
        ZipFile zip = new ZipFile(zipFile);
        for (Enumeration<? extends ZipEntry> e = zip.entries(); e.hasMoreElements();) {
            ZipEntry entry = e.nextElement();
            String path = entry.getName();
            if (isIn(path, prefixes)) {
                OutputStream out = fs.openToWrite(path);
                Utils.copyData(zip.getInputStream(entry), out);
                out.close();
            }
        }
        zip.close();
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public InputStream openToRead(String path) throws IOException {
        path = getAbsPath(path);
        if (written.containsKey(path)) {
            return new ByteArrayInputStream((written.get(path).toByteArray()));
        } else if (read.containsKey(path)) {
            return manager.add(zip.getInputStream(read.get(path)));
        } else {
            throw new FileNotFoundException("No resource for: " + path);
        }
    }

    @Override
    public OutputStream openToWrite(String path) throws IOException {
        path = getAbsPath(path);
        ByteArrayOutputStream retVal = new ByteArrayOutputStream();
        read.remove(path);
        written.put(path, retVal);
        return retVal;
    }

    private String getAbsPath(String path) {
        path = (path.startsWith("/") ? path : pwd + path);
        return path.substring(1);
    }

    @Override
    public FileType getType(String path) {
        if (written.containsKey(path)) {
            return FileType.FILE;
        }
        ZipEntry e = read.get(path);
        if (e == null) {
            return FileType.DOES_NOT_EXIST;
        } else {
            return (e.isDirectory() ? FileType.DIR : FileType.FILE);
        }
    }

    @Override
    public String[] list(String path, Filter filter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    private void log(String msg) {
        if (verbose) {
            System.out.println(msg);
        }
    }

    public List<String> listAllFiles() {
        ArrayList<String> retVal = new ArrayList<String>();
        retVal.addAll(read.keySet());
        retVal.addAll(written.keySet());
        return retVal;
    }

    @Override
    public void close() throws IOException {
        Closables cls = new Closables();
        this.manager.close();
        ZipOutputStream out = new ZipOutputStream(new FileOutputStream(destination));

        try {
            for (String path : read.keySet()) {
                byte[] data;
                if (written.containsKey(path)) {
                    data = written.get(path).toByteArray();
                    log("  -  Changed: " + path);
                    written.remove(path);
                } else {
                    data = Utils.readBytesFromStream(zip.getInputStream(read.get(path)));
                }
                addZipEntry(out, path, data);
            }
            for (String path : written.keySet()) {
                addZipEntry(out, path, written.get(path).toByteArray());
                log("  -  Added: " + path);
            }
            out.finish();
            out.close();
        } catch (Exception e) {
            throw new IOException("Can not update zip file:" + destination.getAbsolutePath());
        } finally {
            cls.close();
        }
    }

    private void addZipEntry(ZipOutputStream out, String path, byte[] data) throws IOException {
        ZipEntry entry = new ZipEntry(path);
        CRC32 crc = new CRC32();
        crc.update(data, 0, data.length);
        entry.setSize(data.length);
        entry.setCrc(crc.getValue());
        out.putNextEntry(entry);
        out.write(data);
        out.closeEntry();
    }

    public Iterable<String> namesFromSource() {
        return read.keySet();
    }

    @Override
    public void switchTo(String path) {
        this.pwd = path.endsWith("/") ? path : path + "/";
    }
}
