/*
 * %W% %E%
 * @Copyright
 */

package com.sun.tck.jc.javatest;

import com.sun.javacard.cjck.invoke.CardServicesPool;
import com.sun.javatest.DefaultTestRunner;
import com.sun.javatest.TestDescription;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import com.sun.javacard.globalimpl.ServiceValidator;
import com.sun.javatest.Status;
import com.sun.javatest.TestResult;
import com.sun.javatest.WorkDirectory;
import com.sun.javatest.util.I18NResourceBundle;
import java.io.PrintWriter;

/**
 * This TestRunner implementation puts tests in a queue and allows
 * rerun tests by putting them to the tail of the queue.
 *
 * @author Mikhail Smirnov
 */
public class QueueTestRunner extends DefaultTestRunner {
        
    final TestsCompletionMonitor testCompletionMonitor = new TestsCompletionMonitor();
    private Queue<TestDescription> testQueue;
    private int startedCount = 0;
    private boolean started = false;
    private static QueueTestRunner currentTestRunner;
    private static I18NResourceBundle i18n = I18NResourceBundle.getBundleForClass(DefaultTestRunner.class);

    public static QueueTestRunner createTestRunner() {
        currentTestRunner = new QueueTestRunner();
        return currentTestRunner;
    }
    
    private QueueTestRunner() {
    }

    public static QueueTestRunner getTestRunner() {
        return currentTestRunner;
    }
    
    @Override
    public synchronized boolean runTests(Iterator itrtr) throws InterruptedException {
        //Before running tests we should ensure that there are available
        //CardService instances that can process tests
        try {
            CardServicesPool.checkState();
        } catch (Exception ex) {
            setErrorStatus(itrtr, ex);
            return false;
        }
        //Initializing static fields that later used for CardServices validation
        ServiceValidator.setWorkingDir(getWorkDirectory().getPath());
        ServiceValidator.setTSRoot(getTestSuite().getRootDir().getPath());
        
        testQueue = putTestsToQueue(itrtr);
        return super.runTests(new Iterator<TestDescription>() {

            public TestDescription next(){
                return testQueue.poll();
            }

            /*
             * To tell that there are no more tests to run, we need to make
             * sure that queue is empty, and that all tests that started are completed.
             * Otherwise some failed test could be put back to queue.
             */
            public boolean hasNext(){
                /*System.out.println("In hasNext(): testQueue.isEmpty="
                        + testQueue.isEmpty() + ", startedCount=" + startedCount
                        + ", started = " + started); //FIXME: debug output */
                if (!testQueue.isEmpty()) {
                    return true;
                } else if (testCompletionMonitor.canContinue()) {
                    return false;
                } else {
                    synchronized(testCompletionMonitor) {
                        while(!testCompletionMonitor.canContinue()) {
                            try {
                                /*System.out.println("Tests queue is empty: need "
                                        + "to wait for completion of executing tests"); //FIXME: debug output
                                 */
                                testCompletionMonitor.wait();
                            } catch (InterruptedException ex) {
                            }
                        }
                        return !testQueue.isEmpty();
                    }
                }
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }

        });
    }

    /**
     * Sets error status to all tests given by an iterator
     * passed as a parameter with details provided by Throwable object
     * passed as another parameter
     */
    protected void setErrorStatus(Iterator itrtr, Throwable ex) {
        Status s = Status.error(ex.getLocalizedMessage());
        WorkDirectory workDir = getWorkDirectory();
        while (itrtr.hasNext()) {
            TestDescription td = (TestDescription)itrtr.next();
            TestResult tr;

	    tr = new TestResult(td);
	    TestResult.Section trs = tr.createSection(i18n.getString("dtr.details"));
	    PrintWriter pw = trs.createOutput(i18n.getString("dtr.stackTrace"));
	    ex.printStackTrace(pw);
	    pw.close();
	    tr.setStatus(s);

            try {
                tr.writeResults(workDir, getBackupPolicy());
                notifyFinishedTest(tr);
            }
            catch (Exception e) {
                workDir.log(i18n, "dtr.unexpectedThrowable",
                    new Object[] {td.getRootRelativeURL(), e});
            }
        }
    }

    /**
     * Concurrency should be at least equal to the number of available card terminals
     */
    @Override
    public int getConcurrency() {
        return CardServicesPool.getCapacity();
    }

    /**
     * Creates and fills initial queue of tests descriptions
     * @param itrtr reference to tests descriptions iterator
     * @return queue of tests descriptions
     */
    protected Queue<TestDescription> putTestsToQueue(Iterator itrtr) {
        Queue<TestDescription> queue = new ConcurrentLinkedQueue<TestDescription>();
        while (itrtr.hasNext()) {
            TestDescription td = (TestDescription)itrtr.next();
            RerunnableTestDescription rtd = new RerunnableTestDescription(td);
            queue.add(rtd);
        }
        return queue;
    }

    /**
     * Returns test back to the queue for re-running
     * @param td Test Description of a test being returned
     */
    public void returnToQueue(TestDescription td) {
        ((RerunnableTestDescription) td).setRerun();
        synchronized(testCompletionMonitor) {
            testQueue.add(td);
            testCompletionMonitor.notify();
        }
    }

    /**
     * This method is called to signal that some test is started
     */
    public void testStarted() {
        started = true;
        startedCount++;
    }

    /**
     * This method is called to signal that some test is completed
     */
    public void testCompleted() {
        startedCount--;
        synchronized(testCompletionMonitor) {
            testCompletionMonitor.notify();
        }
    }

    class TestsCompletionMonitor {
        //we should wait if testQueue is empty, tests execution started
        //and not all started tests are completed.
        boolean canContinue() {
            return !testQueue.isEmpty() || (started && startedCount == 0);
        }
    }
}
