/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.terminal;

import com.sun.javacard.cjck.userinterface.AppletID;
import com.sun.javacard.cjck.userinterface.AppletProperties;
import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.javacard.cjck.userinterface.FatalException.Scope;
import com.sun.tck.bvtool.etsi.security.KeyDescr;
import com.sun.tck.bvtool.etsi.security.KeyType;
import com.sun.tck.me.utils.User;
import com.sun.tck.me.utils.Utils;
import java.io.Serializable;
import java.security.GeneralSecurityException;
import java.util.ArrayList;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class Card {
    private static final int MAX_COUNTER = 0xFFFF;
    public static final int DEFAULT_KEY = 1;
    static final int NUMBER_OF_KEYS = 16;
    public static final int NUMBER_OF_GSM_KEYS = 5;
    public int lowCounter;
    public int highCounter;
    public int counterStep = 16;
    public boolean isInitalized;
    private int gsmKeyOffset = CardStore.DEFAULT_GSM_KEYS_OFFSET;
    private User user = new User(User.NOT_INTERACTIVE);

    CardState state;

    private static SecurityLevel[] levels = new SecurityLevel[] {
        new SecurityLevel(Card.CounterMode.NO_COUNTER, false),        //#0
        new SecurityLevel(Card.CounterMode.NO_COUNTER, true),         //#1
        new SecurityLevel(Card.CounterMode.NO_CHECKING, false),       //#2
        new SecurityLevel(Card.CounterMode.NO_CHECKING, true),        //#3
        new SecurityLevel(Card.CounterMode.PROCESS_IF_NEXT, false),   //#4
        new SecurityLevel(Card.CounterMode.PROCESS_IF_NEXT, true),    //#5
        new SecurityLevel(Card.CounterMode.PROCESS_IF_HIGHER, false), //#6
        new SecurityLevel(Card.CounterMode.PROCESS_IF_HIGHER, true)   //#7
    };

    public void notifyRegisteredCapFile(AppletID packageID, AppletProperties[] applets) {
        String aidRep = Utils.canonize(packageID.getBytes());
        if (!this.state.installedCapFiles.containsKey(aidRep)) {
            this.state.installedCapFiles.put(aidRep, toStringList(applets));
        }
        this.state.flush();
    }

    public void notifyDeletedCapFile(AppletID aid) {
        String aidRep = Utils.canonize(aid.getBytes());
        if (this.state.installedCapFiles.containsKey(aidRep)) {
            this.state.installedCapFiles.remove(aidRep);
        }
        this.state.flush();
    }

    public void setUser(User user) {
        this.user = (user != null) ? user : new User(User.NOT_INTERACTIVE);
    }

    public void logCardInfo() {
        user.log("COUNTER=" + getCounter());
        user.log("SECURITY-LEVEL=" + levels[state.currentSecurityLevel]);
        user.log("KEY-VERSION=" + state.keyVersion);
    }

    static class Pair implements Serializable {
        private static final long serialVersionUID = -2627281906843010373L;

        transient private KeyDescr KID;
        transient private KeyDescr KIC;

        private int counter;

        public Pair(KeyDescr KID, KeyDescr KIC, int counter) {
            this.KID = KID;
            this.KIC = KIC;
            this.counter = counter;
        }

        public KeyDescr getKID() {
            return KID;
        }

        public KeyDescr getKIC() {
            return KIC;
        }

        @Override
        public String toString() {
            return "(Pair " + "KID=" + KID + " KIC=" + KIC + " counter=" + counter + ')';
        }
    }

    Card() {
        state = new CardState();
        this.state.keyVersion = DEFAULT_KEY;
    }

    public Card(CardState state, String iccid, int keyVersion, String KID, String KIC) throws GeneralSecurityException {
        this.state = (state == null) ? new CardState() : state;
        this.state.iccid = iccid;
        this.state.keyVersion = keyVersion;
        addKey(KeyType.TRIPLE_DES_CBC_2_OF_3, state.keyVersion, KID, KIC);
    }

    public Card(CardState state, String iccid, String[] keys)
            throws GeneralSecurityException {
        this.state = (state == null) ? new CardState() : state;
        this.state.keyVersion = DEFAULT_KEY;
        configure(iccid, keys);
    }

    final void configure(String iccid, String[] keys) throws GeneralSecurityException {
        this.state.iccid = iccid;
        for (int id = 1; id < NUMBER_OF_GSM_KEYS; id++) {
            addKey(KeyType.TRIPLE_DES_CBC_2_OF_3, id,
                    keys[gsmKeyOffset + (id - 1) * 3 + 1],
                    keys[gsmKeyOffset + (id - 1) * 3]);
        }
    }

    public void reset() {
        state.reset();
    }

    public int changeKeyVersion() throws FatalException {
        int newKeyVersion = state.keyVersion + 1;
        if ((newKeyVersion >= NUMBER_OF_GSM_KEYS)
                || (state.keys[newKeyVersion] == null)) {
            throw new FatalException(Scope.Card,
                    "Number of the supported keys has been exceed");
        } else {
            state.keyVersion++;
        }
        lowCounter = 0;
        highCounter = 0xFFF0;
        user.log("Change-key-version-to:" + state.keyVersion);
        return state.keyVersion;
    }

    public void increaseCounter() throws FatalException {
        CounterMode mode = levels[state.currentSecurityLevel].counterMode;
        if (mode == CounterMode.PROCESS_IF_HIGHER) {
            lowCounter = getCounter();
            setCounter(lowCounter + counterStep);
            return;
        }
        if (highCounter == 0) {
            highCounter = MAX_COUNTER + 1;
        }
        lowCounter = getCounter();
        checkCountersRange();
        int newCounter = (lowCounter + highCounter)/2 - 1;
        if (newCounter == getCounter()) {
            newCounter++;
        }
        setCounter(newCounter);
        user.log("Increase-Counter-to:" + (newCounter + 1));
    }

    public void decreaseCounter() throws FatalException {
        highCounter = getCounter();
        checkCountersRange();
        int newCounter = (lowCounter + highCounter)/2 - 1;
        if (newCounter == getCounter()) {
            newCounter--;
        }
        setCounter(newCounter);
        user.log("Decrease-Counter-to:" + (newCounter + 1));
    }

    private void checkCountersRange() throws FatalException {
        if ((highCounter - lowCounter) < 2) {
            changeKeyVersion();
        }
    }

//////////////////////////

    public byte[] getPhone() {
        return state.phone;
    }

    public static enum ReplyMode {
        NO_REPLY(0), REQUIRED(1), ONLY_IF_ERROR(2);
        private byte code;
        public byte code() {
            return code;
        }
        private ReplyMode(int code) {
            this.code = (byte)code;
        }
    }

    public static enum CounterMode {
        NO_COUNTER(0),
        NO_CHECKING(1),
        PROCESS_IF_HIGHER(2),
        PROCESS_IF_NEXT(3);
        private byte code;
        public byte code() {
            return code;
        }
        private CounterMode(int code) {
            this.code = (byte)code;
        }
    }

    public static enum SignatureMode {
        NONE(0),
        RedundancyCheck(1),
        CryptographicChecksum(2),
        DigitalSignature(3);

        private byte code;
        public byte code() {
            return code;
        }
        private SignatureMode(int code) {
            this.code = (byte)code;
        }
    }

    private static class SecurityLevel {
        private ReplyMode replyMode = ReplyMode.REQUIRED;
        private SignatureMode replySignature = SignatureMode.CryptographicChecksum;
        private boolean isReplyCiphered = false;
        boolean doEncryption;
        Card.CounterMode counterMode;
        public SecurityLevel(Card.CounterMode counterMode, boolean doEncryption) {
            this.doEncryption = doEncryption;
            this.counterMode = counterMode;
        }

        @Override
        public String toString() {
            return "(SecurityLevel doEncryption=" + doEncryption + " counterMode=" + counterMode + ')';
        }
    }

    public void increaseSecurityLevel() throws FatalException {
        state.currentSecurityLevel++;
        if (state.currentSecurityLevel >= levels.length) {
            throw new FatalException(Scope.Card, "There is no higher security level");
        }
        if (user != null) {
            user.log("Increasing security level to #" 
                    + state.currentSecurityLevel
                    + "=" + levels[state.currentSecurityLevel]);
        }
    }

    public void setCounter(int counter) {
        this.state.keys[state.keyVersion].counter = counter;
    }

    public int getCounter() {
        return this.state.keys[state.keyVersion].counter;
    }

    public void confirmCounter() throws FatalException {
        if (this.state.keys[state.keyVersion].counter >= MAX_COUNTER) {
            changeKeyVersion();
        }
        this.state.keys[state.keyVersion].counter++;
    }

    public boolean isReplyCiphered() {
        return levels[state.currentSecurityLevel].isReplyCiphered;
    }

    public final void addKey(KeyType type, int keyId, String KID, String KIC) throws GeneralSecurityException {
        KeyDescr kid = (KID == null) ? null : new KeyDescr(type, keyId, Utils.parse(KID));
        KeyDescr kic = (KIC == null) ? null : new KeyDescr(type, keyId, Utils.parse(KIC));
        int index = ((keyId & 0xF0) == 0) ? keyId : ((keyId >> 4) & 0x0F);
        int counter = (state.keys[index] == null) ? 1 : state.keys[index].counter;
        state.keys[index] = new Pair(kid, kic, counter);
    }

    public KeyDescr getKID() {
        return getKID(state.keyVersion << 4);
    }

    public KeyDescr getKID(int KID) {
        KID = 0x0F & (KID >> 4);
        Pair pair = state.keys[KID];
        return (pair == null) ? null : pair.KID;
    }

    public KeyDescr getKIC() {
        return getKIC(state.keyVersion << 4);
    }

    public KeyDescr getKIC(int KIC) {
        if (!levels[state.currentSecurityLevel].doEncryption) {
            return null;
        }
        KIC = 0x0F & (KIC >> 4);
        Pair pair = state.keys[KIC];
        return (pair == null) ? null : pair.KIC;
    }

    public void setKeyVersion(int version) {
        this.state.keyVersion = version;
    }

    public void init() throws Exception {
    }

    public void setSecurityLevel(int level) {
        this.state.currentSecurityLevel = level;
    }

    public short encodeSPI() {
        byte first = (byte)(levels[state.currentSecurityLevel].counterMode.code() << 3);
        KeyDescr key = getKID();
        first |=  (key != null) ? key.checksum.getCode() : 0;
        key = getKIC();
        first |= (key != null) ? 4 : 0;

        byte second = 0;
        second |= levels[state.currentSecurityLevel].replyMode.code();
        second |= levels[state.currentSecurityLevel].replySignature.code() << 2; // 01: PoR response with simple RC applied to it
        second |= isReplyCiphered() ? 0x10 : 0x00;
        second |= 0x01;
        // 0229
        return (short)((first << 8) | 0x21);//(second & 0xFF));
//        return (short)0x0221;
        // return 0x0229;
    }

    @Override
    public String toString() {
        return "(SESSION security-level#" + state.currentSecurityLevel + "=" + levels[state.currentSecurityLevel]
                + "\n    ICCID=" + this.state.iccid
                + "\n    counter=" + this.state.keys[state.keyVersion].counter
                + "\n    KID=" + this.getKID()
                + "\n    KIC=" + this.getKIC()
                + "\n    SPI=" + Integer.toBinaryString(encodeSPI()) + ")";
    }

    public void flush() {
        state.flush();
    }

    private static ArrayList<String> toStringList(AppletProperties[] list) {
        ArrayList<String> retVal = new ArrayList<String>();
        for (AppletProperties applet : list) {
            retVal.add(Utils.canonize(applet.getAID().getBytes()));
        }
        return retVal;
    }
}
