/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.tlv;

import com.sun.tck.me.utils.Utils;
import java.util.ArrayList;

/**
 * This class is used to encode a hierarchical TLV and LV structures.
 * It maintain the stack of the TLV/LV structures.
 * It will create a length fields automatically when the either LV structure is
 * ended or TLVEncoder is finalized.
 *
 * @author Maxim V. Sokolnikov
 */
public class TLVEncoder {

    private TLVBuffer out;

    /**
     * Creates a TLVEncoder with the given the initial array buffer.
     * The buffer will be expanded, when it is necessary.
     */
    public TLVEncoder(int length) {
        out = new TLVBuffer(new byte[length]);
    }

    private static class StackEntry {
        TLVBuffer length;
        String name;
        StackEntry(String name, TLVBuffer length) {
            this.name = name;
            this.length = length;
        }
    }

    private ArrayList<StackEntry> stack = new ArrayList<StackEntry>();

    /**
     * Starts a LV structure. The length fields will be created automatically,
     * when this LV is ended. The created LV structure pushed on stack.
     *
     * @param name name of the structure. It is used for verification purpose
     * only. The value can be null, which means that no verification is required.
     *
     * @throws EncodingException
     */
    public void startLV(String name) throws EncodingException {
       stack.add(new StackEntry(name, out.createChild(1)));
    }

    /**
     * Writes the tag with the given tag value and start a LV structure with null name.
     * It is equal to creation of the TLV structure
     *
     * @param tag
     * @throws EncodingException
     */
    public void startTag(int tag) throws EncodingException {
        startTag(tag, null);
    }

    /**
     * Writes the tag with the given tag value and start a LV structure with given name.
     * It is equal to creation of the TLV structure
     *
     * @param tag
     * @throws EncodingException
     */
    public void startTag(int tag, String name) throws EncodingException {
       out.write(tag);
       startLV(name);
    }

    /**
     * Invokes <code>end(String name)</code> with null name.
     */
    public void end() throws EncodingException {
        end(null);
    }

    /**
     * Ends TLV or LV structure. The method pushes the LV structure from stack
     * and creates the length fields for this structure.
     *
     * @throws EncodingException
     * <ul>
     *   <li>if the name is not null and does not match the name of the top LV structure.</li>
     *   <li>Stack is empty</li>
     * </ul>
     */
    public void end(String name) throws EncodingException {
        if (stack.isEmpty()) {
            throw new EncodingException("Stack is empty.");
        }
        StackEntry entry = stack.get(stack.size() - 1);
        if ((name != null) && !name.equals(entry.name)) {
            throw new EncodingException("Tag mismatch: Expected " + name
                    + " Found " + entry.name);
        }
        stack.remove(stack.size() - 1);
        TLVUtils.writeLength(entry.length, out.getPos() - entry.length.end());
    }

    /**
     * Returns the buffer for writing.
     * @return
     */
    public TLVBuffer out() {
        return out;
    }

    /**
     * Ends all LV structures on stack.
     * @throws EncodingException
     */
    public void close() throws EncodingException {
        for (int i = stack.size() - 1; i >= 0; i--) {
            end(null);
        }
    }

    /**
     * Ends all LV structures on stack and returns the created byte array
     * @return
     * @throws EncodingException
     */
    public byte[] toByteArray() throws EncodingException {
        close();
        byte[] retVal = new byte[out.getPos()];
        System.arraycopy(out.getBuffer(), 0, retVal, 0, retVal.length);
        return retVal;
    }

    @Override
    public String toString() {
        return "(TLVEncoder isFinalized=" + stack.isEmpty() + " data=" + Utils.canonize(out.getBuffer(), 0, out.getPos()) + ')';
    }


    public static void main(String[] args) throws Exception {
        TLVEncoder out = new TLVEncoder(8); // 8 is only minimal length
        out.startTag(0xf1); // starts TLV of the first level #1
          out.startTag(0xf2); // starts TLV of the second level #2
          out.out().write(0x01020304L, 4); // Writes four bytes from the given long
            out.startTag(0xE2); // starts TLV of the third level #3
            out.out().write(0x01020304L, 4); // Writes four bytes from the given long
            out.out().write(0x05); // writes single bytes
            out.end(); // ends TLV of the third level #3
          out.end();   // ends TLV of the second level #2
          out.startLV(null); // starts LV structure (without tag) #4
          out.out().write(0x1112131415L, 5); // Wites five bytes from the given long
          out.end();        // ends LV #4
          out.startTag(0xf3); // starts TLV of the second level #5
          out.out().write(new byte[] {(byte)0x21, (byte)0x22, (byte)0x23, (byte)0x24});
          out.end(); // ends TLV of the second level #5
        out.end(); // ends TLV of the second level #1
        System.out.println(Utils.canonize(out.toByteArray()));
    }
 }
