/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.tlv;

import com.sun.tck.bvtool.etsi.gsm.ETSITags;
import com.sun.tck.me.utils.Utils;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public abstract class TLV extends TreeNodeImpl implements Encodable {
    public static final byte UNDEFINED = (byte)0x00;
    private byte tag = UNDEFINED;
    private TLVPrototypeFactory factory;
    protected String name;
    protected ConstantBundle constants = new ConstantBundle(false, ETSITags.class);

    public TLV() {
        super(null, "TLV");
    }

    public void addByte(String name, int byteValue) {
        add(name, new TreeNode.Data(name, byteValue));
    }

    protected void setTagInternal(byte tag, String name) {
        this.tag = tag;
        if ((this.name == null) && (name == null)) {
            this.name = constants.find(tag);
        } else if (name != null) {
            this.name = name + "(0x" + Integer.toHexString(tag & 0xff) + ")";
        }
    }

    public void setTag(byte tag, String name) throws EncodingException {
        if ((this.tag != UNDEFINED) && (this.tag != tag)) {
            throw new EncodingException("Inconsistent. Expected tag: 0x"
                    + Integer.toHexString(this.tag & 0xff)
                    + " Found 0x" + Integer.toHexString(tag & 0xff));
        }
        setTagInternal(tag, name);
    }

    public TLVPrototypeFactory getFactory() {
        return factory;
    }

    public void setFactory(TLVPrototypeFactory factory) {
        this.factory = factory;
    }

    protected abstract void writeData(TLVBuffer buff) throws Exception;
    
    public byte getTag() {
        return tag;
    }

    public void encode(TLVBuffer buff) throws EncodingException {
        try {
            buff.write(tag);
            // remember the position of the length and reserve 1 byte for short length
            TLVBuffer lengthPointer = buff.createChild(1);

            writeData(buff);
            // encode the length
            TLVUtils.writeLength(lengthPointer, buff.getPos() - lengthPointer.end());
        } catch (EncodingException e) {
            throw e;
        } catch (Exception e) {
            throw new EncodingException("Encoding problem:", e);
        }
    }

    public abstract void decodeData(TLVPrototypeFactory processor, TLVBuffer in) throws Exception;

    public void decode(TLVPrototypeFactory processor, TLVBuffer in)
            throws EncodingException {
        try {
            tag = (byte)TLVUtils.readTag(in);
            int length = TLVUtils.readLength(in);
            String entry = Integer.toHexString(tag & 0xFF);
            decodeData(processor, in.createChild(length));
        } catch (EncodingException e) {
            throw e;
        } catch (Exception e) {
            throw new EncodingException("Can not parse TLV.", e);
        }
    }

    public String getName() {
        return (name == null) ? Utils.findName(tag, ETSITags.class) : name;
    }

    public byte[] toByteArray() throws EncodingException {
        TLVBuffer buff = new TLVBuffer(256000);
        this.encode(buff);
        byte[] retVal = new byte[buff.getPos()];
        System.arraycopy(buff.getBuffer(), 0, retVal, 0, retVal.length);
        return retVal;
    }

    @Override
    protected StringBuffer toStringHeader() {
        return new StringBuffer(getClass().getSimpleName()).append("{Tag=").append(getName());
    }

    @Override
    public Object value() {
        return this;
    }

}
