/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.gp;

import java.util.ArrayList;

/**
 * This class contains definitions of the SW codes according to GP 2.1.1
 * Specification.
 * 
 * @author Maxim V. Sokolnikov
 */
public class GPConstants {

    private static class Descr {
        int sw;
        int mask = 0xFFFF;
        String name;

        Descr(String name, int sw, int mask) {
            this.name = name;
            this.sw = sw;
            this.mask = mask;
        }
    }
    private static GPConstants defaultName;
    private static GPConstants load;
    private static GPConstants install;
    private static GPConstants tck;
    private static GPConstants delete;

    private ArrayList<Descr> list = new ArrayList<Descr>();
    private GPConstants[] parents;


    private GPConstants(GPConstants... parents) {
        this.parents = parents;
    }

    /**
     * Returns the name for the given SW code. If the current table does not
     * contain the definition for the code, then the parents are searched.
     */
    public String findName(int sw) {
        sw = sw & 0xFFFF;
        for (Descr descr : list) {
            if ((descr.mask & sw) == descr.sw) {
                return descr.name + "(0x" + Integer.toHexString(sw) + ")";
            }
        }
        for (GPConstants gp : parents) {
            String name = gp.findName(sw);
            if (name != null) {
                return name;
            }
        }
        return "Unknown(0x" + Integer.toHexString(sw) + ")";
    }

    private void add(int sw, String name, int mask) {
        this.list.add(new Descr(name, sw, mask));
    }

    private void add(int sw, String name) {
        add(sw, name, 0xFFFF);
    }

    /**
     * Returns a SW code table with the common GP and ISO7816-4 SW codes.
     */
    public static synchronized GPConstants getDefault() {
        if (defaultName != null) {
            return defaultName;
        }
        defaultName = new GPConstants(new GPConstants[0]);
        defaultName.add(0x9000, "ISO-7816-4 - SW OK");
        defaultName.add(0x6400, "GP 2.1.1 - No Specific Diagnosis");
        defaultName.add(0x6700, "GP 2.1.1 - Wrong length in Lc");
        defaultName.add(0x6881, "GP 2.1.1 - Logical channel not supported or is not active");
        defaultName.add(0x6982, "GP 2.1.1 - Security status not satisfied");
        defaultName.add(0x6985, "GP 2.1.1 - Conditions of use not satisfied");
        defaultName.add(0x6A86, "GP 2.1.1 - Incorrect P1 P2");
        defaultName.add(0x6D00, "GP 2.1.1 - Checking error: Invalid instruction");
        defaultName.add(0x6E00, "GP 2.1.1 - Checking error: Invalid class");
        defaultName.add(0x6100, "ISO-7816-4 - Normal Processing: SW2 encodes the number of data bytes still available", 0xFF00);
        defaultName.add(0x6200, "ISO-7816-4 - Warning processing: State of non-volatile memory is unchanged.", 0xFF00);
        defaultName.add(0x6300, "ISO-7816-4 - Warning processing: State of non-volatile memory has changed.", 0xFF00);
        defaultName.add(0x6400, "ISO-7816-4 - Execution error: State of non-volatile memory is unchanged", 0xFF00);
        defaultName.add(0x6500, "ISO-7816-4 - Execution error: State of non-volatile memory has changed", 0xFF00);
        defaultName.add(0x6600, "ISO-7816-4 - Execution error: Security-related issues", 0xFF00);
        defaultName.add(0x6700, "ISO-7816-4 - Checking error: Wrong length; no further indication");
        defaultName.add(0x6800, "ISO-7816-4 - Checking error: Functions in CLA not supported", 0xFF00);
        defaultName.add(0x6900, "ISO-7816-4 - Checking error: Command not allowed", 0xFF00);
        defaultName.add(0x6a00, "ISO-7816-4 - Checking error: Wrong parameters P1-P2", 0xFF00);
        defaultName.add(0x6b00, "ISO-7816-4 - Checking error: Wrong parameters P1-P2");
        defaultName.add(0x6c00, "ISO-7816-4 - Checking error: Wrong Le field; SW2 encodes the exact number of available data bytes", 0xFF00);
        defaultName.add(0x6f00, "ISO-7816-4 - Checking error: No precise diagnosis");
        return defaultName;
    }

    /**
     * Returns a SW code table for the GP install command.
     */
    public static synchronized GPConstants getInstall() {
        if (install != null) {
            return install;
        }
        install = new GPConstants(getDefault());
        install.add(0x6581, "GP 2.1.1 - Memory failure");
        install.add(0x6A88, "GP 2.1.1 - Referenced data not found");
        install.add(0x6A80, "GP 2.1.1 - Incorrect parameters in data field");
        install.add(0x6A84, "GP 2.1.1 - Not enough memory space");
        return install;
    }

    /**
     * Returns a SW code table for the GP install command.
     */
    public static synchronized GPConstants getLoad() {
        if (load != null) {
            return load;
        }
        load = new GPConstants(getDefault());
        load.add(0x6581, "GP 2.1.1 - Memory failure");
        load.add(0x6A84, "GP 2.1.1 - Not enough memory space");
        return load;
    }

    /**
     * Returns a SW code table for the GP delete command.
     */
    public static synchronized GPConstants getDelete() {
        if (delete != null) {
            return delete;
        }
        delete = new GPConstants(getDefault());
        delete.add(0x6581, "GP 2.1.1 - Memory failure");
        delete.add(0x6A88, "GP 2.1.1 - Referenced data not found");
        delete.add(0x6A82, "GP 2.1.1 - Application not found");
        delete.add(0x6A80, "GP 2.1.1 - Incorrect values in command data");
        return delete;
    }

    /**
     * Returns a SW code table with the TCK test execution codes.
     * It also contains the GP codes.
     */
    public static synchronized GPConstants getTck() {
        if (tck != null) {
            return tck;
        }
        tck = new GPConstants(getInstall(), getDelete());
        tck.add(0x9b00, "TCK - TestPassed");
        tck.add(0x9b00, "TCK - TestFailed", 0xFF00);
        tck.add(0x9c00, "TCK - TestUnexpectedError", 0xFF00);
        return tck;
    }
}
