/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.data.sms;

import com.sun.javacard.cjck.userinterface.CommandAPDU;
import com.sun.javacard.cjck.userinterface.FatalException;
import com.sun.tck.bvtool.terminal.Card;
import com.sun.tck.bvtool.etsi.data.CompactApduPacket;
import com.sun.tck.bvtool.etsi.gsm.ETSITags;
import com.sun.tck.bvtool.etsi.tlv.ConstructedTLV;
import com.sun.tck.bvtool.etsi.tlv.Encodable;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.RawTLV;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer;
import com.sun.tck.bvtool.etsi.tlv.TLV;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class SMS_PP_TLV extends ConstructedTLV {
    private byte[] phone = new byte[] {0x12, 0x34, 0x56, 0x78};
    private Encodable packet;
    private Date date;
    private boolean isConcatinated = false;
    private byte refNumber;
    private byte sequenceNumber;
    private byte numberOfMessages;

    StringBuffer tpduString = new StringBuffer("    TPDU={\n");

    public TLV createDeviceIdentities() {
        // 12.7
        // Device identities: Network -> SIM
        return new RawTLV(ETSITags.DEVICE_IDENTITIES_TAG, new byte[] {(byte)0x83, (byte)0x81});
    }
        // 12.1
        /* defined in GSM 11.11
        Value Definition
        0     Unknown
        1     International Number
        2     National Number
        3     Network Specific Number
        4     Subscriber Number
        6     Abbreviated Number
        7     Reserved
         */
    /*
        int TON = 0; // UNKNOWN
        int NPI = 0; // UNKNOWN
        byte[] data = new byte[1 + phone.length / 2 + phone.length % 2];
         // TON-NPI
        int pos = 0;
        data[pos] = (byte)(0x80 | (TON << 4) | NPI);
        int offset = ((phone.length % 2) == 0) ? 0 : -1;
        while (offset < phone.length) {
            int first = ((offset < 0) ? 0 : phone[offset]) & 0xF;
            int second = phone[offset + 1] & 0xF;
            data[pos++] = (byte)((first << 4) | second);
            offset += 2;
        }
        TLV address = new RawTLV(ETSITags.ADDRESS_TAG, data);
        return address;
    }
*/
    public TLV createTPDU() throws EncodingException {
        // 12.13
        // GSM 23.040, page 43, Chapter 9.2.2.1
        /*
Basic elements of the SMS-DELIVER type:
Abbr. Reference P1) R2) Description

TP-MTI TP-Message-Type-Indicator      M 2b    Parameter describing the message type.
TP-MMS TP-More-Messages-to-Send       M  b    Parameter indicating whether or not there are more messages to send
TP-RP TP-Reply-Path                   M  b    Parameter indicating that Reply Path exists.
TP-UDHI TP-User-Data-Header-Indicator O  b    Parameter indicating that the TP-UD field contains a Header
TP-SRI TP-Status-Report-Indication    O  b    Parameter indicating if the SME has requested a status report.

TP-OA TP-Originating-Address          M 2-12o Address of the originating SME.
TP-PID TP-Protocol-Identifier         M  o    Parameter identifying the above layer protocol, if any.
TP-DCS TP-Data-Coding-Scheme          M  o    Parameter identifying the coding scheme within the TP-User-Data.
TP-SCTS TP-Service-Centre-Time-Stamp  M 7o    Parameter identifying time when the SC received the message.
TP-UDL TP-User-Data-Length            M  I    Parameter indicating the length of the TP-User-Data field to follow.
TP-UD TP-User-Data                    O 3)

1) Provision; Mandatory (M) or Optional (O).
2) Representation; Integer (I), bit (b), 2 bits (2b), Octet (o), 7 octets (7o), 2-12 octets (2-12o).
3) Dependent on the TP-DCS.*/
        TLVBuffer out = new TLVBuffer(256);
        // 1 byte - TP-MTI, TP-MMS, TP-SRI, TP-UDHI, TPRP
        out.write(0x40); // there's header in TP-UD
        tpduString.append("        header=0x40\n");
        encodeTP_OA(out);
        encodeTP_PID(out);
        encodeTP_DCS(out);
        encodeTP_SCTS(out);
        encodeTP_UD_as_LV(out);
        byte[] data = new byte[out.getPos()];
        System.arraycopy(out.getBuffer(), 0, data, 0, data.length);
        return new RawTLV((byte)(ETSITags.SMS_TPDU_Tag | 0x80), data);
    }
    private void encodeTP_OA(TLVBuffer out) {
        // TP-OA  - 2-12 bytes
        out.write(0x088021436587L, 6);
        if (true) {
            return;
        }
        phone = (phone == null) ? new byte[0] : phone;
        out.write(phone.length * 2);
        tpduString.append("        TP_OA(length)=").append(phone.length * 2).append("\n");
        // 12.1
        /* defined in GSM 11.11
        Value Definition
        0     Unknown
        1     International Number
        2     National Number
        3     Network Specific Number
        4     Subscriber Number
        6     Abbreviated Number
        7     Reserved
         */
        int TON = 0; // UNKNOWN
        int NPI = 1; // GSM application
         // TON-NPI
        int ton_npi = 0x80 | (TON << 4) | NPI;
        out.write(0x80 | (TON << 4) | NPI);
        tpduString.append("        TP_OA(TPN/NPI)=0x").append(Integer.toHexString(ton_npi)).append("\n");
        tpduString.append("        TP_OA(phone)=");
        for (int i = 0; i < phone.length; i++) {
            this.writeDigitForGSM11_11S(out, phone[i], 16);
        }
        tpduString.append('\n');
    }

    // 1 byte
    private void encodeTP_PID(TLVBuffer out) {
        tpduString.append("        TP_PID=0x7F\n");
        out.write(0x7F); // TP-PID SIM data download
    }
    // 1 byte
    private void encodeTP_DCS(TLVBuffer out) {
        tpduString.append("        TP_DCS=0xF6\n");
        out.write(0xF6);
    }

    // 7 bytes
    private void encodeTP_SCTS(TLVBuffer out) {
        /*
        Calendar calendar = Calendar.getInstance();
        Date date = (this.date == null) ? new Date() : this.date;
        calendar.setTime(date);
        tpduString.append("        TP_SCTS(").append(date).append(")=");
        writeDigitForGSM11_11S(out, calendar.get(Calendar.YEAR), 10);
        writeDigitForGSM11_11S(out, calendar.get(Calendar.MONTH), 10);
        writeDigitForGSM11_11S(out, calendar.get(Calendar.DAY_OF_MONTH), 10);
        writeDigitForGSM11_11S(out, calendar.get(Calendar.HOUR_OF_DAY), 10);
        writeDigitForGSM11_11S(out, calendar.get(Calendar.MINUTE), 10);
        writeDigitForGSM11_11S(out, calendar.get(Calendar.SECOND), 10);
        writeDigitForGSM11_11S(out, 0, 10);
        tpduString.append('\n');*/
        for (int i = 0; i < 7; i ++) {
            out.write(0);
        }
                /*
                 *
                 Year	1	These semi-octets are in "Swapped Nibble" mode
Month	1
Day	1
Hour	1
Minute	1
Second	1
Timezone	1	Relation to GMT. One unit is 15min. If MSB=1, value is negative.                 */

    }

    private void encodeUDHL(TLVBuffer out) {
        if (isConcatinated) {
            out.write((sequenceNumber == 1) ? 0x07 : 0x05); //: UDHL
            out.write(0x00); // identifies this Header as a concatenation control header defined in TS 23.040 [3].
            out.write(0x03); // IEIDLa
            // IEDa 3 octets These octets contain
            out.write(refNumber); // the reference number,
            out.write(numberOfMessages); // total number of messages in the sequence
            out.write(sequenceNumber); // sequence number and
            if (sequenceNumber == 1) {
                out.write(0x70); // 0: IEI (Information Element Identifier) or 0700
                out.write(0x00); //: IEDL (Information Element Identifier Data Length)
            }
        } else {
            out.write(0x02); //: UDHL
            out.write(0x70); // 0: IEI (Information Element Identifier) or 0700
            out.write(0x00); //: IEDL (Information Element Identifier Data Length)
        }
    }

    private void encodeTP_UD_as_LV(TLVBuffer out) throws EncodingException {

        TLVBuffer TP_UDL = out.createChild(1);
        encodeUDHL(out);
        packet.encode(out);
        TP_UDL.write(out.getPos() - TP_UDL.end());
    }

    public void init() throws EncodingException {
        addIfNonNull("device-identities", createDeviceIdentities());
        addIfNonNull("address", createAddress());
        addIfNonNull("tpdu", createTPDU());
    }

    public TLV createAddress() {
        return new RawTLV((byte)0x86, "80 89674523");
    }

    public void addIfNonNull(String name, TLV tlv) {
        if (tlv != null) {
            add(name, tlv);
        }
    }

    private void writeDigitForGSM11_11S(TLVBuffer out, int value, int rem) {
        value = value % (rem*rem);
        int val = (value % rem) << 4 + value / rem;
        String hex = Integer.toHexString(val & 0xFF);
        tpduString.append((hex.length() == 1) ? "0" : "").append(hex);
        out.write(val);
    }
    /*

 Description           Section  M/O  Min  Length
 SMS-PP download tag     13.1    M    Y      1
 Length (A+B+C)           -      M    Y   1 or 2
 Device identities       12.7    M    Y      A
 Address                 12.1    O    N      B
 SMS TPDU (SMS-DELIVER)  12.13   M    Y      C
     * 
     */

    public SMS_PP_TLV() throws Exception {
        setTag(ETSITags.SMS_PP_DOWNLOAD_TAG, null);
    }

    public SMS_PP_TLV(Encodable packet, boolean doInit) throws EncodingException {
        setTag(ETSITags.SMS_PP_DOWNLOAD_TAG, null);
        this.packet = packet;
        if (doInit) {
            init();
        }
    }

    private static int refCount = 0;

    private static synchronized byte createReferenceNumber() {
        return (byte)++refCount;
    }

    public static List<SMS_PP_TLV> create(Card card, int length, CommandAPDU... apdus) throws EncodingException, FatalException {
        CompactApduPacket packet = new CompactApduPacket(apdus);
        packet.init(card);
        packet.setCompactMode(true);
        byte[] data = packet.toByteArray();
        if (data.length <= length) {
            SMS_PP_TLV sms = new SMS_PP_TLV(new Data("compact-apdu", data), false);
            sms.phone = card.getPhone();
            sms.init();
            return Collections.singletonList(sms);
        } else {
            ArrayList<SMS_PP_TLV> retVal = new ArrayList<SMS_PP_TLV>();
            int count = data.length / length;
            count += ((data.length % length) > 0) ? 1 : 0;
            int refNumber = createReferenceNumber();
            for (int i = 0; i < count; i++) {
                int offset = i * length;
                int toDo = Math.min(data.length - offset, length);
                Data part = new Data("data-" + i, data, offset, toDo);
                SMS_PP_TLV sms = new SMS_PP_TLV(part, false);
                sms.isConcatinated = true;
                sms.refNumber = (byte)refNumber;
                sms.numberOfMessages = (byte)(count);
                sms.sequenceNumber = (byte)(i + 1);
                sms.phone = card.getPhone();
                sms.init();
                retVal.add(sms);
            }
            return retVal;
        }
    }

    public String getShortDescription() {
        return isConcatinated
                ? ("(concatenated-short-message referenceNumber=" + refNumber
                   + " sequenceNumber=" + sequenceNumber
                   + " numberOfMessages=" + numberOfMessages + ")")
                : "(command-packet)";

    }
}
