/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.data;

import com.sun.tck.bvtool.terminal.Card;
import com.sun.tck.bvtool.etsi.tlv.TLV;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer.Position;
import com.sun.tck.bvtool.etsi.tlv.TLVPrototypeFactory;
import com.sun.tck.bvtool.etsi.tlv.Encodable;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.TLVUtils;
import com.sun.tck.bvtool.etsi.tlv.TreeNode;
import java.security.GeneralSecurityException;
import static com.sun.tck.me.utils.Tracer.trace;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public abstract class SecuredPacketTLV extends TLV {

    @Override
    protected final void writeData(TLVBuffer buff) throws Exception {
    }

    @Override
    public final void decodeData(TLVPrototypeFactory processor, TLVBuffer in) throws Exception {
    }

    private static enum State {
        CREATED,
        ENCODED,
        SIGNED,
        ENCRYPTED,
        DECRYPTED,
        VERIFYED,
    }

    private State state = State.CREATED;

    /**
     * The header represents the packed header as described in ETSI TS 102 225.
     * <p>
     * The lifecycle of the header in case of encoding is following
     * <ol>
     *   <li><code>init(Session session, long counter)</code></li>
     *   <li><code>setTAR, getTAR</code></li>
     *   <li><code>TLV.encode()</code> it is assumed that the header instance
     *             is also the TLV object. During encoding the header should
     *             keep the pointers on the checksum and the padding byte,
     *             because the value of these fields can not be calculated at
     *             this time.</li>
     *   <li><code>setData(TLVBuffer buff)</code> At this point the padding
     *             should be calculated.</li>
     *   <li><code>padding()</code></li>
     *   <li>packet tlv is created including the padding bytes being added,
     *             but not encrypted.</li>
     *   <li>createChecksum(TLVBuffer buff)</li>
     *   <li>the secured data in packet TLV is encrypted.<li>
     * </ol>

     */
    public static interface HeaderEncoder {

        public void setBuffer(TLVBuffer fullTLV, Position startBuffer, Position value);

        public void setData(byte[] data);

        public void write(TLVBuffer out) throws EncodingException;

        public void sign() throws GeneralSecurityException, EncodingException;

        public void encrypt() throws GeneralSecurityException, EncodingException;

    }

    public static interface Header extends TreeNode {

        /**
         * Initializes the header object.
         * @param session
         * @param counter
         */
        public void init(Card session);

        public int getTAR();

        public void setTAR(int tar);

        public HeaderDecoder decoder();
        public HeaderEncoder encoder();
    }

    public static interface HeaderDecoder {

        /**********************************************************/
        /*          Decoding methods                              */
        /**********************************************************/
        public void setBuffer(TLVBuffer fullTLV, Position startBuffer, Position value);
        public void parseHeader() throws EncodingException;

        public void decrypt() throws GeneralSecurityException;

        public void parseDecryptedHeader() throws EncodingException;

        public void verify() throws GeneralSecurityException, EncodingException;

        public TLVBuffer getSecuredData();
    }

    public SecuredPacketTLV() {
        header = createHeader();
        add("command-header", header);
    }

    protected Header header;
    protected Card card;
    protected boolean isCompact = true;

    protected abstract Header createHeader();

    public void setCompactMode(boolean isCompact) {
        this.isCompact = isCompact;
    }
    public void init(Card card) {
        this.card = card;
        header.init(card);
    }

    @Override
    public void decode(TLVPrototypeFactory processor, TLVBuffer in) throws EncodingException {
        Position start = in.createPosition();
        HeaderDecoder decoder = header.decoder();
        Position mark = in.createPosition();
        int length;
        if (isCompact) {
            length = (int)in.read(2);
        } else {
            setTag((byte) TLVUtils.readTag(in), null);
            length = TLVUtils.readLength(in);
        }
        Position dataStart = in.createPosition();
        TLVBuffer current = in.createChild(mark, length + dataStart.pos - mark.pos);
        trace("ENCRYPTED-STEP:" + current);
        decoder.setBuffer(current, start, dataStart);
        decoder.parseHeader();
        try {
            decoder.decrypt();
            trace("DECRYPTED-STEP:" + current);
            state = State.ENCRYPTED;
            decoder.parseDecryptedHeader();
            decoder.verify();
            TLVBuffer secured = decoder.getSecuredData();
            decodeSecuredData(processor, secured);
            state = State.VERIFYED;
        } catch (GeneralSecurityException e) {
            throw new EncodingException("security configuration error", e);
        } catch (Exception e) {
            throw new EncodingException("Unknown Exception", e);
        }
    }

    protected void decodeSecuredData(TLVPrototypeFactory processor,
            TLVBuffer securedData) throws Exception {
        securedData.reset();
        while (securedData.available() > 0) {
            TLV child = processor.create(TLVUtils.lookupTag(securedData));
            child.decode(processor, securedData);
            add(child);
        }
    }

    protected void writeSecuredData(TLVBuffer buff) throws EncodingException {
        for (TreeNode node : childNodes().values()) {
            if (node instanceof Encodable) {
                ((Encodable)node).encode(buff);
            }
        }
    }

    @Override
    public void encode(TLVBuffer buff) throws EncodingException {
        HeaderEncoder encoder = header.encoder();
        Position startBuffer = buff.createPosition();
        if (!isCompact) {
            buff.write(getTag());
        }
        TLVBuffer lengthBuff = buff.createChild(isCompact ? 2 : 1);
        Position startBody = buff.createPosition();

        encoder.setBuffer(buff, startBuffer, startBody);
//        Position startData = buff.createPosition();

        TLVBuffer securedBuff = new TLVBuffer(1024);
        writeSecuredData(securedBuff);
        byte[] securedData = new byte[securedBuff.getPos()];
        System.arraycopy(securedBuff.getBuffer(), 0, securedData, 0, securedData.length);

        encoder.setData(securedData);
        encoder.write(buff);
        trace("ENCODED       :" + buff);
        int length = buff.getPos() - startBody.pos;
        if (isCompact) {
            lengthBuff.write(length, 2);
        } else {
            TLVUtils.writeLength(lengthBuff, length);
        }
        state = State.ENCODED;
        try {
            encoder.sign();
            state = State.SIGNED;
            trace("SIGNED-STEP   :" + buff);
            encoder.encrypt();
            state = State.ENCRYPTED;
            trace("ENCRYPTED-STEP:" + buff);
        } catch (GeneralSecurityException e) {
            throw new EncodingException("security configuration error", e);
        }
    }
}
