/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.data;

import com.sun.tck.me.utils.Tracer;
import com.sun.tck.bvtool.terminal.Card;
import com.sun.tck.bvtool.etsi.data.SecuredPacketTLV.HeaderDecoder;
import com.sun.tck.bvtool.etsi.data.SecuredPacketTLV.HeaderEncoder;
import com.sun.tck.bvtool.etsi.security.Checksum;
import com.sun.tck.bvtool.etsi.security.KeyDescr;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer.Position;
import com.sun.tck.bvtool.etsi.tlv.TreeNode;
import com.sun.tck.me.utils.Utils;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import javax.crypto.Cipher;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public abstract class HeaderTLV implements TreeNode, SecuredPacketTLV.Header,
        SecuredPacketTLV.HeaderEncoder, SecuredPacketTLV.HeaderDecoder {
    protected Card card;
    protected long counter;

    protected TLVBuffer buffer;
    protected Position startValue;

    protected TLVBuffer checkSumSection;
    protected TLVBuffer.Position decodedCheckSum;
    protected byte[] checkSum;

    protected byte PCNTR;
    protected byte KID;

    private boolean isInitialized = false;

    private int tar = 0x000000; //0xB20100;

    public ArrayList<TLVBuffer> includeToCheckSum = new ArrayList<TLVBuffer>();
    public ArrayList<TLVBuffer> includeToCipher = new ArrayList<TLVBuffer>();
    protected byte[] securedData;
    protected Position securedDataPos;
    protected int headerlength;
    protected Position startEncryptedData;
    protected Position startSignedData;

    protected void initData() {
    }

    void init() {
        if (isInitialized) {
            return;
        }
        includeToCheckSum.clear();
        includeToCipher.clear();
        initData();
        isInitialized = true;
    }

    public byte getKID() {
        return KID;
    }

    public byte getPCNTR() {
        return PCNTR;
    }

    public long getCNTR() {
        return counter;
    }


    public void setTAR(int tar) {
        this.tar = tar;
    }

    public int getTAR() {
        return this.tar;
    }

    private Checksum checksum() {
        if (card == null) {
            return null;
        } 
        KeyDescr key = (card == null) ? null : card.getKID();
        return (key == null) ? null : key.checksum;
    }

    public void init(Card card) {
        this.card = card;
        this.counter = card.getCounter();
    }

    private KeyDescr getCipher() {
        return (card == null) ? null : card.getKIC();
    }

    public void sign() throws GeneralSecurityException {
        if (checkSumSection != null) {
            KeyDescr key = (card == null) ? null : card.getKID();
            checkSum = calculateCheckSum(key);
            checkSumSection.write(checkSum);
            Tracer.trace("createChecksum:" + Utils.canonize(checkSum));
        }
    }

    private byte[] calculateCheckSum(KeyDescr key) throws GeneralSecurityException {
        Checksum checksum;
        if ((key == null) || ((checksum = key.checksum) == null)) {
            return new byte[0];
        }
        int start = (startSignedData == null) ? buffer.start() : startSignedData.pos;
        checksum.update(buffer.getBuffer(), start,
                checkSumSection.start() - start);
        checksum.update(securedData, 0, securedData.length);
        checksum.update(new byte[16], 0, PCNTR);
        Tracer.trace("SIGNING:"
                + Utils.canonize(buffer.getBuffer(), start, checkSumSection.start() - start)
                + Utils.canonize(securedData)
                + Utils.canonize(new byte[16], 0, PCNTR));
        return checksum.calculateCheckSum();
    }

    public void verify() throws GeneralSecurityException, EncodingException {
        if (checkSumSection == null) {
            return;
        }
        KeyDescr key = (card == null) ? null : card.getKID(KID);
        byte[] calculated = calculateCheckSum(key);
        if (!Arrays.equals(checkSum, calculated)) {
            throw new EncodingException("verification fails. Expected:0x"
                    + Utils.canonize(checkSum) + " Received:0x"
                    + Utils.canonize(calculated));
        }
    }

    public HeaderDecoder decoder() {
        return this;
    }

    public HeaderEncoder encoder() {
        return this;
    }

    public void setBuffer(TLVBuffer fullTLV, Position start, Position data) {
        startSignedData = start;
        buffer = fullTLV;
        startValue = data;
    }

    public void write(TLVBuffer out) throws EncodingException {
        init();
        Position lengthPos = out.createPosition();
        out.skip(1);
        writeHeaderData(out);
        Checksum checksum = checksum();
        if ((checksum != null) && (checksum.getLength() != 0)) {
            // Reserve space for RC/CC/DS
            // the value will be calculated later, because it includes CHL
            // and CPL, which are calculated when packet length is determinated.
            checkSumSection = out.createChild(checksum.getLength());
        }
        lengthPos.write(out.getPos() - lengthPos.pos - 1);
        securedDataPos = out.createPosition();
        out.write(securedData);
        // write padding bytes
        out.write(0L, PCNTR);
    }

    public void setData(byte[] securedData) {
        this.securedData = securedData;
        KeyDescr cipher = getCipher();
        if (cipher != null) {
            int blockSize = getCipher().blockSize;
            int rem = (securedData.length + 6) % blockSize;
            PCNTR = (byte)((rem == 0) ? 0 : blockSize - rem);
        } else {
            PCNTR = 0;
        }
    }

    public void encrypt() throws GeneralSecurityException {
        KeyDescr cipher = getCipher();
        if (cipher == null) {
            return;
        }
        process(cipher.cipher, buffer.getPos());
    }

    public void decrypt() throws GeneralSecurityException {
        KeyDescr cipher = getCipher();
        if (cipher == null) {
            return;
        }
        process(cipher.decipher, buffer.end());
    }

    private void process(Cipher cipher, int end) throws GeneralSecurityException {
        byte[] buff =  buffer.getBuffer();
        int offset = startEncryptedData.pos;
        int length = end - offset;
        Tracer.trace("Processing:" + Utils.canonize(buff, offset, length));
        Tracer.trace("process offset=" + offset + " length=" + length);
        byte[] tmp = new byte[length];
        cipher.doFinal(buff, offset, length, tmp, 0);
        System.arraycopy(tmp, 0, buff, offset, length);
    }

    public abstract void writeHeaderData(TLVBuffer out) throws EncodingException;

    public Map<String, ? extends TreeNode> childNodes() {
        return Collections.emptyMap();
    }

    public Object value() {
        return this;
    }
}
