/*
 * %W% %E%
 *
 * Copyright 2010, Oracle and/or its affiliates. All rights reserved.
 * Oracle PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sun.tck.bvtool.etsi.data;

import com.sun.tck.bvtool.etsi.security.KeyDescr;
import com.sun.tck.bvtool.etsi.tlv.EncodingException;
import com.sun.tck.bvtool.etsi.tlv.TLVBuffer;
import com.sun.tck.bvtool.etsi.tlv.TLVUtils;
import com.sun.tck.me.utils.Utils;
import java.security.GeneralSecurityException;

/**
 *
 * @author Maxim V. Sokolnikov
 */
public class CommandHeader extends HeaderTLV {
/*
 * ----------------------------------------------------------------------------
 * Command Header Identifier (CHI)   | 1 octet  | Identifies the Command Header.
 * ----------------------------------------------------------------------------
 * Command Header Length (CHL)       | variable | This shall indicate the number of octets from and including the SPI
 *                                   |          | to the end of the RC/CC/DS.
 * ----------------------------------------------------------------------------
 * Security Parameter Indicator(SPI) | 2 octets | see detailed coding in clause 5.1.1.
 * ----------------------------------------------------------------------------
 * Ciphering Key Identifier (KIc)    | 1 octet  | Key and algorithm Identifier for ciphering.
 * ----------------------------------------------------------------------------
 * Key Identifier (KID)              | 1 octet  | Key and algorithm Identifier for RC/CC/DS.
 * ----------------------------------------------------------------------------
 * Toolkit Application Reference(TAR)| 3 octets | Coding is application dependent as defined in TS 101 220 [8].
 * ----------------------------------------------------------------------------
 * Counter (CNTR)                    | 5 octets | Replay detection and Sequence Integrity counter.
 * ----------------------------------------------------------------------------
 * Padding Counter (PCNTR)           | 1 octet  | This indicates the number of padding octets used for ciphering at
 *                                   |          | the end of the secured data.
 * ----------------------------------------------------------------------------
 * Redundancy Check (RC),            | variable | Length depends on the algorithm. A typical value is 8 octets if
 * Cryptographic Checksum (CC) or    |          | used, and for a DS could be 48 or more octets; the minimum
 * Digital Signature (DS)            |          | should be 4 octets.
 */
    // Issuer Security Domain '00 00 00' TS 102 226 [19] / compact data format
    // Issuer Security Domain 'B2 01 00' TS 102 226 [19] / expanded data format
    //                                       or automatic data format detection

    private short SPI;
    private byte KIc;
    private TLVBuffer securedDataBuff;

    @Override
    protected void initData() {
        SPI = card.encodeSPI();
        KIc = (byte) encode(card.getKIC());
        KID = (byte) encode(card.getKID());
    }

    private int encode(KeyDescr key) {
        return (key == null) ? 0 : key.encodeKey();
    }

    @Override
    public void writeHeaderData(TLVBuffer out) {
        out.write(SPI, 2);
        out.write(KIc);
        out.write(KID);
        out.write(getTAR(), 3);             // TAR
        startEncryptedData = out.createPosition();
        out.write(this.counter, 5);    // CNTR
        out.write(PCNTR);
    }

    public TLVBuffer encrypted;


    public void decrypt(TLVBuffer padded) throws GeneralSecurityException, EncodingException {
        encrypted.setPosTo(encrypted.start());
        this.counter = encrypted.read(5);
        PCNTR = (byte)encrypted.read();
        if (checkSumSection != null) {
            checkSumSection.setPosTo(checkSumSection.start());
            checkSum = checkSumSection.readArray(checkSumSection.available());
        }
    }


    @Override
    public String toString() {
        init();
        return getClass().getSimpleName() + "{\n"
                + "    SPI=" + TLVUtils.toBinaryString(SPI, 2, " ") + "\n"
                + "    KIc=" + TLVUtils.toBinaryString(KIc, 1, "") + "\n"
                + "    KID=" + TLVUtils.toBinaryString(KID, 1, "") + "\n"
                + "    TAR=0x" + Integer.toHexString(getTAR() & 0xFFFFFF) + "\n"
                + "    CNTR=0x" + Long.toHexString(counter) + "\n"
                + "    PCNTR=0x" + Integer.toHexString(PCNTR & 0xFF) + "\n"
                + "    RC/CC/DS=0x" + Utils.canonize(checkSum)
                + "}";
    }

    @Override
    public void parseHeader() throws EncodingException {
        this.headerlength = buffer.read();
        SPI = (short) buffer.read(2);
        KIc = (byte)buffer.read();
        KID = (byte)buffer.read();
        setTAR((int)buffer.read(3));     // TAR
        startEncryptedData = buffer.createPosition();
    }

    public void parseDecryptedHeader() throws EncodingException {
        this.counter = buffer.read(5); // CNTR
        PCNTR = (byte)buffer.read();
        int sigLength = headerlength - buffer.getPos() + startValue.pos + 1;
        checkSum = buffer.readArray(sigLength);
        securedDataBuff = buffer.createChild(buffer.available() - PCNTR);
    }


    public TLVBuffer getSecuredData() {
        return securedDataBuff;
    }
}
