/*
 * %W% %E%
 *
 * @Copyright
 */

package com.sun.jck.lib;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.net.Socket; 

public abstract class MessageClient implements Runnable
{
    public MessageClient(String name, String hostAndPort) throws IOException {
	this(name, openSocket(hostAndPort));
    }

    public MessageClient(String name, String host, int port) throws IOException {
	this(name, new Socket(host, port));
    }

    public MessageClient(String name, Socket socket) throws IOException {
	this.name = name;
	this.socket = socket;
	in = new DataInputStream(new BufferedInputStream(socket.getInputStream()));
	out = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
    }

    public synchronized void broadcast(String[] args) throws IOException {
	if (listener == null)
	    throw new IllegalStateException();
	out.writeByte(BROADCAST);
	out.writeShort(args.length);
	for (int i = 0; i < args.length; i++)
	    out.writeUTF(args[i]);
	out.flush();
    }

    public synchronized void send(String taskName, String[] args) throws IOException {
	if (listener == null)
	    throw new IllegalStateException();
	out.writeByte(SEND);
	out.writeUTF(taskName);
	out.writeShort(args.length);
	for (int i = 0; i < args.length; i++)
	    out.writeUTF(args[i]);
	out.flush();
    }

    public synchronized void close() {
	if (state < CLOSING) {
	    state = CLOSING;
	    
	    try {
		in.close();
	    }
	    catch (IOException ignore) {
	    }
	    
	    try {
		out.close();
	    }
	    catch (IOException ignore) {
	    }
	    
	    try {
		socket.close();
	    }
	    catch (IOException ignore) {
	    }
	    
	    listener.interrupt();
	    listener = null;
	}
    }

    public synchronized void start() throws InterruptedException {
	if (listener != null)
	    throw new IllegalStateException();
	listener = new Thread(this, "MessageClient");
	listener.start();
	while (state == STARTING)
	    wait();
    }

    public void run() {
	synchronized (this) {
	    if (listener == null)
		listener = Thread.currentThread();
	    else if (listener != Thread.currentThread())
		throw new IllegalStateException();
	}

	try {
	    out.writeByte(NAME);
	    out.writeUTF(name);
	    out.flush();

	    synchronized (this) {
		state = RUNNING;
		notifyAll();
	    }

	    while (state == RUNNING) {
		int code;
		try {
		    code = in.readByte();
		}
		catch (EOFException e) {
		    return;
		}

		switch (code) {
		case MESSAGE:
		    String from = in.readUTF();
		    int n = in.readShort();
		    String[] args = new String[n];
		    for (int i = 0; i < n; i++)
			args[i] = in.readUTF();
		    handleMessage(from, args);
		    break;
		default:
		}
	    }
	}
	catch (IOException e) {
	    handleException(e);
	}
	finally {
	    synchronized (this) {
		state = CLOSED;
		notifyAll();
	    }
	}
    }

    public synchronized void waitUntilClosed() throws InterruptedException {
	while (state != CLOSED)
	    wait();
    }
    
    protected abstract void handleException(Exception e);

    protected abstract void handleMessage(String from, String[] args);

    private static Socket openSocket(String hostAndPort) throws IOException {
	int colon = hostAndPort.indexOf(':');
	if (colon == -1)
	    throw new IllegalArgumentException(I18n.getString("no.port.specified"));
	return new Socket(hostAndPort.substring(0, colon),
			  Integer.parseInt(hostAndPort.substring(colon+1)));
    }

    private String name;
    private Socket socket;
    private DataInputStream in;
    private DataOutputStream out;
    private Thread listener;
    private static final int STARTING = 0, RUNNING = 1, CLOSING = 2, CLOSED = 3;
    private int state;

    static final int NAME = 1;
    static final int BROADCAST = 2;
    static final int SEND = 3;
    static final int MESSAGE = 100;
}
