/*
 * %W% %E% Jonathan Gibbons
 * 
 * Copyright  2002 Sun Microsystems, Inc. All rights reserved. 
 * Use is subject to license terms.
 */

package com.sun.jck.lib;

import java.io.File;
import java.util.Map;
import java.util.ResourceBundle;

import com.sun.javatest.InterviewParameters;
import com.sun.javatest.Script;
import com.sun.javatest.TestDescription;
import com.sun.javatest.TestEnvironment;
import com.sun.javatest.TestFilter;
import com.sun.javatest.TestFinder;
import com.sun.javatest.TestSuite;
import com.sun.javatest.WorkDirectory;
import com.sun.javatest.finder.BinaryTestFinder;
import com.sun.javatest.interview.LegacyParameters;
import com.sun.javatest.util.BackupPolicy;
import com.sun.javatest.util.I18NResourceBundle;
import com.sun.javatest.util.StringArray;
import com.sun.jck.interview.JCKParameters;

public class JCKTestSuite extends TestSuite {

    public JCKTestSuite(File root, Map tsInfo, ClassLoader cl) throws Fault {
	super(canonicalizeRoot(root), tsInfo, cl);
    }

    private static File canonicalizeRoot(File f) throws Fault {
	if (f.isFile()) {
	    if (f.getName().equalsIgnoreCase("testsuite.html"))
		return f.getParentFile().getParentFile();
	    else
		throw new Fault(i18n, "ts.badRootFile", f);
	}
	else if (new File(f, "testsuite.jtt").exists())
	    return f;
	else if (new File(f, "testsuite.html").exists())
	    return f.getParentFile();
	else if (new File(new File(f, "tests"), "testsuite.html").exists())
	    return f;
	else
	    throw new Fault(i18n, "ts.badRootFile", f);
    }

    public void init(String[] args) throws Fault {
	String defaultArgs = System.getProperty("JCKTestSuite.defaultArgs");
	if (defaultArgs != null) {
	    decodeArgs(StringArray.split(defaultArgs));
	}

	decodeArgs(args);

	// validate args ...
	// ... must specify one or other of -compiler and -runtime
	if (!compilerFlag && !runtimeFlag)
	    throw new Fault(i18n, "ts.needCompOrRuntime");

	// ... must not specify both of -compiler and -runtime
	if (compilerFlag && runtimeFlag)
	    // this could be relaxed if we were really careful with the commands
	    // involved, because we could run tests *twice* if necessary: once
	    // as a compiler test, once as a runtime test
	    throw new Fault(i18n, "ts.multCompOrRuntime");

	// ... must not specify two or more mode flags
	if (certifyFlag && developerFlag || 
	    developerFlag && precompileFlag ||
	    precompileFlag && certifyFlag) 
	    throw new Fault(i18n, "ts.badMode");

	// ... if no mode flag set, default to -certify
	if (!certifyFlag && !developerFlag && !precompileFlag) 
	    mode = CERTIFY;

	// ... must not precompile compiler tests
	if (precompileFlag && compilerFlag)
	    throw new Fault(i18n, "ts.precompileCompiler");
    }

    protected void decodeArgs(String[] args) throws Fault{
	// decode args
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equalsIgnoreCase("-webWalk")) {
		finderMode = JCKTestFinder.WEB_WALK;
	    }
	    else if (args[i].equalsIgnoreCase("-dirWalk")) {
		finderMode = JCKTestFinder.DIR_WALK;
	    }  
	    else if (args[i].equalsIgnoreCase("-verify")) {
		finderVerifyFlag = true;
	    }  
	    else if (args[i].equals("-compiler")) {
		product = COMPILER;
		compilerFlag = true;
	    }
	    else if (args[i].equals("-runtime")) {
		product = RUNTIME;
		runtimeFlag = true;
	    }
	    else if (args[i].equals("-certify")) {
		mode = CERTIFY;
		certifyFlag = true;
	    }
	    else if (args[i].equals("-developer")) {
		mode = DEVELOPER;
		developerFlag = true;
	    }
	    else if (args[i].equals("-precompile")) {
		mode = PRECOMPILE;
		precompileFlag = true;
	    }
	    else if (args[i].equals("-precompileClassDir") && i+1 < args.length) {
		scriptPrecompileClassDir = args[++i];
	    }
	    else 
		throw new Fault(i18n, "ts.badArg", args[i]);
	}
    }

    public File getTestsDir() {
	String testsEntry = (String) (getTestSuiteInfo().get("tests"));
	if (testsEntry == null || testsEntry.length() == 0) // default
	    return new File(getRootDir(), "tests");
	else {
	    File f = new File(testsEntry);
	    if (f.isAbsolute())
		return f;
	    else
		return new File(getRootDir(), f.getPath());
	}
    }

    public TestFilter createTestFilter(TestEnvironment env) {
	return new JCKTestFilter(env, product, mode);
    }

    public TestFinder createTestFinder() throws Fault {
	try {

	    File jtdFile;
	    String jtdEntry = (String) (getTestSuiteInfo().get("testsuite.jtd"));
	    if (jtdEntry == null || jtdEntry.length() == 0) // default
		jtdFile = new File(new File(getRootDir(), "tests"), "testsuite.jtd");
	    else {
		File f = new File(jtdEntry);
		if (f.isAbsolute())
		    jtdFile = f;
		else
		    jtdFile = new File(getRootDir(), f.getPath());
	    }

	    File testsDir = getTestsDir();

	    //File jtdFile = new File(rd, "testsuite.jtd");
	    //File testsDir = new File(rd, "tests");

	    if (jtdFile.exists()) {
		BinaryTestFinder f = new BinaryTestFinder();
		f.init((new String[] { "-binary", jtdFile.getPath() }), testsDir, null);
		return f;
	    }
	    else {	
		JCKTestFinder f = new JCKTestFinder();
		if (finderMode == JCKTestFinder.WEB_WALK) {
		    f.setMode(JCKTestFinder.WEB_WALK);
		    f.setRoot(new File(testsDir, "testsuite.html"));
		}
		else {
		    f.setMode(JCKTestFinder.DIR_WALK);
		    f.setRoot(testsDir);
		}
		f.setVerify(finderVerifyFlag);
		return f;
	    }
	}
	catch (TestFinder.Fault e) {
	    throw new Fault(i18n, "ts.errorInitFinder", e.getMessage());
	}
    }

    public Script createScript(TestDescription td, String[] exclTestCases, TestEnvironment scriptEnv, 
			       WorkDirectory workDir, 
			       BackupPolicy backupPolicy) throws Fault {
	JCKScript s = new JCKScript();
	// generic script init
	s.initArgs(null); 	// disables arg checking in JCKScript
	s.initTestDescription(td);
	s.initExcludedTestCases(exclTestCases);
	s.initTestEnvironment(scriptEnv);
	s.initWorkDir(workDir);
	s.initBackupPolicy(backupPolicy);
	s.initClassLoader(getClassLoader());
	// JCKScript-specific stuff
	s.setProduct(product);
	s.setMode(mode);
	if (mode == PRECOMPILE) {
	    s.setPrecompileClassDir(scriptPrecompileClassDir);
	}
	return s;
    }

    public InterviewParameters createInterview() throws TestSuite.Fault {
	// should consider passing MODE to wizard and have it decide
	// whether to use refCompile entries or not

	String p = (product == RUNTIME ? "runtime" : "compiler");

	try {
	    /*if (mode == CERTIFY) {
		String p = (product == RUNTIME ? "runtime" : "compiler");
		return new JCKParameters(this, p);
	    }
	    else 
		return new LegacyParameters(this);*/
	    return new JCKParameters(this, p, mode);
	}
	catch (InterviewParameters.Fault e) {
	    throw new Fault(i18n, "ts.errorInitInterview", e.getMessage());
	}
    }

    public int getProduct() {
	return product;
    }

    public static String[] getRequiredEntries() {
	return new String[] {
	    // required
	    "keywords",
	    "source",
	    "title"
	};
    }

    public static String[] getValidEntries() {
	return new String[] {
	    // required
	    "keywords",
	    "source",
	    "title",
	    // optional
	    "context",
	    "executeArgs",
	    "executeClass",
	    "executeNative",
	    "id",		// defined and used internally by JavaTest
	    "remote",
	    "rmicClass",	// temp for backwards compatible for Kestrel
	    "rmicClasses",
	    "selectIf",
	    "executeIf",	// temp for backwards compatible
	    "timeout"
	};
    }

    public static String[] getValidKeywords() {
	return new String[] {
	    // approved
	    "compiler",
	    "runtime",
	    "positive",
	    "negative",
	    "idl_inherit",
	    "idl_tie",
	    "interactive",
	    "jniinvocationapi",
	    "only_once",
	    "rmi_iiop",
	    "rmi_v11",
	    // will eventually be superceded/deprecated
	    "serial"
	};
    }

    // Flags are set from command line options....
    //
    // exactly one of these must be specified
    private boolean compilerFlag = false;
    private boolean runtimeFlag = false;
    // at most one of these must be specified
    private boolean certifyFlag = false;
    private boolean precompileFlag = false;
    private boolean developerFlag = false;
    
    public static final int COMPILER = JCKScript.COMPILER;
    public static final int RUNTIME = JCKScript.RUNTIME;
    private int product;
    //public static final int CERTIFY = 1;
    //public static final int PRECOMPILE = 2;
    //public static final int DEVELOPER = 3;
    /*public static final int CERTIFY = JCKParameters.CERTIFY;
    public static final int PRECOMPILE = JCKParameters.PRECOMPILE;
    public static final int DEVELOPER = JCKParameters.DEVELOPER;*/
    public static final int CERTIFY = JCKScript.CERTIFY;
    public static final int PRECOMPILE = JCKScript.PRECOMPILE;
    public static final int DEVELOPER = JCKScript.DEVELOPER;
    private int mode;
    private String scriptPrecompileClassDir;
    private boolean finderVerifyFlag;
    private int finderMode;

    private static final I18NResourceBundle i18n = I18NResourceBundle.getBundleForClass(JCKTestSuite.class);
} 
