/*
 * %W% %E%
 *
 * @Copyright
 */

package com.sun.jck.lib;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Set;
import com.sun.javatest.Script;
import com.sun.javatest.Status;
import com.sun.javatest.TestDescription;
import com.sun.javatest.TestEnvironment;
import com.sun.javatest.util.StringArray;

/**
 * Script to execute JCK tests, based on the keywords in the test description.
 */
public class JCKScript extends Script 
{

    /**
     * Initialize the test description, and extract keywords
     */
    public void initTestDescription(TestDescription td) {
	super.initTestDescription(td);
	// get necessary keywords for later use; "interactive" is needed early on
	// to determine timeout values
	Set keys = td.getKeywordTable();
	compilerKey = (keys.contains("compiler"));
	coreAPIKey = (keys.contains("coreapi"));		// backwards compatibility
	extensionAPIKey = (keys.contains("extensionapi"));	// does anyone use this?
	idlInheritKey = (keys.contains("idl_inherit"));
	idlTieKey = (keys.contains("idl_tie"));
	interactiveKey = (keys.contains("interactive"));
	javaCompilerKey = (keys.contains("javacompiler"));
	negativeKey = (keys.contains("negative"));
	optionalAPIKey = (keys.contains("optionalapi"));
	positiveKey = (keys.contains("positive"));
	rmicKey = (keys.contains("rmic"));
	rmiIIOPKey = (keys.contains("rmi_iiop"));
	rmiV11Key = (keys.contains("rmi_v11"));
	runtimeKey = (keys.contains("runtime"));
	serialKey = (keys.contains("serial"));
	vmKey = (keys.contains("virtualmachine"));
    }


    /** Run the script.
     * @param env	The test environment giving the details of how to run the test
     * @return		The result of running the script
     */
    public Status run(String[] args, TestDescription td, TestEnvironment env) {
	if (args != null) {
	    Status s = init(args);
	    // sigh, exceptions would be cleaner :-)
	    if (s != null)
		return s;
	}

	// get context info
	String[] context = StringArray.split(td.getParameter("context"));
	if (context != null && (mode == DEVELOPER || mode == CERTIFY)) {
	    StringBuffer sb = new StringBuffer();
	    for (int i = 0; i < context.length; i++) {
		try {
		    String[] contextValue = env.lookup(context[i]);
		    if (contextValue == null || contextValue.length == 0)
			return Status.error(I18n.getString("undefined.context", context[i]));
		    if (contextValue.length != 1)
			return Status.error(I18n.getString("bad.context",
                                context[i], StringArray.join(contextValue)));
		    sb.append('-');
		    sb.append(context[i]);
		    sb.append(' ');
		    sb.append(contextValue[0]);
		    sb.append(' ');
		}
		catch (TestEnvironment.Fault e) {
		    return Status.error(I18n.getString("bad.context",
                                context[i], e.getMessage()));
		}
	    }
	    contextArgs = sb.toString();
	}

	switch (product) {
	case COMPILER:
	    return runCompilerTest();
	case RUNTIME:
	    return runRuntimeTest();
	default:
	    throw new IllegalStateException(I18n.getString("invalid.product"));
	}
    }

    public void setProduct(int p) {
	product = p;
    }

    public void setMode(int m) {
	mode = m;
    }

    public void setPrecompileClassDir(String d) {
	precompileClassDir = d;
    }

    public int getTestTimeout() {
	if (interactiveKey)
	    return 0;

	String s = td.getParameter("timeout");
	int t = (s == null ? 10 * 60 : Integer.parseInt(s));

	float factor = 1;
	try {
	    String f = env.lookup("javatestTimeoutFactor")[0];
	    if (f != null)
		factor= Float.parseFloat(f);
	}
	catch (TestEnvironment.Fault f) {
	}

	return (int) (t * factor);
    }

    private Status init(String[] args) {
	// Flags are set from command line options....
	//
	// exactly one of these must be specified
	boolean compilerFlag = false;
	boolean runtimeFlag = false;
	// at most one of these must be specified
	boolean certifyFlag = false;
	boolean precompileFlag = false;
	boolean developerFlag = false;

	// decode args
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-compiler")) {
		product = COMPILER;
		compilerFlag = true;
	    }
	    else if (args[i].equals("-runtime")) {
		product = RUNTIME;
		runtimeFlag = true;
	    }
	    else if (args[i].equals("-certify")) {
		mode = CERTIFY;
		certifyFlag = true;
	    }
	    else if (args[i].equals("-developer")) {
		mode = DEVELOPER;
		developerFlag = true;
	    }
	    else if (args[i].equals("-precompile")) {
		mode = PRECOMPILE;
		precompileFlag = true;
	    }
	    else if (args[i].equals("-noJTRIfPass")) {
		setJTRIfPassed(false);
	    }
	    else if (args[i].equals("-precompileClassDir") && i+1 < args.length)
		precompileClassDir = args[++i];
	    else 
		return error_unrecognizedArg.augment(args[i]);
	}

	// validate args ...
	// ... must specify one or other of -compiler and -runtime
	if (!compilerFlag && !runtimeFlag)
	    return error_noCompilerRuntimeFlag;

	// ... must not specify both of -compiler and -runtime
	if (compilerFlag && runtimeFlag)
	    // this could be relaxed if we were really careful with the commands
	    // involved, because we could run tests *twice* if necessary: once
	    // as a compiler test, once as a runtime test
	    return error_bothCompilerRuntimeFlag;

	// ... must not specify two or more mode flags
	if (certifyFlag && developerFlag || 
	    developerFlag && precompileFlag ||
	    precompileFlag && certifyFlag) 
	    return error_badModeFlags;

	// ... if no mode flag set, default to -certify
	if (!certifyFlag && !developerFlag && !precompileFlag) {
	    certifyFlag = true;
	    mode = CERTIFY;
	}

	// ... must not precompile compiler tests
	if (precompileFlag && compilerFlag)
	    return error_precompileCompilerTest;

	return null;
    }

    private Status runCompilerTest() {
        PrintWriter trOut = getTestResult().getTestCommentWriter();

	// verify it is a compiler test
	// compiler keyword is preferred; javaCompiler and rmic are
	// deprecated alternates
	boolean isCompilerTest = compilerKey || (javaCompilerKey || rmicKey);
	if (!isCompilerTest)
	    return error_notACompilerTest;

	// Exercise the test component(s)...

	// ... javac step
	trOut.println(I18n.getString("testing.java.compiler"));
	File[] srcs = td.getSourceFiles();
	for (int i = 0; i < srcs.length; i++) {
	    String srcPath = srcs[i].getPath();
	    if (!srcPath.endsWith(".java"))
		return error_notJavaSource.augment(srcPath);
	}

	Status compileStatus;
	if (serialKey)
	    compileStatus = compileIndividually(TEST_COMPILE, srcs);
	else
	    compileStatus = compileTogether(TEST_COMPILE, srcs);

	// ... optional rmic step
	String rmiClasses = td.getParameter("rmicClasses");
	// backwards compatibility
	if (rmiClasses == null)
	    rmiClasses = td.getParameter("rmicClass");
	if (rmiClasses != null) {
	    // check that setup compilation succeeded before 
	    // proceeding to rmic.
	    if (!compileStatus.isPassed())
		return error_compFailUnexp.augment(compileStatus);

	    trOut.println(I18n.getString("testing.rmi.compiler"));
	    // allow keyword-dependent variants of the command
	    String rmicCmd = TEST_RMIC;
	    if (rmiIIOPKey)
		rmicCmd += ".iiop";
	    else if (rmiV11Key)
		rmicCmd += ".v11";

	    // since we know that javac passed, we can overwrite the compileStatus
	    compileStatus = rmiCompile(rmicCmd, StringArray.split(rmiClasses));
	}

	// Test component has been executed; now check the results...

	if (negativeKey) {
	    // ... negative test: check that compilation failed--
	    // swap pass and fail, propogate others
	    trOut.println(I18n.getString("negative.test.check"));
	    switch (compileStatus.getType()) {
	    case Status.PASSED:
		return fail_compSuccUnexp;
	    case Status.FAILED:
		return pass_compFailExp;
	    default:
		return compileStatus;
	    }
	}
	else {
	    // ... positive test: check that code executes on reference platform
	    if (compileStatus.isPassed()) {
		trOut.println(I18n.getString("positive.test.check"));
		String executeClass = td.getParameter("executeClass");
		String executeArgs = getExecuteArgs(td);
		if (executeClass == null || executeClass.length() == 0)
		    return error_noExecuteClass;
		return execute(REF_EXECUTE, executeClass, executeArgs);
	    }
	    else
		return fail_compFailUnexp.augment(compileStatus);
	} 
    }

    private Status runRuntimeTest() {
        PrintWriter trOut = getTestResult().getTestCommentWriter();

	// verify it is a runtime test
	// runtime keyword is preferred; others are deprecated alternates
	boolean isRuntimeTest = runtimeKey || 
	    ((javaCompilerKey && positiveKey) ||
	     vmKey ||
	     (coreAPIKey || optionalAPIKey || extensionAPIKey));
	
	if (!isRuntimeTest)
	    return error_notARuntimeTest;
	
	// Compile test if necessary using the reference platform ...
	// Use error (not fail) if there is a problem
	if (mode == DEVELOPER || mode == PRECOMPILE) {
	    // ... javac step
	    trOut.println("precompiling.sources");
	    File[] srcs = td.getSourceFiles();

	    boolean allJavaSrc = true; // assume true until proven otherwise
	    for (int i = 0; i < srcs.length && allJavaSrc; i++) {
		String srcPath = srcs[i].getPath();
		allJavaSrc = srcPath.endsWith(".java");
	    }
	    
	    Status compileStatus;
	    if (serialKey || !allJavaSrc)
		compileStatus = compileIndividually(REF_COMPILE, srcs);
	    else if (precompileClassDir != null) 
		compileStatus = compileIfNecessary(REF_COMPILE, srcs, precompileClassDir);
	    else
		compileStatus = compileTogether(REF_COMPILE, srcs);
	    
	    // ... optional rmic step
	    String rmiClasses = td.getParameter("rmicClasses");
	    // backwards compatibility
	    if (rmiClasses == null)
		rmiClasses = td.getParameter("rmicClass");
	    if (rmiClasses != null) {
		// check that setup compilation succeeded before 
		// proceeding to rmic.
		if (!compileStatus.isPassed())
		    return error_compFailUnexp.augment(compileStatus);
		
		trOut.println(I18n.getString("precompiling.rmi"));
		// allow keyword-dependent variants of the command
		String rmicCmd = REF_RMIC;
		if (rmiIIOPKey)
		    rmicCmd += ".iiop";
		else if (rmiV11Key)
		    rmicCmd += ".v11";
		// since we know that javac passed, we can overwrite the compileStatus
		compileStatus = rmiCompile(rmicCmd, StringArray.split(rmiClasses));
	    }

	    if (!compileStatus.isPassed())
		return error_compFailUnexp.augment(compileStatus);
	}

	if (mode == DEVELOPER || mode == CERTIFY) {
	    // Exercise the test component...
	    Status executeStatus;
	    String executeArgs = getExecuteArgs(td);
	    String executeClass = td.getParameter("executeClass");
	    if (executeClass != null) {
		String remote = td.getParameter("remote");
		if (remote == null) {
		    // classic simple runtime test
		    trOut.println(I18n.getString("executing.test"));
		    executeStatus = execute(TEST_EXECUTE, executeClass, executeArgs);
		}
		else {
		    // distributed test
		    try {
			trOut.println(I18n.getString("starting.remote"));
			MessageSwitch ms = new MessageSwitch();
			env.put("testMsgSwitch", ms.getHostAndPort());
			RemoteManager rMgr = new RemoteManager(remote, env, getTestResult());
			try {
			    trOut.println(I18n.getString("executing.test"));
			    executeStatus = execute(TEST_EXECUTE, executeClass, executeArgs);
			    ms.close();
			    rMgr.waitUntilDone(15/*seconds*/);
			    Status rms = rMgr.getStatus();
			    executeStatus = RemoteManager.mergeStatus(executeStatus, rms);
			}
			finally {
			    rMgr.dispose();
			}
		    }
		    catch (InterruptedException e) {
			executeStatus = Status.error(I18n.getString("remote.exit.timeout"));
		    }
		    catch (IOException e) {
			executeStatus = Status.error(I18n.getString("cant.create.msgswitch", e));
		    }
		    catch (RemoteManager.Fault e) {
			executeStatus = Status.error(e.getMessage());
		    }
		}
	    }
	    else {
		String executeNative = td.getParameter("executeNative");
		if (executeNative != null)
		    executeStatus = executeNative(executeNative, executeArgs);
		else
		    return error_noExecuteClass; // yes, class
	    }
	    // Test component has been executed; now check the results...

	    if (negativeKey) {
		// ... negative test: check that execution failed--
		// swap pass and fail, propogate others
		switch (executeStatus.getType()) {
		case Status.PASSED:
		    return fail_execSuccUnexp;
		case Status.FAILED:
		    return pass_execFailExp.augment(executeStatus);
		default:
		    return executeStatus;
		}
	    }
	    else
		// ... positive test: check that code executes on reference platform
		return executeStatus;
	}
	else
	    // just precompiling, and if we're here, it must have passed
	    return pass_compSuccExp;	
    }

    protected Status compileOne(String command, String src) {
	if (src.endsWith(".idl") && (idlTieKey || idlInheritKey)) {
	    // check the first option
	    if (idlInheritKey) {
		Status s = super.compileOne(command + ".inherit", src);
		// if that failed, or if it is the only option,
		// we're done; otherwise, we passed and want the
		// other option, so drop through
		if (!s.isPassed() || !idlTieKey)
		    return s;
	    }
	    return super.compileOne(command + ".tie", src);
	}
	else
	    return super.compileOne(command, src);
    }

    private String getExecuteArgs(TestDescription td) {
	String executeArgs = td.getParameter("executeArgs");
	if (contextArgs == null)
	    return executeArgs;
	else if (executeArgs == null)
	    return contextArgs;
	else
	    return (contextArgs + ' ' + executeArgs);
    }

    private Status executeNative(String executeNative, String executeArgs) {
	try {
	    String[] args = (executeArgs == null ? nullArgs : env.resolve(executeArgs));
	    if (excludedTestCases != null)
		args = exclude(args, excludedTestCases);
	    env.put("testExecuteNative", executeNative);
	    env.put("testExecuteArgs", args);
	    return invokeCommand(TEST_EXECUTE_NATIVE);
	}
	catch (TestEnvironment.Fault e) {
	    trOut.println(I18n.getString("evaluating.exec.args", executeArgs));
	    trOut.println(e.toString());
	    return error_badExecuteArgs;
	}
    }

    public static final int COMPILER = 1;
    public static final int RUNTIME = 2;
    private int product;
    public static final int CERTIFY = 1;
    public static final int PRECOMPILE = 2;
    public static final int DEVELOPER = 3;
    private int mode;

    // special option for use in precompile mode
    private String precompileClassDir;

    // Keys are set from test description keywords
    //
    private boolean compilerKey;
    private boolean coreAPIKey;
    private boolean extensionAPIKey;
    private boolean idlInheritKey;
    private boolean idlTieKey;
    private boolean interactiveKey;
    private boolean javaCompilerKey;
    private boolean negativeKey;
    private boolean optionalAPIKey;
    private boolean positiveKey;
    private boolean rmicKey;
    private boolean rmiIIOPKey;
    private boolean rmiV11Key;
    private boolean runtimeKey;
    private boolean serialKey;
    private boolean vmKey;

    // Context, determined from context spec in test description, and env
    private String contextArgs;

    private static final String REF_COMPILE  = "refCompile";
    private static final String REF_EXECUTE  = "refExecute";
    private static final String REF_RMIC     = "refRMIC";
    private static final String TEST_COMPILE = "testCompile";
    private static final String TEST_EXECUTE = "testExecute";
    private static final String TEST_EXECUTE_NATIVE = "testExecuteNative";
    private static final String TEST_RMIC    = "testRMIC";

    private static final Status
	error_badModeFlags = Status.error(I18n.getString("error.bad.mode.flags")),
        error_bothCompilerRuntimeFlag = Status.error(I18n.getString("error.both.compiler.runtime.flag")),
        error_noCompilerRuntimeFlag = Status.error(I18n.getString("error.no.compiler.runtime.flag")),
        error_notACompilerTest = Status.error(I18n.getString("error.not.compiler.test")),
        error_notARuntimeTest = Status.error(I18n.getString("error.not.runtime.test")),
	error_notJavaSource = Status.error(I18n.getString("error.not.java.source")),
        error_precompileCompilerTest = Status.error(I18n.getString("error.precompile.compiler.test")),
        error_unrecognizedArg = Status.error(I18n.getString("error.unrecognized.arg"));
    
    private static final String[] nullArgs = { };
}
